/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.controller;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import no.esito.jvine.communication.ActionMessageCompilation;
import no.esito.util.StringUtil;

import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

/**
 * An implementation of the DialogConstant interface to be used when there is
 * a need to represent external dialogs in a jVine application. Objects of this
 * class should be created with the name of the dialog as given in the dialog model. 
 */
public class ExternalDialogConstant implements DialogConstant {

    /** The name of the dialog object as defined in the G9 dialog model */
    private final String g9Name;


    private ActionMessageCompilation reactor;
    
    /**
     * Used to build a message for communicating with this external dialog.
     * @return the value
     */
    public ActionMessageCompilation getReactor() {
        return reactor;
    }

    /**
     * Used to build a message for communicating with this external dialog.
     * @param reactor the value
     */
    public void setReactor(ActionMessageCompilation reactor) {
        this.reactor = reactor;
    }

    /**
     * Create a new ExternalDialogConstant for the given dialog name.
     * The name is a mandatory parameter.
     *
     * @param g9Name the g9 name of the dialog model
     */
    public ExternalDialogConstant(String g9Name) {
        Assert.notNull(g9Name, "The G9 name for the dialog cannot be null");
        this.g9Name = g9Name;
    }

    @Override
    public ViewConstant getParent() {
        return null;
    }

    @Override
    public Collection<ViewConstant> getAll() {
        Set<ViewConstant> all = new HashSet<ViewConstant>();
        all.add(this);
        return all;
    }

    @Override
    public String name() {
        return StringUtil.unCamelize(g9Name, "_").toUpperCase();
    }

    @Override
    public String getG9Name() {
        return g9Name;
    }

    @Override
    public String getInternalName() {
        return StringUtils.uncapitalize(g9Name);
    }

    @Override
    public String toString() {
        return getInternalName();
    } 

    @Override
    public String getTitle() {
        return StringUtil.unCamelize(g9Name);
    }

    @Override
    public WindowType getWindowType() {
        return WindowType.DOCUMENT_WINDOW;
    }

    @Override
    public int getMaximumNumberOfInstances() {
        return -1;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + g9Name.hashCode();
        return result;
    }

    @Override
    public boolean equals(Object that) {
        if ( this == that ) {
            return true;
        }
        if ( !(that instanceof ExternalDialogConstant) ) {
            return false;
        }
        ExternalDialogConstant that2 = (ExternalDialogConstant) that;

        return this.g9Name.equals(that2.getG9Name());
    }

}
