/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.converter;

import java.util.Date;

import no.esito.log.Logger;
import no.g9.support.convert.AttributeConverter;
import no.g9.support.convert.ConvertContext;
import no.g9.support.convert.ConvertException;

import org.joda.time.DateMidnight;
import org.joda.time.DateTime;
import org.joda.time.base.BaseDateTime;

/**
 * Converts between a Joda DateTime/DateMidnight attribute in the domain model and a Date in the view (target).
 * This converter is used for DateTime and DateMidnight attributes in the DateTimeEntry component, as it does
 * not support Joda time types.
 *
 * If there already is another converter present on the attribute, it is set as a chained converter.
 *
 * @param <M> the model type, either DateTime/DateMidnight or the model type of the chained converter
 */
@SuppressWarnings("deprecation")
public class JodaDateTimeConverter <M> implements AttributeConverter<M, Date> {

    // Model type class
    private Class<M> modelType;

    private AttributeConverter<M,BaseDateTime> chainedConverter;

    private static final Logger log = Logger.getLogger(JodaDateTimeConverter.class);

    /**
     * Create a new converter without a chained converter.
     *
     * @param modelType the model type to use, either DateTime or DateMidnight
     */
    public JodaDateTimeConverter(Class<M> modelType) {
        this(null, modelType);
    }

    /**
     * Create a new converter with the given chained converter.
     * The model type is set to the model type of the chained converter.
     * If the chained converter is null, the model type is set from the second parameter.
     *
     * @param chainedConverter the chained converter, or null
     * @param modelType the model type to use if no chained converter is given, either DateTime or DateMidnight
     */
    public JodaDateTimeConverter(AttributeConverter<M,BaseDateTime> chainedConverter, Class<M> modelType) {
        super();
        if (chainedConverter != null) {
            this.chainedConverter = chainedConverter;
            this.modelType = chainedConverter.getModelType();
        } else {
            if (modelType != DateTime.class && modelType != DateMidnight.class) {
                throw new IllegalArgumentException("The model type must be org.joda.time.DateTime or org.joda.time.DateMidnight");
            }
            this.modelType = modelType;
        }
    }

    @Override
    public Date fromModel(M value, ConvertContext context) throws ConvertException {
        if (log.isTraceEnabled()) {
            log.trace("Convert from model: " + value);
        }
        if (chainedConverter != null) {
            BaseDateTime fromModel = chainedConverter.fromModel(value, context);
            if (fromModel == null) {
                return null;
            }
            return fromModel.toDate();
        }
        if (value == null) {
            return null;
        }
        BaseDateTime jodaDate = (BaseDateTime) value;
        return jodaDate.toDate();
    }

    @SuppressWarnings("unchecked")
    @Override
    public M toModel(Date value, ConvertContext context) throws ConvertException {
        if (log.isTraceEnabled()) {
            log.trace("Convert to model: " + value);
        }
        BaseDateTime bdt = getJodaDate(value);
        if (chainedConverter != null) {
            return chainedConverter.toModel(bdt, context);
        }
        return (M) bdt;
    }

    private BaseDateTime getJodaDate(Date value) {
        if (value == null) {
            return null;
        }
        if (modelType == DateTime.class) {
            return new DateTime(value);
        }
        return new DateMidnight(value);
    }

    @Override
    public Class<Date> getTargetType() {
        return Date.class;
    }

    @Override
    public Class<M> getModelType() {
        return modelType;
    }

}
