/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.converter;

import no.esito.log.Logger;
import no.g9.support.convert.AttributeConverter;
import no.g9.support.convert.ConvertContext;
import no.g9.support.convert.ConvertException;

/**
 * Converts between a Number (Integer or Long) attribute in the domain model and a String in the view (target).
 * This converter is used for Number attributes in the AutoComplete component, as it only
 * supports the String type.
 *
 * If there already is another converter present on the attribute, it is set as a chained converter.
 *
 * @param <M> the model type, either Integer/Long or the model type of the chained converter
 */
public class NumberConverter <M> implements AttributeConverter<M, String> {

    // Model type class
    private Class<M> modelType;

    private AttributeConverter<M,Number> chainedConverter;

    private static final Logger log = Logger.getLogger(NumberConverter.class);

    /**
     * Create a new converter without a chained converter.
     *
     * @param modelType the model type to use, either Integer or Long
     */
    public NumberConverter(Class<M> modelType) {
        this(null, modelType);
    }

    /**
     * Create a new converter with the given chained converter.
     * The model type is set to the model type of the chained converter.
     * If the chained converter is null, the model type is set from the second parameter.
     *
     * @param chainedConverter the chained converter, or null
     * @param modelType the model type to use if no chained converter is given, either Integer or Long
     */
    public NumberConverter(AttributeConverter<M,Number> chainedConverter, Class<M> modelType) {
        super();
        if (chainedConverter != null) {
            this.chainedConverter = chainedConverter;
            this.modelType = chainedConverter.getModelType();
        } else {
            if (modelType != Integer.class && modelType != Long.class) {
                throw new IllegalArgumentException("The model type must be Integer or Long");
            }
            this.modelType = modelType;
        }
    }

    @Override
    public String fromModel(M value, ConvertContext context) throws ConvertException {
        if (log.isTraceEnabled()) {
            log.trace("Convert from model: " + value);
        }
        if (chainedConverter != null) {
            Number fromModel = chainedConverter.fromModel(value, context);
            if (fromModel == null) {
                return null;
            }
            return fromModel.toString();
        }
        if (value == null) {
            return null;
        }
        Number number = (Number) value;
        return number.toString();
    }

    @SuppressWarnings("unchecked")
    @Override
    public M toModel(String value, ConvertContext context) throws ConvertException {
        if (log.isTraceEnabled()) {
            log.trace("Convert to model: " + value);
        }
        Number number = getNumber(value);
        if (chainedConverter != null) {
            return chainedConverter.toModel(number, context);
        }
        return (M) number;
    }

    private Number getNumber(String value) {
        if (value == null) {
            return null;
        }
        if (modelType == Integer.class) {
            return Integer.valueOf(value);
        }
        return Long.valueOf(value);
    }

    @Override
    public Class<String> getTargetType() {
        return String.class;
    }

    @Override
    public Class<M> getModelType() {
        return modelType;
    }

}
