/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.converter;

import java.text.MessageFormat;

import no.g9.exception.G9BaseException;
import no.g9.support.convert.AttributeConverter;
import no.g9.support.convert.ConvertContext;
import no.g9.support.convert.ConvertException;

/**
 * A converter that fomats and converts values to (and from) String.
 *
 * Example of usage: A converter that converts dates to the form 2008-12-31.
 * {@code
 * &lt;bean id="StringConverter"
 *     class="no.g9.client.core.converter.StringConverter"&gt;
 *     <property name="modelTypeClass" value="java.util.Date" /&gt;
 *     <property name="format" value="{0,date,yyyy-MM-dd}" /&gt;
 * &lt;/bean&gt;
 *
 * This bean is configurable through the following properties:
 * <li> modelTypeClass: Qualified class name of model type
 * <li> format: Format rule, @see java.text.MessageFormat}
 *
 * @param <M> model type
 */
public class StringConverter <M> implements AttributeConverter<M, String> {

    // Model type class
    private Class<M> modelType;

    //Message format
    private MessageFormat format;


    /**
     * Sets model type class from name
     * @param className Qualified name of the model type class
     * @exception G9BaseException If class not found
     */
    @SuppressWarnings("unchecked")
    public void setModelTypeClass(String className) {
        try {
            modelType = (Class<M>) Class.forName(className);
        } catch (ClassNotFoundException e) {
            throw new G9BaseException("Unable to get model type class", e);
        }
    }

    /**
     * Sets the format to apply
     * @param format the format to set
     */
    public void setFormat(String format) {
        this.format = new MessageFormat(format);
    }

    @Override
    public Class<M> getModelType() {
        return modelType;
    }

    @Override
    public Class<String> getTargetType() {
        return String.class;
    }

    @SuppressWarnings("unchecked")
    @Override
    public M toModel(String value, ConvertContext context) throws ConvertException {
        if (value == null) {
            return null;
        }
        try {
            M[] objs = (M[]) format.parse(value);
            return objs[0];
        } catch (Exception e) {
            throw new ConvertException(context, value, "Unable to parse", e);
        }
    }

    @Override
    public String fromModel(Object value, ConvertContext context) throws ConvertException {
        if (value == null) {
            return null;
        }
        try {
            return format.format(new Object[] {value});
        } catch (Exception e) {
            throw new ConvertException(context, value, "Unable to format", e);
        }
    }

}
