/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.util;

import java.util.Collections;
import java.util.List;

import no.esito.jvine.util.ViewConstantHelper;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.controller.DialogObjectType;

/**
 * Utility methods related to DialogObjectConstant.
 */
public class DialogObjectConstantHelper {

    /**
     * Get the tab order for a window block. The returned list
     * is sorted according to the tab order in the dialog model.
     * Dialog model widgets which are removed from the tab order are
     * placed first in the list, as they have a tab index of -1.
     * Dialog model widgets which cannot be part of the tab order
     * have a tab index of 0. Dialog model widgets which are part
     * of the tab order have a tab index &gt; 0.
     *
     * @param windowBlock the window block with a tab order
     * @return the list of widgets in the window block, sorted according to the tab order
     */
    public static List<DialogObjectConstant> getTabOrder(DialogObjectConstant windowBlock) {
        if (!DialogObjectType.WindowBlock.equals(windowBlock.getType())) {
            return null;
        }
        List<DialogObjectConstant> descendants = ViewConstantHelper.getDescendants(windowBlock);
        Collections.sort(descendants, new TabIndexComparator());
        return descendants;
    }

    /**
     * Check if attribute type is of Java time.
     * 
     * @param field to check
     * @return true if LocalDate, LocalTime or LocalDateTime, else false
     */
    public static boolean isJavaTimeType(DialogObjectConstant field) {
        return field.getAttribute().getAttributeType().getName() == "java.time.LocalDate" 
        		|| field.getAttribute().getAttributeType().getName() == "java.time.LocalTime"
        		|| field.getAttribute().getAttributeType().getName() == "java.time.LocalDateTime";
    }
    
    /**
     * Check if attribute type is LocalDate.
     * 
     * @param field to check
     * @return true if LocalDate, else false
     */
    public static boolean isLocalDate(DialogObjectConstant field) {
        return field.getAttribute().getAttributeType().getName() == "java.time.LocalDate";
    }

    /**
     * Check if attribute type is LocalTime.
     * 
     * @param field to check
     * @return true if LocalTime, else false
     */
    public static boolean isLocalTime(DialogObjectConstant field) {
        return field.getAttribute().getAttributeType().getName() == "java.time.LocalTime";
    }

    /**
     * Check if attribute type is LocalDateTime.
     * 
     * @param field to check
     * @return true if LocalDateTime, else false
     */
    public static boolean isLocalDateTime(DialogObjectConstant field) {
        return field.getAttribute().getAttributeType().getName() == "java.time.LocalDateTime";
    }

    /**
     * Check if attribute type is Joda DateTime.
     * 
     * @param field to check
     * @return true if DateTime, else false
     */
    public static boolean isJodaType(DialogObjectConstant field) {
        return field.getAttribute().getAttributeType().getName() == "org.joda.time.DateTime" || field.getAttribute().getAttributeType().getName() == "org.joda.time.DateMidnight";
    }

    /**
     * Check if widget type is ace_AutoCompleteEntry.
     * 
     * @param field to check
     * @return true if ace_AutoCompleteEntry, else false
     */
    public static boolean isAutoCompleteEntry(DialogObjectConstant field) {
        return field.getType() == DialogObjectType.DataComponent && "ace_AutoCompleteEntry".equals(field.getComponentType());
    }
    
    /**
     * Check if widget type is ace_DateTimeEntry.
     * 
     * @param field to check
     * @return true if ace_DateTimeEntry, else false
     */
    public static boolean isDateTimeEntry(DialogObjectConstant field) {
        return field.getType() == DialogObjectType.DataComponent && "ace_DateTimeEntry".equals(field.getComponentType());
    }
    
}
