/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.validator;

import java.text.ParseException;
import java.text.SimpleDateFormat;

/**
 * A configurable date field validator.
 *
 * This class uses a java.text.SimpleDateFormat to validate fields. Refer to the
 * documentation of that class to see how patters are defined.
 *
 * Example of usage:
 * {@code
 * <bean id="dateValidator"
 *     class="no.g9.client.core.validator.DateValidator">
 *     <property name="datePattern" value="yyyy.MM.dd" />
 * </bean>
 *
 * This bean supports the following properties:
 * <li> datePattern: Pattern to validate against. @see java.text.SimpleDateFormat
 * <li> setLenient: Specify whether or not date/time parsing is to be lenient
 * <li> failMessageNumber: Directs what message number to return on validation
 *      fail.
 * <li> failMessageArgs: An array that specifies fail message arguments}
 *
 */
public class DateValidator implements FieldValidator {

    private SimpleDateFormat pattern;
    private String failMessageID = ValidationResult.DEFAULT_FAIL_MESSAGE;
    private Object[] failMessageArgs = null;
    private boolean lenient = false;

    /**
     * Sets the pattern to match against
     * @param pattern The date pattern to validate against
     * @see java.text.SimpleDateFormat
     */
    public void setDatePattern(String pattern) {
        this.pattern = new SimpleDateFormat(pattern);
        this.pattern.setLenient(lenient);
    }

    /**
     * @param lenient the lenient to set
     */
    public void setLenient(boolean lenient) {
        this.lenient = lenient;
        if (pattern != null) {
            pattern.setLenient(lenient);
        }
    }

    /**
     * Sets message number to return on failed validations.
     * @param messageID The message number to return on failed validations
     */
    public void setFailMessageNumber(String messageID) {
        this.failMessageID = messageID;
    }

    /**
     * Sets message text to return on failed validations.
     * @param messageArgs The message text to return on failed validations
     */
    public void setFailMessageArgs(Object[] messageArgs) {
        this.failMessageArgs = (messageArgs != null) ? messageArgs.clone() : null;
    }

    @Override
    public ValidationResult validate(Object value, ValidateContext context) {
        if (value == null) return ValidationResult.DEFAULT_OK_RESULT;
        try {
            pattern.parse(value.toString());
            return ValidationResult.DEFAULT_OK_RESULT;
        } catch (ParseException e) {
            return new ValidationResult(false, failMessageID, failMessageArgs);
        }
    }

}
