/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.validator;

import java.util.Map;

import no.g9.exception.G9BaseException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;

/**
 * Exception thrown to indicate that one or more validators has failed
 * validation. The exception keeps a map of the failed validation results and
 * validate contexts.
 *
 */
public class ValidationException extends G9BaseException {

    private final Map<ValidationResult, ValidateContext> validationResult;

	/**
	 * Create a new validation exception.
	 * 
	 * @param validationResult map of failed validation results and corresponding validate context
	 */
	public ValidationException(Map<ValidationResult, ValidateContext> validationResult) {
		this.validationResult = validationResult;

		Message message = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.CF_VALIDATION_FAILED_MAIN);
		message.setContext(null);

		for (Map.Entry<ValidationResult, ValidateContext> entry : validationResult.entrySet()) {
			ValidateContext validateContext = entry.getValue();

			if (validationResult.size() > 1) {
				Message childMessage = MessageSystem.getMessageFactory().getMessage(entry.getKey().getMsgNumber(), entry.getKey().getMsgArgs());

				childMessage.setContext(validateContext);
				message.addMessage(childMessage);
			} else {
				message = MessageSystem.getMessageFactory().getMessage(entry.getKey().getMsgNumber(), entry.getKey().getMsgArgs());
				message.setContext(validateContext);
			}
		}
		setErrMsg(message);
	}

    /**
     * Get the map of failed validation results and corresponding validate
     * contexts.
     *
     * @return map of failed validations and contexts
     */
    public Map<ValidationResult, ValidateContext> getValidationResults() {
        return validationResult;
    }

}
