/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.table;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Observable;
import java.util.Observer;

import no.esito.jvine.view.ViewModelImpl;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.BooleanProperty;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.Property;
import no.g9.client.core.view.ViewModel;
import no.g9.client.core.view.table.ListRowComparator.Sorting;
import no.g9.os.RoleConstant;

/**
 * The default table model. This contains the complete collection of
 * <code>ListRow</code>s. The selection model is set to
 * {@link TableModel.SelectionModel#DEFAULT}.
 *
 * @param <T>
 *            The generated ListRow class
 */
public class DefaultTableModel<T extends ListRow> implements TableModel<T> {
    private final TableData<T> tableData;
    private final TableData<T> tableView;
    private final List<DialogObjectConstant> columns = new ArrayList<DialogObjectConstant>();
    private final List<RowFilter<?, ListRow>> rowFilters = new ArrayList<RowFilter<?, ListRow>>();
    private final ViewModelImpl viewModel;

    /**
     * Create a new table model based on the specified columns.
     *
     * @param columns the column definition of this table model
     * @param viewModel the view model for the table
     */
    public DefaultTableModel(List<DialogObjectConstant> columns, ViewModel viewModel) {
        this.columns.addAll(columns);
        this.viewModel = (ViewModelImpl) viewModel;
        tableData = new TableData<T>(columns);
        tableView = new TableData<T>(columns);
        tableData.addObserver(new Observer() {

            @Override
            public void update(Observable o, Object arg) {
                filterTableView();
                sortTableView();
            }
        });
    }

    /**
     * Create a new table model based on the specified columns.
     *
     * @param columns the column definition of this table model
     * @param viewModel the view model for the table 
     */
    public DefaultTableModel(DialogObjectConstant[] columns, ViewModel viewModel) {
        this(Arrays.asList(columns), viewModel);
    }

    @Override
    public List<T> getTableData() {
        return tableData;
    }

    @Override
    public List<T> getTableView() {
        return tableView;
    }

    @Override
    public void addRowFilter(RowFilter<?, ListRow> rowFilter) {
        this.rowFilters.add(rowFilter);
        filterTableView();
    }

    @Override
    public void addRowFilters(Collection<RowFilter<?, ListRow>> rowFilters) {
        this.rowFilters.addAll(rowFilters);
        filterTableView();
    }

    @Override
    public void removeRowFilter(RowFilter<?, ListRow> rowFilter) {
        this.rowFilters.remove(rowFilter);
        filterTableView();
    }

    @Override
    public void removeRowFilters(Collection<RowFilter<?, ListRow>> rowFilters) {
        this.rowFilters.removeAll(rowFilters);
        filterTableView();
    }

    @Override
    public void clearRowFilters() {
        this.rowFilters.clear();
    }

    @Override
    public Collection<RowFilter<?, ListRow>> getRowFilters() {
        return Collections.unmodifiableCollection(this.rowFilters);
    }

    // ////////////////////////////////////////////////////////
    // /////////////// Selection model //////////////
    // ////////////////////////////////////////////////////////
    private SelectionModel selectionModel = SelectionModel.DEFAULT;
    private List<ListRowComparator<T>> comparatorList;
    private ListRowComparator<T> firstComparator;

    @Override
    public void setSelectionModel(SelectionModel selectionModel) {
        this.selectionModel = selectionModel;
        if (selectionModel == SelectionModel.NO_SELECT) {
            unselectAll();
        }
    }

    @Override
    public SelectionModel getSelectionModel() {
        return selectionModel;
    }

    // ////////////////////////////////////////////////////////
    // /////////////// Cell value methods //////////////
    // ////////////////////////////////////////////////////////

    @Override
    public Object getValueAt(int rowIndex, DialogObjectConstant columnIdentifier) {
        tableData.checkColumn(columnIdentifier);
        return tableData.get(rowIndex).getValue(columnIdentifier);
    }

    @Override
    public void setValueAt(int rowIndex, DialogObjectConstant columnIdentifier,
            Object value) {
        tableData.checkColumn(columnIdentifier);
        T listrow = tableData.get(rowIndex);
        listrow.setValue(columnIdentifier, value);
    }

    @Override
    public void filterTableView() {
        tableView.clear();
        for (T row : tableData) {
            processRow(row, false);
        }
    }

    private void processRow(T row, boolean checkTableView) {
        boolean inTableView = checkTableView ? tableView.contains(row) : false;
        boolean passFilters = passFilters(row);
        if (passFilters && !inTableView) {
            tableView.add(row);
        } else if (!passFilters && inTableView) {
            tableView.remove(row);
        }
    }

    private boolean passFilters(T row) {
        for (RowFilter<?, ListRow> filter : rowFilters) {
            if (!filter.passFilter(row)) {
                return false;
            }
        }
        return true;
    }

    // ////////////////////////////////////////////////////////
    // //////////////// row and column counts /////////////////
    // ////////////////////////////////////////////////////////

    @Override
    public int getTableDataRowCount() {
        return tableData.size();
    }

    @Override
    public int getTableViewRowCount() {
        return tableView.size();
    }

    @Override
    public int getColumnCount() {
        return columns.size();
    }

    @Override
    public int indexOf(ListRow row) {
        return tableData.indexOf(row);
    }

    @Override
    public int toTableDataIndex(int tableViewIndex) {
        T row = getTableView().get(tableViewIndex);
        return indexOf(row);
    }

    // ////////////////////////////////////////////////////////
    // ///////////////adding and removing rows/////////////////
    // ////////////////////////////////////////////////////////
    @Override
    public T get(int rowIndex) {
        return tableData.get(rowIndex);
    }

    @SuppressWarnings("unchecked")
	@Override
    public void insert(int rowIndex, T... row) {
        insert(rowIndex, Arrays.asList(row));

    }

    @Override
    public void insert(int rowIndex, Collection<T> rows) {
        tableData.addAll(rowIndex, rows);
    }

    @SuppressWarnings("unchecked")
	@Override
    public void add(T... row) {
        add(Arrays.asList(row));
    }

    @Override
    public void add(Collection<T> rows) {
        tableData.addAll(rows);
    }

    @Override
    public T remove(int rowIndex) {
        return tableData.remove(rowIndex);
    }

    @SuppressWarnings("unchecked")
	@Override
    public boolean remove(T... row) {
        return removeAll(Arrays.asList(row));
    }

    @Override
    public boolean removeAll(Collection<T> rows) {
        boolean isChanged = tableData.removeAll(rows);
        tableView.removeAll(rows);
        return isChanged;
    }

    @Override
    public void clear() {
        tableData.clear();
    }

    @Override
    public boolean isEmpty() {
        return tableData.isEmpty();
    }

    // ////////////////////////////////////////////////////////
    // /////////// SELECTING ROWS /////////////////////////
    // ////////////////////////////////////////////////////////
    @Override
    public void setSelected(int rowIndex, boolean selected) {
        T row = get(rowIndex);
        setSelected(row, selected);
    }



    @SuppressWarnings("boxing")
    @Override
    public void setSelected(T row, boolean selected) {
        switch (selectionModel) {
        case SINGLE_SELECT:
            unselectAll();
            //$FALL-THROUGH$
        case MULTI_SELECT:
            row.setSelected(selected);
            break;
        case NO_SELECT:
            if (selected) {
                throw new IllegalStateException(
                        "Current selection model prohibits selection of a row");
            }
            row.setSelected(selected);
            break;
        default:
            break;
        }
        viewModel.updateTableFieldValues(getTableRole());
    }

    @Override
    public void setSelected(boolean selected) {
        if (selected && (selectionModel == SelectionModel.SINGLE_SELECT || selectionModel == SelectionModel.NO_SELECT)) {
            throw new IllegalStateException("Current selection model prohibits selection of a row");
        }
        for (ListRow row : tableData) {
            row.setSelected(Boolean.valueOf(selected));
        }
        viewModel.updateTableFieldValues(getTableRole());
    }

    private void unselectAll() {
        for (ListRow row : tableData) {
            row.setSelected(Boolean.FALSE);
        }
        viewModel.updateTableFieldValues(getTableRole());
    }

    @Override
    public boolean isSelected(int rowIndex) {
        ListRow row = tableData.get(rowIndex);
        return row.isRowSelected();
    }

    @Override
    public List<T> getSelected() {
        ArrayList<T> accum = new ArrayList<T>();
        for (T listRow : tableData) {
            if (listRow.isRowSelected()) {
                accum.add(listRow);
            }
        }
        return accum;
    }

    @Override
    public int getSelectionCount() {
        return getSelected().size();
    }

    /**
     * @return the role for the table
     */
    protected RoleConstant getTableRole() {
    	for (RoleConstant role : viewModel.getListRoles().values()) {
    		if (viewModel.getTableModel(role) == this) {
    			return role;    			
    		}
    	}
    	return null;
    }

    /**
     * @return the view model used by this table model
     */
    protected ViewModel getViewModel() {
    	return viewModel;
    }

    // ////////////////////////////////////////////////////////
    // //////// ENABLED PROPERTY ///////////
    // ////////////////////////////////////////////////////////

    @Override
    public void setEnabled(boolean enabled) {
        setProperty(BooleanProperty.ENABLED, Boolean.valueOf(enabled));

    }

    @SuppressWarnings("boxing")
    @Override
    public void setRowEnabled(int rowIndex, boolean enabled) {
        setRowProperty(rowIndex, BooleanProperty.ENABLED, enabled);
    }



    @SuppressWarnings("boxing")
    @Override
    public void setRowEnabled(T row, boolean enabled) {
        setRowProperty(row, BooleanProperty.ENABLED, enabled);

    }

    @SuppressWarnings("boxing")
    @Override
    public void setColumnEnabled(DialogObjectConstant columnIdentifier,
            boolean enabled) {
        setColumnProperty(columnIdentifier, BooleanProperty.ENABLED, enabled);
    }

    @SuppressWarnings("boxing")
    @Override
    public void setCellEnabled(int rowIndex,
            DialogObjectConstant columnIdentifier, boolean enabled) {
        this.setCellProperty(rowIndex, columnIdentifier,
                BooleanProperty.ENABLED, enabled);
    }

    @SuppressWarnings("boxing")
    // Can't be null so boxing is fine :)
    @Override
    public boolean isCellEnabled(int rowIndex,
            DialogObjectConstant columnIdentifier) {
        return getCellProperty(rowIndex, columnIdentifier,
                BooleanProperty.ENABLED);
    }

    // ////////////////////////////////////////////////////////
    // ////// SHOW PROPERTY /////////////////////////////
    // ////////////////////////////////////////////////////////

    @Override
    public void setShown(boolean shown) {
        setProperty(BooleanProperty.SHOWN, Boolean.valueOf(shown));
    }

    @SuppressWarnings("boxing")
    @Override
    public void setRowShown(int rowIndex, boolean shown) {
        setRowProperty(rowIndex, BooleanProperty.SHOWN, shown);
    }



    @SuppressWarnings("boxing")
    @Override
    public void setRowShown(T row, boolean shown) {
        setRowProperty(row, BooleanProperty.SHOWN, shown);
    }

    @SuppressWarnings("boxing")
    @Override
    public void setColumnShown(DialogObjectConstant columnIdentifier,
            boolean shown) {
        setColumnProperty(columnIdentifier, BooleanProperty.SHOWN, shown);
    }

    @SuppressWarnings("boxing")
    @Override
    public void setCellShown(int rowIndex,
            DialogObjectConstant columnIdentifier, boolean shown) {
        setCellProperty(rowIndex, columnIdentifier, BooleanProperty.SHOWN,
                shown);
    }

    @SuppressWarnings("boxing")
    // Can't be null, boxing is fine :)
    @Override
    public boolean isCellShown(int rowIndex,
            DialogObjectConstant columnIdentifier) {
        return getCellProperty(rowIndex, columnIdentifier,
                BooleanProperty.SHOWN);
    }

    // ////////////////////////////////////////////////////////
    // ///////GENERAL PROPERTY ////
    // ////////////////////////////////////////////////////////

    @Override
    public <U> void setProperty(Property<U> property, U propertyValue) {
        for (T row : tableData) {
            setRowProperty(row, property, propertyValue);
        }
    }

    @Override
    public <U> void setRowProperty(int rowIndex, Property<U> property,
            U propertyValue) {
        T row = tableData.get(rowIndex);
        setRowProperty(row, property, propertyValue);
    }

    @Override
    public <U> void setRowProperty(T row, Property<U> property, U propertyValue) {
        for (DialogObjectConstant columnIdentifier : columns) {
            row.<U> setProperty(columnIdentifier, property, propertyValue);
        }
    }

    @Override
    public <U> void setColumnProperty(DialogObjectConstant columnIdentifier,
            Property<U> propertyName, U value) {
        tableData.checkColumn(columnIdentifier);
        for (ListRow listRow : tableData) {
            listRow.<U> setProperty(columnIdentifier, propertyName, value);
        }
    }

    @Override
    public <U> void setCellProperty(int rowIndex,
            DialogObjectConstant columnIdentifier, Property<U> propertyName,
            U propertyValue) {
        tableData.checkColumn(columnIdentifier);
        ListRow listRow = tableData.get(rowIndex);
        listRow.<U> setProperty(columnIdentifier, propertyName, propertyValue);
    }

    @Override
    public <U> U getCellProperty(int rowIndex,
            DialogObjectConstant columnIdentifier, Property<U> propertyName) {
        tableData.checkColumn(columnIdentifier);
        ListRow listRow = tableData.get(rowIndex);
        U value = listRow.getProperty(columnIdentifier, propertyName);
        return value;
    }

    @Override
    public DialogObjectConstant getFirstSortingColumn() {
        if (firstComparator == null) {
            return null;
        }
        ListRowComparator<T> found = firstComparator;
        while (found != null && found.getSorting() == Sorting.NO_SORT) {
            found = found.getNextComparator();
        }
        return found != null ? found.getColumn() : null;
    }

    @Override
    public void setFirstSortingColumn(DialogObjectConstant column) {
        if (comparatorList == null) {
            comparatorList = new ArrayList<ListRowComparator<T>>();
        }
        ListRowComparator<T> found = null;
        for (ListRowComparator<T> comparator : comparatorList) {
            if (comparator.getColumn().equals(column)) {
                found = comparator;
                break;
            }
        }

        if (found == null) {
            found = new ListRowComparator<T>(column, 0);
        }
        comparatorList.remove(found);
        comparatorList.add(0, found);
        firstComparator = found;
    }

    @Override
    public void sortTableView() {
        if (firstComparator != null) {
            Collections.sort(getTableView(), firstComparator);
        }
    }

    /**
     * Get the current comparator list.
     * Note that this method is intended for internal use.
     * Programmers should use the methods defined in <code>TableModel</code>.
     *
     * @return the current comparator list
     */
    public List<ListRowComparator<T>> getListRowComparator() {
        return comparatorList;
    }

    @Override
    public void setListRowComparator(List<ListRowComparator<T>> comparatorList) {
        this.comparatorList = comparatorList;
        firstComparator = ListRowComparator.chainComparator(comparatorList);
    }

    @Override
    public void toggleSorting() {
        if (firstComparator != null) {
            firstComparator.toggleSorting();
        }
    }

    @Override
    public Sorting getSorting() {
        if (firstComparator != null) {
            return firstComparator.getSorting();
        }
        return null;
    }

    @Override
    public void setSorting(Sorting sortDirection) {
        if (firstComparator != null) {
            firstComparator.setSorting(sortDirection);
        }
    }

    @Override
    public void prettyPrint(PrintStream out) {
        if (tableData != null) {
            tableData.printTable(out);
            printSelectedLines(out);
        } else {
            out.println("No data");
        }
    }

    private void printSelectedLines(PrintStream out) {
        String line = "Selected rows are: [";
        boolean flag = false;
        for (int i = 0; i < tableData.size(); i++) {
            if (isSelected(i)) {
                flag = true;
                line += i + ", ";
            }
        }
        if (flag) {
            line = line.substring(0, line.length() - ", ".length());
        }
        line += "]";

        out.println(line);
    }


}
