/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.table;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;
import java.util.Observable;

import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.ListRow;

/**
 * TableData is used to keep all rows in a TableModel.
 *
 * @param <E>
 *            the implementing ListRow class
 */
class TableData<E extends ListRow> extends Observable implements List<E> {

    private List<E> list = new ArrayList<E>();
    private boolean rowCheck = false;
    private final List<DialogObjectConstant> columns;

    /**
     * Construct a new TableData instance. The TableData will only accept
     * ListRows that matches the specified column definition.
     *
     * @param columnDefinition
     *            list defining the table data's columns
     */
    public TableData(List<DialogObjectConstant> columnDefinition) {
        columns = Collections.unmodifiableList(columnDefinition);
    }

    @Override
    public void add(int index, E row) {
        checkRow(row);
        list.add(index, row);
        setChanged();
        notifyObservers();
    }

    /**
     * Inserts a row or rows at the specified position. Shifts the row at that
     * position (if any) an any subsequent rows down.
     *
     * @param rowIndex
     *            index at which the specified row is to be inserted
     * @param row
     *            row to be inserted
     * @return <code>true</code> if the table data changed as a result of
     *         invoking this method
     * @throws IndexOutOfBoundsException
     *             if the specified row index is out of range (
     *             <code>rowIndex < 0 || rowIndex > getRowCount()</code>)
     * @throws NullPointerException
     *             if the specified row is <code>null</code>
     *
     * @throws IllegalArgumentException
     *             if the rows does not match the column definitions
     */
    @SuppressWarnings("unchecked")
	boolean insert(int rowIndex, E... row) {
        return addAll(rowIndex, Arrays.asList(row));
    }

    /**
     * Inserts a row or rows at the specified position. Shifts the row at that
     * position (if any) an any subsequent rows down. The internal ordering of
     * the inserted rows are determined by the iteration order of the
     * <code>rows</code> collection.
     *
     * @param rowIndex
     *            index at which the specified row is to be inserted
     * @param rows
     *            rows to be inserted
     * @return <code>true</code> if the table data changed as a result of
     *         invoking this method
     * @throws IndexOutOfBoundsException
     *             if the specified row index is out of range (
     *             <code>rowIndex < 0 || rowIndex > getRowCount()</code>)
     * @throws NullPointerException
     *             if the specified row is <code>null</code>
     * @throws IllegalArgumentException
     *             if the rows does not match the column definitions
     */
    boolean insert(int rowIndex, Collection<E> rows) {
        return addAll(rowIndex, rows);
    }

    @Override
    public boolean add(E row) {
        checkRow(row);
        boolean changed = list.add(row);
        if (changed) {
            setChanged();
            notifyObservers();
        }
        return changed;
    }

    @Override
    public boolean addAll(Collection<? extends E> rows) {
        checkRow(rows);
        boolean changed = list.addAll(rows);
        if (changed) {
            setChanged();
            notifyObservers();
        }
        return changed;
    }

    @Override
    public boolean addAll(int index, Collection<? extends E> rows) {
        checkRow(rows);
        boolean changed = list.addAll(index, rows);
        if (changed) {
            setChanged();
            notifyObservers();
        }
        return changed;
    }

    @Override
    public void clear() {
        int currSize = size();
        list.clear();
        if (size() != currSize) {
            setChanged();
            notifyObservers();
        }
    }

    @Override
    public boolean contains(Object o) {
        return list.contains(o);
    }

    @Override
    public boolean containsAll(Collection<?> c) {
        return list.containsAll(c);
    }

    @Override
    public E get(int index) {
        return list.get(index);
    }

    @Override
    public int indexOf(Object o) {
        return list.indexOf(o);
    }

    @Override
    public boolean isEmpty() {
        return list.isEmpty();
    }

    @Override
    public Iterator<E> iterator() {
        return list.iterator();
    }

    @Override
    public int lastIndexOf(Object o) {
        return list.lastIndexOf(o);
    }

    @Override
    public ListIterator<E> listIterator() {
        return list.listIterator();
    }

    @Override
    public ListIterator<E> listIterator(int index) {
        return list.listIterator(index);
    }

    @Override
    public E remove(int index) {
        E changed = list.remove(index);
        if (changed != null) {
            setChanged();
            notifyObservers();
        }
        return changed;
    }

    @Override
    public boolean remove(Object o) {
        boolean changed = list.remove(o);
        if (changed) {
            setChanged();
            notifyObservers();
        }
        return changed;
    }

    /**
     * Removes specified row or rows at the specified index. Subsequent rows (if
     * any) are shifted up.
     *
     * @param row
     *            the row or rows to remove.
     * @return <code>true</code> if this list changed as a result of the call
     * @throws NoSuchElementException
     *             if the specified rows are not found in this list. If this
     *             exception is thrown, no rows are removed.
     */
    @SuppressWarnings("unchecked")
	boolean remove(E... row) {
        return removeAll(Arrays.asList(row));
    }

    @Override
    public boolean removeAll(Collection<?> c) {
        boolean changed = list.removeAll(c);
        if (changed) {
            setChanged();
            notifyObservers();
        }
        return changed;
    }

    @Override
    public boolean retainAll(Collection<?> c) {
        boolean changed = list.retainAll(c);
        if (changed) {
            setChanged();
            notifyObservers();
        }
        return changed;
    }

    @Override
    public E set(int index, E element) {
        E row = list.set(index, element);
        setChanged();
        notifyObservers();
        return row;
    }

    @Override
    public int size() {
        return list.size();
    }

    @Override
    public List<E> subList(int fromIndex, int toIndex) {
        return list.subList(fromIndex, toIndex);
    }

    @Override
    public Object[] toArray() {
        return list.toArray();
    }

    @Override
    public <T> T[] toArray(T[] a) {
        return list.toArray(a);
    }

    /**
     * Get the number of columns in this table model.
     *
     * @return the column count
     */
    public int getColumnCount() {
        return columns.size();
    }

    /**
     * Checks if this row has the same columns as this table.
     *
     * @param row
     *            ListRow to check
     *
     * @throws NullPointerException
     *             if the specified row is <code>null</code>
     * @throws IllegalArgumentException
     *             if the structure does not match.
     *
     */
    void checkRow(E row) {
        if (rowCheck) {
            return;
        }
        if (row == null) {
            throw new NullPointerException();
        }
        if (!columns.containsAll((row.getFields())) && !row.getFields().containsAll(columns)) {

            String msg = "Row " + row
                    + " does not match column definition";
            throw new IllegalArgumentException(msg);
        }
        rowCheck = true;
    }

    /**
     * Check that the rows column definitions match that of this table.
     *
     * @param rows collection of rows to test
     */
    void checkRow(Collection<? extends E> rows) {
        for (E e : rows) {
            checkRow(e);
            break;
        }
    }

    /**
     * Checks if this column is a part of this table.
     *
     * @param columnIdentifier
     *            constant denoting the column
     *
     * @throws NoSuchElementException
     *             if the specified column is not part of this table model
     *
     */
    void checkColumn(DialogObjectConstant columnIdentifier) {
        if (!columns.contains(columnIdentifier)) {
            throw new NoSuchElementException();
        }
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((columns == null) ? 0 : columns.hashCode());
        result = prime * result + ((list == null) ? 0 : list.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TableData<?>)) {
            return false;
        }
        TableData<?> other = (TableData<?>) obj;
        if (columns == null) {
            if (other.columns != null) {
                return false;
            }
        } else if (!columns.equals(other.columns)) {
            return false;
        }
        if (list == null) {
            if (other.list != null) {
                return false;
            }
        } else if (!list.equals(other.list)) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        return "TableData size=" + size() + " [columns=" + columns + "]";
    }

    private String colSep = " | ";

    /**
     * Pretty print table contents
     * @param out where to print
     */
    void printTable(PrintStream out) {
        printHeader(out);
        printBody(out);
    }

    private void printHeader(PrintStream out) {
        String[] headers = getColumns();
        String line = "";

        for (String string : headers) {
            line += string + colSep;
        }
        line = line.substring(0, line.length() - colSep.length());
        out.println(line);
        String lb = "";
        for (int i = 0; i < line.length(); i++) {
            lb +="-";
        }
        out.println(lb);
    }

    private String[] getColumns() {
        List<String> cols = new ArrayList<String>();
        for (DialogObjectConstant col : columns) {
            cols.add(col.toString());
        }
        return cols.toArray(new String[cols.size()]);
    }

    private void printBody(PrintStream out) {
        for (ListRow row : list) {
            String line = "";
            for (DialogObjectConstant column : columns) {
                String colHeader = column.toString();
                String value = String.valueOf(row.getValue(column));
                value = formatValue(colHeader, value);
                line += value + colSep;
            }
            line = line.substring(0, line.length() - colSep.length());
            out.println(line);
        }
    }

    private String formatValue(String columnName, String cellValue) {
        if (cellValue.length() > columnName.length()) {
            if (columnName.length() <= 3) {
                cellValue = "...".substring(0, columnName.length());
            } else {
                cellValue = cellValue.substring(0, columnName.length() - 3) + "...";
            }
        } else {
            for (int i = cellValue.length() - 1; i < columnName.length() - 1; i++) {
                cellValue += " ";
            }
        }
        return cellValue;
    }
}
