/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.tree;

import java.util.Collection;
import java.util.List;
import java.util.NoSuchElementException;

import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.Property;
import no.g9.client.core.view.PropertyManager;
import no.g9.client.core.view.table.RowFilter;

/**
 * The general tree model used by jVine targets. The tree model is used to 
 * hold all the data that can be displayed by a tree view in addition to various
 * properties concerning node, row and cells. 
 * <p>
 * {@link TreeNode} is used to represent nodes. Each node 
 * contains a {@link ListRow} as its data representation. When setting a state or 
 * property on a node, the state or property is propagated to the nodes statemap 
 * and the ListRow's {@link PropertyManager}. The tree model has a list of root nodes.
 * 
 * <p>
 * The tree model maintains two lists of root TreeNodes:
 * <ul>
 * <li>The <em>tree data</em> list is the complete list of root nodes in this model. 
 * The list can be accessed by {@link #getTreeData()} method.
 * 
 * <li>The <em>view data</em> list is the filtered and sorted list of root nodes in this model.
 * The list can be accessed by {@link #getTreeView()} method.
 * </ul>
 *
 * @param <T> The actual implementing TreeNode class
 * @param <L> The actual implementing ListRow class
 */
public interface TreeModel<T extends TreeNode, L extends ListRow> {
	
    /**
     * Enumerates the various node selection models.
     */
    public enum SelectionModel implements Property<Enum<SelectionModel>> {

        /** No node selection is possible. */
        NO_SELECT,

        /** At most one node can be selected */
        SINGLE_SELECT,

        /** Selection of several node is possible */
        MULTI_SELECT;

        @Override
        public Enum<SelectionModel> getDefaultValue() {
            return DEFAULT;
        }

        /** The default selection model which is SINGLE_SELECT */
        public static final SelectionModel DEFAULT = SINGLE_SELECT;

        @Override
        public String toString() {
            return SelectionModel.class.getSimpleName() + ": " + name();
        }
    }
    
    /**
     * Enumerates the various grouping modes.
     *
     */
    public enum Grouping implements Property<Enum<Grouping>> {
    	
    	/** No grouping. */
    	NONE(0),
    	
    	/** Grouped by node type, each node type is grouped ascending according to
    	 *  concatenated cell values. */
        GROUPED(2),
        
        /** Grouped ascending to concatenated cell values without regard to node type. */
        INTERLEAVED_ASCENDING(1),
        
        /** Grouped descending to concatenated cell values without regard to node type. */
        INTERLEAVED_DESCENDING(-1);

    	@SuppressWarnings("javadoc")
        int value;
    	
    	Grouping(int value) {
    		this.value = value;
    	}
    	
        @Override
        public Enum<Grouping> getDefaultValue() {
            return DEFAULT;
        }

        /** The default grouping mode is GROUPED */
        public static final Grouping DEFAULT = GROUPED;

        @Override
        public String toString() {
            return Grouping.class.getSimpleName() + ": " + name();
        }
    }

    /**
     * Get the tree root node view list. This is the sorted and filtered list of 
     * root nodes presented to dialog views.
     * <p>
     * <b>Note</b> that any changes made directly to this list will not be
     * detected by the tree model. Thus adding or removing rows to this list
     * might not yield the desired result.
     *
     * @return tree root node view list
     */
    List<T> getTreeView();

    /**
     * Get the tree root node data list. This is the list used for maintaining the
     * unsorted and un-filtered tree root node list.
     *
     * <p>
     * <b>Note</b> that any changes made directly to this list will be detected
     * by the tree model and trigger and update of the tree view list.
     *
     * @return tree root node data list
     */
    List<T> getTreeData();
    
    
    /**
     * Get the concrete data model. For eagerly fetched trees it will be a List of 
     * root nodes, for lazily loaded trees it will be an implementation of LazyNodeDataModel.
     *  
     * @return the datamodel for the tree.
     */
    @SuppressWarnings("javadoc")
    Object getDataModel();

	  ////////////////////////////////
	 /// Filtering //////////////////
    ////////////////////////////////
    /**
     * Add the row filter to this tree model. The <em>tree view</em> list will
     * be updated.
     *
     * @param rowFilter
     *            row filter(s) to add
     */
	void addRowFilter(RowFilter<?, ListRow> rowFilter);
	
    /**
     * Add the row filters to this tree model. The <em>tree view</em> list
     * will be updated.
     *
     * @param rowFilters
     *            row filter(s) to add
     */
	void addRowFilters(Collection<RowFilter<?, ListRow>> rowFilters);
	
    /**
     * Remove the row filter from this tree model. The
     * <em>tree view</em> list will be updated.
     *
     * @param rowFilter
     *            row filter(s) to remove
     */
	void removeRowFilter(RowFilter<?, ListRow> rowFilter);
	
    /**
     * Remove the row filters from this tree model. The
     * <em>tree view</em> list will be updated.
     *
     * @param rowFilters
     *            row filters to remove
     */
	void removeRowFilters(Collection<RowFilter<?, ListRow>> rowFilters);
	
	/**
	 * Filter the tree, given a list of root nodes.
	 * 
	 * @param srcList source list of root nodes
	 * @param dstList destination list of root nodes
	 */
	void filterTree(List<T> srcList, List<T> dstList); 
	
	  ////////////////////////////////
	 /// Cell values ////////////////
	////////////////////////////////
    /**
     * Get the cell value at the specified node and column.
     *
     * @param node
     *            tree data node
     * @param columnIdentifier
     *            constant denoting the column
     * @return the cell value
     */
	Object getValueAt(T node, DialogObjectConstant columnIdentifier);
	
    /**
     * Set the specified cell value.
     * <p>
     * <b>Note:</b> The tree view is <em>not</em> updated as a result of
     * invoking this method.
     *
     * @param node
     *            tree data's node
     * @param columnIdentifier
     *            constant denoting the column
     * @param value
     *            cell value
     * @see #filterTree(List, List)
     */
	@SuppressWarnings("javadoc")
    void setValueAt(T node, DialogObjectConstant columnIdentifier, Object value);
	
	  ////////////////////////////////
	 /// Counting ///////////////////
	////////////////////////////////
    /**
     * Get the number of nodes in this tree data model.
     *
     * @return the data model's node count
     */
	int getTreeDataNodeCount();
	
    /**
     * Get the number of nodes in this tree view model.
     *
     * @return the view model's node count
     */
	int getTreeViewNodeCount();
	

	/**
	 * Sets the root nodes of this model data tree.
	 * 
	 * @param nodeList the list of root nodes
	 */
	void setRootNodes(List<T> nodeList);
	
	/**
	 * Clears this tree model by removing all root nodes.
	 */
	void clear();
	
	/**
	 * Test if this model contains any nodes.
	 * 
	 * @return <code>true</code> if this model is empty
	 */
	boolean isEmpty();

	  ////////////////////////////////
	 /// Properties /////////////////
	////////////////////////////////
    /**
     * Set the specified property on all cells in the tree.
     *
     * @param <U>
     *            The property value type
     *
     * @param property
     *            the property
     * @param propertyValue
     *            value of the property
     */
	<U> void setProperty(Property<U> property, U propertyValue);
	
    /**
     * Set the specified property on all cells in the node.
     *
     * @param <U>
     *            The property value type
     * @param node
     *            tree data's node
     * @param property
     *            the property
     * @param propertyValue
     *            value of the property
     */
	<U> void setNodeProperty(T node, Property<U> property, U propertyValue);
	
    /**
     * Set the specified property on all cells in the column.
     *
     * @param <U>
     *            The property value type
     * @param columnIdentifier
     *            constant denoting the column
     * @param propertyName
     *            name identifying the property
     * @param value
     *            value of the property
     */
	<U> void setColumnProperty(DialogObjectConstant columnIdentifier, Property<U> propertyName, U value);
	
    /**
     * Set the specified property on the specified cell.
     *
     * @param <U>
     *            The property value type
     * @param node
     *            tree data's node
     * @param columnIdentifier
     *            constant denoting the column
     * @param propertyName
     *            name identifying the property
     * @param propertyValue
     *            value of the property
     */
	<U> void setCellProperty(T node, DialogObjectConstant columnIdentifier, Property<U> propertyName, U propertyValue);
	
    /**
     * Get the property value for the specified cell.
     *
     * @param <U>
     *            The property value type
     * @param node
     *            tree data's node
     * @param columnIdentifier
     *            constant denoting the column
     * @param propertyName
     *            name identifying the property
     * @return value of the property
     */
	<U> U getCellProperty(T node, DialogObjectConstant columnIdentifier, Property<U> propertyName);
	
	  ////////////////////////////////
	 /// Selection & expansion //////
	////////////////////////////////	
	/**
	 * Returns the current selection model - none, single or multi selection.
	 * 
	 * @return The current {@link SelectionModel}
	 */
	SelectionModel getSelectionModel();
	
	/**
	 * Sets the current selection model. See {@link SelectionModel}. 
	 * 
	 * @param selectionModel the selection model to set
	 */
	void setSelectionModel(SelectionModel selectionModel);
	
	/**
	 * Sets all nodes to the 'selected' state.
	 * 
	 * @param selected boolean indicating wheather to set selected or unselected state
	 */
	void setSelected(boolean selected);
	
	/**
	 * Sets the 'selected' state on a single node.
	 * 
	 * @param node the node to set the 'selected' state upon
	 * @param selected boolean indicating wheather to set selected or unselected state
	 */
	void setSelected(T node, boolean selected);
	
	/**
	 * Return the 'selected' state for a given node.
	 * 
	 * @param node the node the check state for
	 * @return true if given node is in a selected state, false otherwise
	 */
	boolean isSelected(T node);
	
	/**
	 * Returns a list of all nodes that are selected.
	 * 
	 * @return list of all nodes that are in the 'selected' state
	 */
	List<T> getSelected();
	
	/**
	 * Sets the 'expanded' state on a single node.
	 * 
	 * @param node the node to set the 'expanded' state upon
	 * @param expanded boolean indicating wheather to set expanded or contracted state
	 */
	void setExpanded(T node, boolean expanded);
	
	/**
	 * Sets all nodes to the 'expanded' state.
	 * 
	 * @param expanded boolean indicating wheather to set selected or unselected state
	 */
	void setExpanded(boolean expanded);
	
	/**
	 * Return the 'expanded' state for a given node.
	 * 
	 * @param node the node the check state for
	 * @return true if given node is in an expanded state, false otherwise
	 */
	boolean isExpanded(T node);
	
	/**
	 * Returns a list of all nodes that are expanded.
	 * 
	 * @return list of all nodes that are in the 'expanded' state
	 */
	List<T> getExpanded();

	  ////////////////////////////////
	 /// Enabled & Shown ////////////
    ////////////////////////////////
    /**
     * Set the enabled property for all cells in the entire tree.
     *
     * @param enabled
     *            if <code>true</code>, all cells in the tree will be enabled
     * @see #isCellEnabled(TreeNode, DialogObjectConstant)
     */
	@SuppressWarnings("javadoc")
    void setEnabled(boolean enabled);
	
    /**
     * Set the enabled property for all cells in the specified node.
     *
     * @param node
     *            the node to set editable
     * @param enabled
     *            if <code>true</code>, all cells in the node will be enabled
     * @see #isCellEnabled(TreeNode, DialogObjectConstant)
     */
	@SuppressWarnings("javadoc")
    void setNodeEnabled(T node, boolean enabled);
	
    /**
     * Set the enabled property for all cells in the specified column.
     *
     * @param columnIdentifier
     *            constant denoting the column
     * @param enabled
     *            if <code>true</code>, all cells in the column will be enabled
     * @see #isCellEnabled(TreeNode, DialogObjectConstant)
     * @throws NoSuchElementException
     *             if the specified column is not part of this tree model
     */
	@SuppressWarnings("javadoc")
    void setColumnEnabled(DialogObjectConstant columnIdentifier, boolean enabled);
	
    /**
     * Set the enabled property the specified cell.
     *
     * @param node
     *            the node
     * @param columnIdentifier
     *            constant denoting the column
     * @param enabled
     *            if <code>true</code>, the cells will be editable
     * @see #isCellEnabled(TreeNode, DialogObjectConstant)
     */
	@SuppressWarnings("javadoc")
    void setCellEnabled(T node, DialogObjectConstant columnIdentifier, boolean enabled);
	
    /**
     * Get the enabled property value of the specified cell. If the enabled
     * property of a cell is <code>false</code> the cell will not be able to
     * receive focus. If the cell is <em>visible</em> and <em>enabled</em> the
     * cell will be able to receive focus.
     *
     * @param node
     *            the node
     * @param columnIdentifier
     *            constant denoting the column
     * @return <code>true</code> if the cell is enabled
     */
	boolean isCellEnabled(T node, DialogObjectConstant columnIdentifier);
	

    /**
     * Set the shown property for all cells in the entire tree.
     *
     * @param shown
     *            if <code>true</code>, all cells in the node will be rendered
     * @see #isCellShown(TreeNode, DialogObjectConstant)
     */
	@SuppressWarnings("javadoc")
    void setShown(boolean shown);
	
    /**
     * Set the shown property for all cells in the specified node.
     *
     * @param node
     *            the node to set shown
     * @param shown
     *            if <code>true</code>, all cells in the node will be rendered
     * @see #isCellShown(TreeNode, DialogObjectConstant)
     */
	@SuppressWarnings("javadoc")
    void setNodeShown(T node, boolean shown);
	
    /**
     * Set the shown property for all cells in the specified column.
     *
     * @param columnIdentifier
     *            constant denoting the column
     * @param shown
     *            if <code>true</code>, all cells in the column will be rendered
     * @see #isCellEnabled(TreeNode, DialogObjectConstant)
     */
	@SuppressWarnings("javadoc")
    void setColumnShown(DialogObjectConstant columnIdentifier, boolean shown);
	
    /**
     * Set the shown property the specified cell.
     *
     * @param node
     *            the node
     * @param columnIdentifier
     *            constant denoting the column
     * @param shown
     *            if <code>true</code>, the cell will be rendered
     * @see #isCellEnabled(TreeNode, DialogObjectConstant)
     */
	@SuppressWarnings("javadoc")
    void setCellShown(T node, DialogObjectConstant columnIdentifier, boolean shown);
	
    /**
     * Get the shown property value of the specified cell. If the shown property
     * of a cell is <code>true</code> the cell will be rendered.
     *
     * @param node
     *            the node
     * @param columnIdentifier
     *            constant denoting the column
     * @return <code>true</code> if the cell is shown
     */
	boolean isCellShown(T node, DialogObjectConstant columnIdentifier);

	  ////////////////////////////////
	 /// Sorting ////////////////////
    ////////////////////////////////
    /**
     * Sort all nodes in list of root nodes . The tree is sorted according to the defined
     * TreeNodeComparator comparator.
     * 
     * @param nodeList list of root nodes 
     *
     * @see #addTreeNodeComparator(DialogObjectConstant, TreeNodeComparator)
     */ 
	void sortTree(List<T> nodeList);
	
    /**
     * Set the TreeNodeComparator to use when sorting this tree model.
     *
	 * @param nodeConst the dialog object constant denoting the tree node
	 * @param comparator the comparator to use for a particular type of node
	 */
	void addTreeNodeComparator(DialogObjectConstant nodeConst, TreeNodeComparator comparator);	
}
