/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.spreadsheet;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import no.esito.util.ServiceLoader;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.controller.DialogObjectType;
import no.g9.client.core.view.table.ListRowComparator;
import no.g9.client.core.view.table.ListRowComparator.Sorting;
import no.g9.client.spreadsheet.WorkbookProvider.WORKBOOK_FORMAT;
import no.g9.os.AttributeConstant;
import no.g9.os.OSRole;
import no.g9.os.RoleConstant;
import no.g9.service.G9Spring;
import no.g9.support.ObjectSelection;
import no.g9.support.convert.AttributeConverter;
import no.g9.support.convert.ConvertException;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.SheetUtil;
import org.apache.poi.ss.util.WorkbookUtil;
import org.joda.time.ReadableDateTime;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;

/**
 * The class exports dialogs to Excel.
 * <p>
 * It serves as a super class for exporters which are tailored for specific dialog models.
 */
public abstract class G9ExcelExporter implements SpreadsheetExporter, Comparator<Object>, Serializable {

	/** Holds all the elements that is part of a dialog. */
	protected Map<DialogObjectConstant, ExportElement> elements = new HashMap<DialogObjectConstant, ExportElement>();

	/**
	 * The workbook provider is used when a new workbook is needed.
	 */
	private WorkbookProvider workbookProvider;

	/**
	 * Get the workbook provider.
	 * <p>
	 * If it is not set, a new workbook provider is created.
	 *
	 * @return the current workbook provider
	 */
	public WorkbookProvider getWorkbookProvider() {
		if (workbookProvider == null) {
			workbookProvider = ServiceLoader.getService(WorkbookProvider.class);
		}
		return workbookProvider;
	}

	/**
	 * Set a new workbook provider.
	 *
	 * @param workbookProvider  the new workbook provider
	 */
	public void setWorkbookProvider(WorkbookProvider workbookProvider) {
		this.workbookProvider = workbookProvider;
	}

	/** Holds the resulting workbook. */
	private Workbook workbook;

	/**
	 * Gets the result workbook.
	 *
	 * @return final product
	 */
	@Override
	public Workbook getWorkbook() {
		return this.workbook;
	}

	/**
	 * Used to indicate whether labels should be bold or not.
	 */
	private boolean labelAsBold = false;

	/**
	 * Sets the {@code labelAsBold} variable.
	 *
	 * @param val  a boolean value telling whether labels should be created as bold in the excel document or not
	 */
	protected void setLabelAsBold(boolean val) {
		labelAsBold = val;
	}

	/**
	 * Creates the workbook in the input file.
	 *
	 * @param os  the data to export
	 * @param workbook  the target workbook
	 * @param bookFormat  the format, i.e. xls vs. xlsx
	 * @return A workbook with a name specified in {@code DefaultExcelExporter}, which is a generated class for a specific dialog model
	 */
	public Workbook createBook(ObjectSelection os, File workbook, WORKBOOK_FORMAT bookFormat) {
		Workbook wb = null;
		try {
			FileOutputStream fileOut = new FileOutputStream(workbook.getPath());
			wb = createBook(os, fileOut, bookFormat);
			fileOut.close();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return wb;
	}

	/**
	 * Creates the workbook in the input file.
	 *
	 * @param os  the data to export
	 * @param workbook  the target workbook
	 * @param bookFormat  the format, i.e. xls vs. xlsx
	 * @return A workbook with a name specified in {@code DefaultExcelExporter}, which is a generated class for a specific dialog model
	 */
	public Workbook createBook(ObjectSelection os, OutputStream workbook, WORKBOOK_FORMAT bookFormat) {
		Workbook wb = createBook(os, bookFormat);
		try {
			wb.write(workbook);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return wb;
	}

	/**
	 * The headmaster creating the book and filling.
	 *
	 * @param os  the object selection
	 * @param bookFormat  the book format
	 * @return A workbook with a name specified in the {@code DefaultExcelExporter}, which is a generated class for a specific dialog model
	 */
	private Workbook createBook(ObjectSelection os, WORKBOOK_FORMAT bookFormat) {
		this.workbook = getWorkbookProvider().createWorkbook(bookFormat);
		setAllMapElements();
		final Sheet defaultSheet = this.workbook.createSheet(WorkbookUtil.createSafeSheetName(getSheetName()));
		exportDialog(os, defaultSheet);
		return this.workbook;
	}

	/**
	 * Initialization of the export.
	 *
	 * @param os  the data
	 * @param defaultSheet  the default sheet
	 */
	protected abstract void exportDialog(ObjectSelection os, Sheet defaultSheet);

	/**
	 * Sets all the {@code DialogObjectConstant} elements in the map.
	 */
	protected abstract void setAllMapElements();

	/**
	 * Gets the sheet name.
	 *
	 * @return the name of the {@code Sheet}
	 */
	protected abstract String getSheetName();

	/**
	 * Gets the generator parameter that tells whether the user want to show hidden elements or not.
	 *
	 * @return true if one wants to show hidden elements, and false else
	 */
	protected abstract boolean getParameterHidden();

	/**
	 * Retrieves the dialog name.
	 *
	 * @return the dialog name
	 */
	protected abstract String getDialogName();

	/**
	 * Creates a cell for the given element.
	 *
	 * @param dialogElement  the element to create
	 * @param sheet  the sheet to create the element in
	 * @param offset  the global offset for the cell
	 * @param isLabel  indicates if the cell should hold a label
	 * @param alignment  the specified alignment
	 * @return the cell with correct alignment
	 */
	protected Cell createCell(DialogObjectConstant dialogElement, Sheet sheet, G9ExcelCell offset, boolean isLabel, short alignment) {
		return createCell(dialogElement, sheet, offset, isLabel, -1, null, alignment);
	}

	/**
	 * Creates the cell for the element. Uses custom column index.
	 *
	 * @param dialogElement  the element to create
	 * @param sheet  the sheet to create the element in
	 * @param offset  the global offset for the cell
	 * @param isLabel  indicates if the cell should hold a label
	 * @param colIndex  used in some special cases to indicate start column
	 * @param alignment  the specified alignment
	 * @return the cell with correct alignment
	 */
	protected Cell createCell(DialogObjectConstant dialogElement, Sheet sheet, G9ExcelCell offset, boolean isLabel, int colIndex, short alignment) {
		return createCell(dialogElement, sheet, offset, isLabel, colIndex, null, alignment);
	}

	/**
	 * Creates the cell for the element. Uses custom column index and column title.
	 *
	 * @param dialogElement  the element to create
	 * @param sheet  the sheet to create the element in
	 * @param offset  the global offset for the cell
	 * @param isLabel  indicates if the cell should hold a label
	 * @param colIndex  used in some special cases to indicate start column. The value -1 signals it is not to be used
	 * @param columnTitle  used to name a cell in case the DialogObjectConstant is not able to provide a string
	 * @param alignment  the specified alignment
	 * @return The cell with correct alignment
	 */
	protected Cell createCell(DialogObjectConstant dialogElement, Sheet sheet, G9ExcelCell offset, boolean isLabel, int colIndex, String columnTitle, 
							  short alignment) {

		ExportElement ee = elements.get(dialogElement);
		Row row = getRow(sheet, ee.getRow(), offset.getRow());

		int column = colIndex == -1 ? ee.getColumn() : colIndex;
		Cell cell = row.createCell(column + offset.getColumn());
		cell.setCellStyle(createCellStyle(sheet, alignment, isLabel && labelAsBold));

		if (isLabel) {
			setLabelCellData(cell, dialogElement, columnTitle);
		}
		else if (addLabelForCheckButton(dialogElement)) {
			// Add a label for a CheckButton if it was missing in the dialog model
			int labelColumn = column + offset.getColumn() - 1;
			if (labelColumn >= 0 && row.getCell(labelColumn) == null) {
				Cell labelCell = row.createCell(labelColumn);
				labelCell.setCellStyle(createCellStyle(sheet, alignment, labelAsBold));
				setLabelCellData(labelCell, dialogElement, columnTitle);
			}
		}
		return cell;
	}

	/**
	 * Adds cell data to labels.
	 * 
	 * @param cell  the cell one wants to add data to
	 * @param dialogObject  the DialogObjectContant related to the label
	 * @param columnTitle  used to name a cell in case the {@code DialogObjectConstant} is unable to provide it
	 */
	private void setLabelCellData(Cell cell, DialogObjectConstant dialogObject, String columnTitle) {
		String cellData;

		/* Checks whether the element should be shown and exported. */
		ExportElement ee = elements.get(dialogObject);
		if ((!ee.isShown() && !getParameterHidden()) || !ee.isExported()) {
			cellData = "";
		} else {
			try {
				cellData = getBundle().getString(dialogObject.getMessageID());
			} catch (MissingResourceException mre) {
				cellData = getBundle().getString(columnTitle);
			}
		}
		if (cellData != null) {
			cell.setCellValue(cellData);
		}
	}

	/**
	 * Creates a cell style given alignment and a boolean telling whether the text shall be bold.
	 * 
	 * @param sheet  the current sheet
	 * @param alignment  a short indicating a {@code CellStyle} alignment
	 * @param asBold  a {@code boolean} telling whether the text shall be bold
	 * @return the new {@code CellStyle}
	 */
	private static CellStyle createCellStyle(Sheet sheet, short alignment, boolean asBold) {
		CellStyle cellStyle = sheet.getWorkbook().createCellStyle();
		Font f = sheet.getWorkbook().getFontAt(cellStyle.getFontIndex());
		if (asBold) {
			cellStyle.setFont(getBoldFont(sheet, f));
		} else {
			cellStyle.setFont(f);
		}
		cellStyle.setAlignment(alignment);
		cellStyle.setVerticalAlignment(CellStyle.VERTICAL_TOP);
		return cellStyle;
	}

	/**
	 * Returns true if a given dialog object constant's parent is a table.
	 *  
	 * @param dialogObject  some {@code DialogObjectConstant}, not null
	 * @return true if the parent of the provided {@code DialogObjectConstant} is a table
	 */
	private static boolean parentIsTable(DialogObjectConstant dialogObject) {
		if (dialogObject.getParent() instanceof DialogObjectConstant) {
			DialogObjectConstant parent = (DialogObjectConstant) dialogObject.getParent();
			if (parent.getType() == DialogObjectType.TableBlock) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Used to get a bold font otherwise equal to the default font used in spreadsheet cells.
	 *
	 * @param sheet  the current spreadsheet
	 * @param f  the default font used in excel cells
	 * @return a bold version of f
	 */
	private static Font getBoldFont(Sheet sheet, Font font) {
		short fontColor = font.getColor();
		short fontHeight = font.getFontHeight();
		String fontName = font.getFontName();
		boolean italic = font.getItalic();
		boolean strikeout = font.getStrikeout();
		short typeOffset = font.getTypeOffset();
		byte underline = font.getUnderline();
		Font bf = sheet.getWorkbook().findFont(Font.BOLDWEIGHT_BOLD, fontColor, fontHeight, fontName, italic, strikeout, typeOffset, underline);
		if (bf == null) {
			bf = sheet.getWorkbook().createFont();
			bf.setBoldweight(Font.BOLDWEIGHT_BOLD);
			bf.setColor(fontColor);
			bf.setFontHeight(fontHeight);
			bf.setFontName(fontName);
			bf.setItalic(italic);
			bf.setStrikeout(strikeout);
			bf.setTypeOffset(typeOffset);
			bf.setUnderline(underline);
		}
		return bf;
	}
	
	/**
	 * Creates cells for a table.
	 *
	 * @param os  the data
	 * @param targetRole  the primary role for the table
	 * @param columnDataList  a list containing data about each column
	 * @param sheet  the {@code Sheet} in which the table should be added to
	 * @param initialCell  the offset before creating the table
	 * @param showHeader  tells whether the columns headers should be shown
	 * @param doc  the {@code DialogObjectConstant}
	 * @return The offset when the table is created
	 */
	protected G9ExcelCell createTable(ObjectSelection os, RoleConstant targetRole, List<TableColumnData> columnDataList,
						  			  Sheet sheet, G9ExcelCell initialCell, boolean showHeader, DialogObjectConstant doc) {
		int colIndex = 0;
		int rowIndex = initialCell.getRow();
		if (!showHeader) {
			rowIndex --;
		}
		setGlobalCompareValues(columnDataList);
		List<Object> targetObjects = getSortedTargetObjects(os, targetRole);

		if (targetObjects != null && !targetObjects.isEmpty()) {
			boolean showingHeader = showHeader;
			for (Object targetObject : targetObjects) {
				rowIndex ++;
				G9ExcelCell offset = addRowData(columnDataList, sheet, initialCell, showingHeader, rowIndex, targetObject);
				colIndex = offset.getColumn();
				rowIndex = offset.getRow();
				showingHeader = false;
			}
		} else if (showHeader) {
			for (TableColumnData columnData : columnDataList) { 
				addColumnHeader(sheet, columnData.getDiaConst(), initialCell, colIndex);
				colIndex ++;
			}
			rowIndex ++;
		}
		return new G9ExcelCell(colIndex, rowIndex);
	}
	
	/**
	 * Adds cell data to all columns in a row. 
	 * <p>
	 * A label is also added if {@code labelAdded} is false.
	 * 
	 * @param columnDataList  a {@code List} containing information about each column
	 * @param sheet  the current sheet
	 * @param initialCell  the initial cell
	 * @param showHeader  is true if this method shall add headers to the spreadsheet
	 * @param rowIndex  the current row index
	 * @param targetObject  data used to populate the current row
	 * @return The max cell used after adding all the elements on this row to the spreadsheet
	 */
	private G9ExcelCell addRowData(List<TableColumnData> columnDataList, Sheet sheet, G9ExcelCell initialCell, 
								   boolean showHeader, int rowIndex, Object targetObject) {
		int colIndex = 0;
		int newRowIndex = rowIndex;
		for (TableColumnData columnData : columnDataList) {
			final DialogObjectConstant column = columnData.getDiaConst();
			
			if (showHeader) {
				addColumnHeader(sheet, column, initialCell, colIndex);
			}
			
			final ExportElement columnEe = elements.get(column);
			columnEe.setColumn(colIndex);
			if (targetObject != null && columnEe.isExported()) {
				final String methodName = columnData.getBcMethod();
				if (methodName != null && !methodName.equals("")) {
					G9ExcelCell offset = callPlacementMethod(methodName, targetObject, sheet, 
							new G9ExcelCell(initialCell.getColumn() + colIndex, rowIndex));
					colIndex = offset.getColumn();
					newRowIndex = offset.getRow();
				} else {
					final Cell cell = createCell(column, sheet, 
							new G9ExcelCell(initialCell.getColumn(), rowIndex), false, columnData.getCellAlignment());
					setCellData(column, getColumnObject(targetObject, column.getAttribute().getAttributeRole()), cell, columnData.getCellFormat());
				}
			}
			colIndex ++;
		}
		if (colIndex > 0) {
			colIndex--;
		}
		return new G9ExcelCell(colIndex, newRowIndex);
	}
	
	/**
	 * Adds column headers to the sheet.
	 * 
	 * @param sheet  the current sheet
	 * @param column  the DialogObjectConstant belonging to current column
	 * @param offset  the initial {@code G9ExcelCell}
	 * @param col  the current column index
	 */
	private void addColumnHeader(final Sheet sheet, final DialogObjectConstant column, final G9ExcelCell offset, final int col) {
		DialogObjectConstant label = column.getLabelComponent();
		if (label == null) {
			label = column;
		}
		String columnTitleID = column.getMessageID();
		columnTitleID = columnTitleID.substring(0, columnTitleID.lastIndexOf("title")) + "column_title";
		ExportElement labelEe = elements.get(label);

		if ((labelEe.isShown() || getParameterHidden()) && labelEe.isExported()) {
			createCell(label, sheet, offset, true, col, columnTitleID, CellStyle.ALIGN_LEFT);
		}
	}
	
	/**
	 * Takes a name of a placement method, contained within a class extending this one, and tries to call it.
	 * <p>
	 * If the provided method is run successfully, the return value of that method is returned.
	 * 
	 * @param methodName  a {@code String} being the name of a public placement method in a class extending this one
	 * @param targetObject  the data that are going to be shown in the spreadsheet
	 * @param sheet  the spreadsheet the provided data is going to be added to
	 * @param initialCell  the initial cell in the spreadsheet
	 * @return the last used cell in the spreadsheet.
	 */
	private G9ExcelCell callPlacementMethod(final String methodName, Object targetObject, Sheet sheet, G9ExcelCell initialCell) {
		try {
			Method method = this.getClass().getMethod(methodName, Object.class, Sheet.class, G9ExcelCell.class);
			try {
				return (G9ExcelCell) method.invoke(this, targetObject, sheet, initialCell);
			} catch(IllegalArgumentException | IllegalAccessException | InvocationTargetException e) {
				System.err.println("Error when adding block to table: \n"
						+ "Could not access the method \"" + methodName + "\" whit parameters Object, Sheet and G9ExcelCell.");
			} catch(ClassCastException e) {
				System.err.println("Error when adding block to table: \n"
						+ "Could not cast return value from " + methodName + " to G9ExcelExporter.");
			}
		} catch(NoSuchMethodException | SecurityException e) {
			System.err.println("Error when adding block to table: \n"
						+ "Could not find a method \"" + methodName + "\" whit parameters Object, Sheet and G9ExcelCell.");
		}
		return initialCell;
	}

	/**
	 * Gets the column object.
	 *
	 * @param targetObject  contains data about role attributes
	 * @param attributeRole  some role constant belonging to the current column
	 * @return the column object
	 */
	protected Object getColumnObject(Object targetObject, RoleConstant attributeRole) {
		OSRole<?> role = getRoleMap().get(attributeRole);
		if (role == null) {
			return null;
		}
		if (targetObject.getClass().equals(role.createNewInstance().getClass())) {
			return targetObject;
		}
		OSRole<?> parent = role.getParent();
		if (targetObject.getClass().equals(parent.createNewInstance().getClass())) {
			return parent.getRelation(targetObject, attributeRole);
		}
		List<OSRole<?>> children = role.getChildren();
		Iterator<OSRole<?>> childIterator = children.iterator();
		while (childIterator.hasNext()) {
			OSRole<?> childRole = childIterator.next();
			Object o = getColumnObject(role.getRelation(targetObject, childRole.getRoleConstant()), childRole.getRoleConstant());
			if (o != null && o.getClass().equals(childRole.createNewInstance().getClass())) {
				return o;
			}
		}
		return null;
	}
	
	/**
	 * Retrieves a sorted list of the data. 
	 * <p>
	 * Typically to be used in lists (list blocks and table blocks).
	 *
	 * @param os  the data
	 * @param targetRole  the target role
	 * @return a {@code List} of target objects. An empty list if no target objects
	 */
	protected List<Object> getSortedTargetObjects(ObjectSelection os, RoleConstant targetRole) {
		Object rootObj = getRootObject(os, targetRole);
		return sortTargetObjects(getDomainObjectFromParent(rootObj, targetRole));
	}

	/**
	 * Sorts a list according to global settings.
	 *
	 * @param targetObjects  a set/list containing values one wants written to the spreadsheet
	 * @return A {@code List} of target objects. An empty list if no target objects
	 */
	private List<Object> sortTargetObjects(Object targetObjects) {
		List<Object> list = new LinkedList<Object>();
		if (targetObjects instanceof Collection<?>) {
			for (Object targetObject : (Collection<?>) targetObjects) {
				list.add(targetObject);
			}
			Collections.sort(list, this);
			return list;
		}
		return null;
	}

	/**
	 * Adds data to a specific cell.
	 *
	 * @param dialogElement  the dialog element
	 * @param domainObject  the domain object
	 * @param cell  the cell to put the data in
	 * @param dateFormat  the format for dates
	 */
	protected void setCellData(DialogObjectConstant dialogElement, Object domainObject, Cell cell, String dateFormat) {

		AttributeConstant attribute = dialogElement.getAttribute();
		OSRole<?> role = getRoleMap().get(attribute.getAttributeRole());

		if (hookFilterElement(dialogElement, domainObject)) {
			return;
		}
		if (role == null || domainObject == null) {
			cell.setCellValue("");
		} else {
			if (domainObject instanceof Collection) {
				if (!((Collection<?>) domainObject).isEmpty()) {
					domainObject = ((Collection<?>) domainObject).iterator().next();
				} else {
					cell.setCellValue("");
					return;
				}
			}
			Object data = role.getValue(domainObject, attribute);
			String cellData = formatAttributeData(attribute, data, dateFormat);
			try {
				if (!cellData.equals("")) {
					// If the cellData is an integer, the cell will be converted as such.
					cell.setCellValue(Integer.parseInt(cellData));
				} else {
					cell.setCellValue(cellData);
				}
			} catch (NumberFormatException nfe) {
				cell.setCellValue(cellData);
			}
		}
	}

	/**
	 * Gets the target object from the object selection. 
	 * <p>
	 * The operation will only navigate over single relations from parent to child.
	 *
	 * @param os  the data
	 * @param targetRole  the target role
	 * @return the domain object for the role, null if none
	 */
	protected Object getTargetObject(ObjectSelection os, RoleConstant targetRole) {
		final Object root = getRootObject(os, targetRole);
		OSRole<?> role = getRoleMap().get(targetRole);
		List<OSRole<?>> roleList = getRoleHierarchy(role);
		Object result = root;
		for (int i = roleList.size() - 1; i > 0; i--) {
			OSRole<?> childRole = roleList.get(i - 1);
			if (roleList.get(i).isMany() ||  result instanceof Collection<?>) {
				result = getChildObject(((Collection<?>) result).iterator().next(), roleList.get(i), childRole.getRoleConstant());
			} else {
				result = getChildObject(result, roleList.get(i), childRole.getRoleConstant());
			}
		}
		return result;
	}
	
	/**
	 * Make sure that the use of getRootObject is not used, when the Root Object is not wanted.
	 *
	 * @param os  the {@code ObjectSelection}
	 * @param targetRole  the target role
	 * @param compare  the compare string
	 * @return the root object(s) for the target role or the domain object for the role. Null if none
	 */
	public Object getRootOrTargetObject(ObjectSelection os, RoleConstant targetRole, String compare){
		Object root = getRootObject(os, targetRole);
		if(root == null) {
			return null;
		} else if(root.toString().equals(compare)) {
			return root;
		} else {
			return getTargetObject(os, targetRole);
		}
	}

	/**
	 * Gets the child object(s) from parent.
	 *
	 * @param parent  the parent object
	 * @param parentRole  the parent role
	 * @param childRole  the role constant to get the object for
	 * @return the child object(s)
	 */
	protected static Object getChildObject(Object parent, OSRole<?> parentRole, RoleConstant childRole) {
		return parentRole.getRelation(parent, childRole);
	}

	/**
	 * Gets the role hierarchy from target to root.
	 *
	 * @param targetRole  the target role
	 * @return a {@code List} of roles from target (first) to root (last). Empty list if targetRole is null
	 */
	protected static List<OSRole<?>> getRoleHierarchy(OSRole<?> targetRole) {
		List<OSRole<?>> roleList = new LinkedList<OSRole<?>>();
		if (targetRole == null) {
			return roleList;
		}
		OSRole<?> parent = targetRole;

		roleList.add(parent);
		while (!parent.isRoot()) {
			parent = parent.getParent();
			roleList.add(parent);
		}
		return roleList;
	}

	/**
	 * Apply the converter to the given field value, converting from model representation to view representation.
	 *
	 * @param <M> model
	 * @param <T> target
	 *
	 * @param attribute  the attribute to convert
	 * @param fieldValue  the value to convert
	 * @return the view representation of the value
	 * @throws ConvertException if conversion fails
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	static <M, T> T convertToViewInternal(AttributeConstant attribute, M fieldValue) throws ConvertException {
		AttributeConverter<M, T> converter = G9Spring.<AttributeConverter> getBean(AttributeConverter.class, attribute.getConverterId());
		return converter.fromModel(fieldValue, null);
	}

	/**
	 * Formats the attribute in question according to the display rule and/or Locale settings.
	 *
	 * @param attribute  the attribute in question
	 * @param attributeValue  the data
	 * @param dateFormat  a {@code String} representation of the date format
	 * @return the formatted string
	 */
	private static String formatAttributeData(AttributeConstant attribute, Object attributeValue, String dateFormat) {
		if (attributeValue == null) {
			return "";
		}
		if (attribute != null) {
			String converterId = attribute.getConverterId();
			if (converterId != null) {
				try {
					attributeValue = convertToViewInternal(attribute, attributeValue);
					if (attributeValue == null || attributeValue == "") {
					    return "";
					}
				} catch (ConvertException e) {
					/*
					 * In this case, the conversion is ignored, and and the
					 * attribute is returned as a string.
					 */
				}
			}
		}
		if (dateFormat != null && !dateFormat.equals("")) {
			if (attributeValue instanceof ReadableDateTime) {
				return ((ReadableDateTime)attributeValue).toString(dateFormat);
			}
			if (attributeValue instanceof LocalDate) {
				return ((LocalDate)attributeValue).format(DateTimeFormatter.ofPattern(dateFormat));
			}
			if (attributeValue instanceof LocalTime) {
				return ((LocalTime)attributeValue).format(DateTimeFormatter.ofPattern(dateFormat));
			}
			if (attributeValue instanceof LocalDateTime) {
				return ((LocalDateTime)attributeValue).format(DateTimeFormatter.ofPattern(dateFormat));
			}
			return new SimpleDateFormat(dateFormat).format(attributeValue);
		}
		if (attributeValue instanceof Boolean) {
			if (((Boolean) attributeValue).booleanValue()) {
				return getGeneralBundle().getString("boolean.true.title");
			}
			return getGeneralBundle().getString("boolean.false.title");
		}

		return attributeValue.toString();
	}

	/**
	 * Lets user code filter the data for this dialog element.
	 *
	 * @param dialogElement  the dialog element
	 * @param domainObject  the domain object
	 * @return true to filter the dialog element, otherwise false
	 */
	protected static boolean hookFilterElement(DialogObjectConstant dialogElement, Object domainObject) {

		// Project specific code to be injected.
		return false;
	}

	/**
	 * Retrieves the row with the given data. 
	 * <p>
	 * Creates a new if it doesn't exist already.
	 *
	 * @param sheet  the addressed sheet
	 * @param localRow  the row number
	 * @param offset  the offset
	 * @return  the row in interest
	 */
	protected static Row getRow(Sheet sheet, int localRow, int offset) {
		int rowNo = localRow + offset;
		if (sheet == null) {
			return null;
		}
		if (sheet.getRow(rowNo) != null) {
			return sheet.getRow(rowNo);
		}
		return sheet.createRow(rowNo);
	}

	/**
	 * Retrieves the maximum row or column.
	 *
	 * @param dialogElements  the {@code DialogObjectConstants} to consider
	 * @param isRow  indicates whether the maximum value should be found for a row or column
	 * @return the highest row/column occupied by a {@code DialogObjectConstant} in {@code dialogElements}
	 */
	private int getMax(DialogObjectConstant[] dialogElements, boolean isRow) {
		if (dialogElements == null) {
			return 0;
		}
		int maxValue = 0;
		for (DialogObjectConstant element : dialogElements) {
			ExportElement ee = elements.get(element);
			if (ee == null) {
				continue;
			}
			if (isRow && ee.getRow() > maxValue) {
				maxValue = ee.getRow();
			}
			if (!isRow && ee.getColumn() > maxValue) {
				maxValue = ee.getColumn();
			}
		}
		return maxValue;
	}

	/**
	 * Creates a {@code G9ExcelCell} with max column and row.
	 *
	 * @param cells  some array containing {@code G9ExcelCells}
	 * @return the cell with max column and row, found among the provided {@code G9ExcelCells}
	 */
	protected static G9ExcelCell getMax(G9ExcelCell[] cells) {
		if (cells == null) {
			return new G9ExcelCell(0, 0);
		}
		int maxColumn = 0;
		int maxRow = 0;
		for (G9ExcelCell cell : cells) {
			if (cell == null) {
				continue;
			}
			if (cell.getColumn() > maxColumn) {
				maxColumn = cell.getColumn();
			}
			if (cell.getRow() > maxRow) {
				maxRow = cell.getRow();
			}
		}
		return new G9ExcelCell(maxColumn, maxRow);

	}
	
	/**
	 * Creates a {@code G9ExcelCell} with max column and row.
	 *
	 * @param offset  the initial cell
	 * @param cell  the new cell
	 * @return the cell with max column and row, based on the provided {@code G9ExcelCells}
	 */
	protected static G9ExcelCell getMax(G9ExcelCell offset, G9ExcelCell cell) {
		G9ExcelCell[] cells = {offset, cell};
		return getMax(cells);
	}

	/**
	 * Gets the maximum cell obtained when combining the provided offset with one of the dialog elements
	 *
	 * @param dialogElements  the dialog elements to consider
	 * @param offset  the maximum cell before dialog elements are added
	 * @return the maximum cell obtained when combining the provided offset with one of the dialog elements
	 */
	protected G9ExcelCell getMaxCell(DialogObjectConstant[] dialogElements, G9ExcelCell offset) {
		G9ExcelCell localOffset;
		if (offset == null) {
			localOffset = new G9ExcelCell(0, 0);
		} else {
			localOffset = offset;
		}
		if (dialogElements == null) {
			return localOffset;
		}
		return new G9ExcelCell(localOffset.getColumn()
				+ getMax(dialogElements, false), localOffset.getRow()
				+ getMax(dialogElements, true));
	}

	/**
	 * Gets the root object(s) for the target role from an object selection
	 *
	 * @param os  the object selection (data)
	 * @param targetRole  the target role
	 * @return the root object(s) of the target role
	 */
	protected Object getRootObject(ObjectSelection os, RoleConstant targetRole) {
		OSRole<?> role = getRoleMap().get(targetRole);
		OSRole<?> root;
		if (role.isRoot()) {
			root = role;
		} else {
			root = role.getRoot();
		}
		Collection<?> targetObjects = os.getRootObjects(root.getRoleConstant().toString());
		if (targetObjects == null || targetObjects.size() < 1) {
			return new LinkedList<Object>();
		}
		if (targetObjects.size() > 1) {
			return targetObjects;
		}
		Iterator<?> it = targetObjects.iterator();
		return it.next();
	}

	/**
	 * Gets the domain object(s) from the parent domain object
	 *
	 * @param parentObject  the parent domain object
	 * @param targetRole  the target role
	 * @return domain object(s) of the target (if the parent object matches the parent role)
	 */
	protected Object getDomainObjectFromParent(Object parentObject, RoleConstant targetRole) {
		if (parentObject == null) return null;
		OSRole<?> role = getRoleMap().get(targetRole);
		if (role.isRoot()) {
			return parentObject;
		}
		List<OSRole<?>> roleList = new LinkedList<OSRole<?>>();
		OSRole<?> currentRole = role;
		while (currentRole.getParent() != null) {
			roleList.add(currentRole);
			currentRole = currentRole.getParent();
		}
		
		Object result = parentObject;
		int i = roleList.size()-1;
		while (currentRole != role && !(result instanceof Collection)) {
			result = currentRole.getRelation(result, roleList.get(i).getRoleConstant());
			currentRole = roleList.get(i);
			i--;
		}
		return result;
	}
	
	/**
	 * Gives a map of role constants and object selection roles.
	 *
	 * @return a {@code Map} from role constant to object selection role
	 */
	protected abstract Map<RoleConstant, OSRole<?>> getRoleMap();

	/**
	 * Gets the resource bundle.
	 *
	 * @return the resource bundle
	 */
	protected abstract ResourceBundle getBundle();

	/**
	 * Gets the application resource bundle.
	 *
	 * @return the application resource bundle
	 */
	protected static ResourceBundle getGeneralBundle() {
		return G9Spring.getBean("g9RuntimeResourceBundle");
	}

	/**
	 * Gets the next available cell.
	 * <p>
	 * Almost like {@code getMaxCell}, but the returned cell does not necessary contain the highest column index possible. 
	 * Instead, the maximum column index on the row having the highest row index, is used.
	 *
	 * @param dialogElements  the dialog elements to consider
	 * @param offset  the maximum cell before dialog elements are added
	 * @return a new cell containing the highest row index and the column index which is highest on that row
	 */
	protected G9ExcelCell getNextAvailableCell(DialogObjectConstant[] dialogElements, G9ExcelCell offset) {
		G9ExcelCell localOffset;
		if (offset == null) {
			localOffset = new G9ExcelCell(0, 0);
		} else {
			localOffset = offset;
		}
		if (dialogElements == null) {
			return localOffset;
		}
		int maxRow = 0;
		int maxCol = 0;
		for (DialogObjectConstant element : dialogElements) {
			ExportElement ee = elements.get(element);
			if (ee == null) {
				continue;
			} else if (ee.getRow() > maxRow) {
				maxRow = ee.getRow();
				maxCol = ee.getColumn();
			} else if (ee.getRow() == maxRow && ee.getColumn() > maxCol) {
				maxCol = ee.getColumn();
			}
		}
		return new G9ExcelCell(localOffset.getColumn() + maxCol,
				localOffset.getRow() + maxRow);
	}

	/*
	 * The following are used for comparison. 
	 * They are set when creating table.
	 */
	private DialogObjectConstant[] listColumns;
	private int[] sortPriority;
	private ListRowComparator.Sorting[] sorting;
	
	/**
	 * Sets global values, used when comparing table rows.
	 * <p>
	 * Takes a list of TableColumnData objects, extract information from these about sorting and DialogObjectConstants, 
	 * and sets the global variables listColumns, sortPriority and sorting. 
	 * This method is used in DefaultExcelExporter-class, right before lists and tables are added to the spreadsheet.
	 *
	 * @param columnDataList  list containing stored information for each column in a list
	 */
	private void setGlobalCompareValues(List<TableColumnData> columnDataList) {
		listColumns = new DialogObjectConstant[columnDataList.size()];
		sortPriority = new int[columnDataList.size()];
		sorting = new ListRowComparator.Sorting[columnDataList.size()];
		
		int nonSortIndex = columnDataList.size() - 1;
		for (int i = 0; i < columnDataList.size(); i++) {
			TableColumnData columnData = columnDataList.get(i);
			
			listColumns[i] = columnData.getDiaConst();
			sorting[i] = columnData.getSorting();
			
			int sortIndex = columnData.getSortIndex();
			if (sortIndex == -1) {
				sortPriority[i] = nonSortIndex;
				nonSortIndex--;
			} else {
				int index = 0;
				for (TableColumnData column : columnDataList) {
					if (column.getSortIndex() != -1 && column.getSortIndex() < sortIndex) {
						index ++;
					}
				}
				sortPriority[i] = index;
			}
		}
	}

	/**
	 * Used to compare two list objects. 
	 * <p>
	 * Uses the OS Role Map to get the target objects value. 
	 * The target objects values are then compared using their {@code compareTo}-method.
	 *
	 * @param o1  a target object
	 * @param o2  also a target object
	 * @return 1 if o1 &gt; o2, 0 if o1 == o2 and -1 if o1 &lt; o2
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	@Override
	public int compare(Object o1, Object o2) {
		DialogObjectConstant column = null;
		ListRowComparator.Sorting sortType = null;
		int result = 0;
		for (int sortPriIndex = 0; sortPriIndex < sortPriority.length; sortPriIndex++) {
			for (int i = 0; i < sortPriority.length; i++) {
				if (i == sortPriority[sortPriIndex]) {
					column = listColumns[i];
					sortType = sorting[i];
					break;
				}
			}
			if (column == null) {
				System.err.println("An element in the priority queue is not is either less than 0 or greater than "
						+ (sortPriority.length - 1));
			} else {
				if (!(Sorting.NO_SORT.equals(sortType))) {
					AttributeConstant attribute = column.getAttribute();
					OSRole<?> role = getRoleMap().get(attribute.getAttributeRole());

					Comparable v1 = (Comparable) role.getValue(getColumnObject(o1, column.getAttribute().getAttributeRole()), attribute);
					Comparable v2 = (Comparable) role.getValue(getColumnObject(o2, column.getAttribute().getAttributeRole()), attribute);

					if (v1 == null) {
						result = (v2 == null) ? 0 : -1;
					} else if (v2 == null) {
						result = 1;
					} else if (Sorting.ASCENDING.equals(sortType)) {
						result = v1.compareTo(v2);
					} else {
						result = v2.compareTo(v1);
					}
				}
			}
			if (result != 0) {
				break;
			}
		}
		return result;
	}

    /**
     * Get the (maximum) number of columns in use for a sheet.
     *
     * @param sheet  the sheet with data
     * @return the  number of columns in use
     */
    protected static short getColumns(Sheet sheet) {
        short maxCol= 0;
        for (short r= 0; r <= sheet.getLastRowNum(); r++) {
            Row row= sheet.getRow(r);
            if (row != null && maxCol < row.getLastCellNum()) {
                maxCol= row.getLastCellNum();
            }
        }
        return maxCol;
    }

    /**
     * Resize all columns according to the data, in all sheets from the workbook that contains the given sheet.
     *
     * @param sheet  a {@code Sheet} contained within the workbook where all cells are going to be resized
     */
    protected static void autoResizeAll(final Sheet sheet) {
		for (short columnIndex = 0; columnIndex < getColumns(sheet); columnIndex++) {
			double width = SheetUtil.getColumnWidth(sheet, columnIndex, false);
		    if (width != -1) {
		        width *= 256;
		        int maxColumnWidth = 255 * 256; // The maximum column width for an individual cell is 255 characters
		        if (width > maxColumnWidth) {
		            width = maxColumnWidth;
		        }
		        sheet.setColumnWidth(columnIndex, (int) (width));
		    }        	
		}
	}
    
    /**
	 * Returns true if label was missing in the dialog model for a {@code CheckButton}. 
	 * 
	 * @param dialogElement  the element to create
	 * @return true if label was missing in the dialog model for a {@code CheckButton}
	 */
	private static boolean addLabelForCheckButton(DialogObjectConstant dialogElement) {
		return dialogElement.getType() == DialogObjectType.CheckButton 
			&& dialogElement.getLabelComponent() == null 
			&& !parentIsTable(dialogElement);
	}
}
