/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.spreadsheet;

import java.io.File;
import java.io.FileNotFoundException;

import no.g9.client.core.controller.DialogConstant;
import no.g9.client.spreadsheet.WorkbookProvider.WORKBOOK_FORMAT;
import no.g9.support.ClientContext;
import no.g9.support.ObjectSelection;

/**
 * This class can be used when one wants to generate workbooks without going through an IceFaces application. 
 */
public class StandaloneSpreadsheetExporter {
	
	/**
	 * Generates a spreadsheet based on provided object selection and dialog.
	 * 
	 * @param os  the object selection containing the data one wants to be exported to the spreadsheet
	 * @param targetDialog  an instance of {@code DialogConstant} indicating which dialog the export should be based on
	 * @param outputDir  the directory one wants the workbook to be generated to
	 * @param format  the wanted spreadsheet format. Supported formats are {@code xls} and {@code xlsx}
	 */
	public static void generateSpreadsheet(ObjectSelection os, DialogConstant targetDialog, WORKBOOK_FORMAT format, String outputDir) {
		try {
			File outputFile = createNewWorkbook(targetDialog, format, outputDir);
		
			// To change from xslx (default) to xls the wanted format is added to the client context
			ClientContext ctx = new ClientContext();
			ctx.setParam(DefaultSpreadsheetService.FORMAT_PARAM, format);
			
			// The spreadsheet export
			DefaultSpreadsheetService spreadsheetService = new DefaultSpreadsheetService();
			spreadsheetService.exportToSpreadsheet(os, targetDialog, outputFile, ctx);
			
			System.out.println("Generated the following file: \t" + outputFile.getAbsolutePath());
		} catch (FileNotFoundException e) {
			System.err.println("Could not create new FileOutputStream - " + e.getMessage());
		}
	}
	
	/**
	 * Creates a new workbook. 
	 * <p>
	 * If the provided directory already contains a workbook with the same name as the file one are currently trying to create, a number is added at
	 * the end of the workbook name.
	 * 
	 * @param targetDialog  an instance of {@code DialogConstant} indicating which dialog the export should be based on
	 * @param format  the wanted spreadsheet format. Supported formats are {@code xls} and {@code xlsx}
	 * @param outputDir  the directory one wants the workbook to be generated to
	 * @return the {@code FileOutoutStream} that will be used in the export
	 * @throws FileNotFoundException  if the file exists but is a directory rather than a regular file, does not exist but cannot be created, or 
	 * 									cannot be opened for any other reason
	 */
	private static File createNewWorkbook(DialogConstant targetDialog, WORKBOOK_FORMAT format, String outputDir) 
														throws FileNotFoundException {
		File outputDirectory = new File(outputDir);
		String formatSuffix = format.getSuffix();
		outputDirectory.mkdir();
		String fileName = targetDialog.getG9Name();
		File tempFile = new File(outputDir + "/" + fileName + formatSuffix);
		
		int counter = 1;
		while(tempFile.exists() && !tempFile.isDirectory()) {
			counter ++;
			tempFile = new File(outputDir + "/" + fileName  + " (" + counter + ")" + formatSuffix);
		}
		if (counter > 1) {
			fileName += " (" + counter + ")";
		}
		File outputStream = new File(outputDir + "/" + fileName + formatSuffix);
		return outputStream;
	}
}