/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.spreadsheet;

import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.table.ListRowComparator;

/**
 * Holds information about a column in a table.
 * <p>
 * The information contained within an instance of this class is used when exporting a table to spreadsheet.
 */
public class TableColumnData {
	
	/** The column's {@code DialogObjectConstant}. */
	private DialogObjectConstant diaConst;
	
	/** A short representing the cell style. The value is set according to the values in {@code CellStyle}. */
	private short cellAlignment;
	
	/** A {@code String} representation of the cell format. */
	private String cellFormat;
	
	/** If the column contains a basic container, the name of the placement method used to place the Basic Container is stored here. */
	private String bcMethod;
	
	/** An index used to keep track of the order columns should be looked at, when comparing different rows. */
	private int sortIndex;
	
	/** An enum representing the sort direction. */
	private ListRowComparator.Sorting sorting;
	
	/**
	 * Assigns the provided data to the class' global variables.
	 * 
	 * @param diaConst  The column's {@code DialogObjectConstant}
	 * @param cellAlignment  A short representing the cell style. The value should be set according to the values in {@code CellStyle}
	 * @param cellFormat  A {@code String} representation of the cell format
	 * @param bcMethod  The name of a placement method, if the current column contains a Basic Container
	 * @param sortIndex  An index used to keep track of the order columns should be looked at, when comparing different rows
	 * @param sorting  An enum representing the sort direction
	 */
	public TableColumnData(final DialogObjectConstant diaConst, final short cellAlignment, final String cellFormat, 
						   final String bcMethod, final int sortIndex, final ListRowComparator.Sorting sorting) {
		this.diaConst = diaConst;
		this.cellAlignment = cellAlignment;
		this.cellFormat = cellFormat;
		this.bcMethod = bcMethod;
		this.sortIndex = sortIndex;
		this.sorting = sorting;
	}
	
	/**
	 * Gets the column's {@code DialogObjectConstant}.
	 * 
	 * @return the column's {@code DialogObjectConstant}
	 */
	public DialogObjectConstant getDiaConst() {
		return diaConst;
	}
	
	/**
	 * Gets a short representing the cell style, used for all cells in the current column.
	 * <p>
	 * The value should be set according to the values in {@code CellStyle}
	 * 
	 * @return a short representing the cell style
	 */
	public short getCellAlignment() {
		return cellAlignment;
	}
	
	/**
	 * Gets a {@code String} representation of the cell format, used for all cells in the current column.
	 * 
	 * @return a {@code String} representation of the cell format
	 */
	public String getCellFormat() {
		return cellFormat;
	}
	
	/**
	 * Gets the name of the placement method used to place the Basic Container within this column. 
	 * <p>
	 * If this column does not contain a Basic Container, the returned value is an empty {@code String}.
	 * 
	 * @return name of the placement method used to place the Basic Container within this column, if the column's content is a Basic Container
	 */
	public String getBcMethod() {
		return bcMethod;
	}

	/**
	 * Gets an enum representing the sort direction, i.e. information about whether this column should be sorted in an ascending or descending order, 
	 * or not sorted at all.
	 * <p>
	 * The value is used when sorting tables.
	 * 
	 * @return an enum representing the sort direction
	 */
	public ListRowComparator.Sorting getSorting() {
		return sorting;
	}

	/**
	 * Sets the sort direction, represented as an enum.
	 * <p>
	 * The value is used when sorting tables.
	 * 
	 * @param sorting  the sort direction represented as an enum
	 */
	public void setSorting(ListRowComparator.Sorting sorting) {
		this.sorting = sorting;
	}

	/**
	 * Gets the index used to keep track of the order columns should be looked at, when comparing different rows.
	 * <p>
	 * The value is used when sorting tables.
	 * 
	 * @return the index telling when this column should be checked when comparing two rows
	 */
	public int getSortIndex() {
		return sortIndex;
	}

	/**
	 * Sets the index used to keep track of the order columns should be looked at, when comparing different rows.
	 * <p>
	 * The value is used when sorting tables.
	 * 
	 * @param sortIndex  the index telling when this column should be checked when comparing two rows
	 */
	public void setSortIndex(int sortIndex) {
		this.sortIndex = sortIndex;
	}
}