/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.action;

import java.lang.reflect.Field;

import no.esito.log.Logger;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.support.ActionType;
import no.g9.support.ActionTypeInfo;
import no.g9.support.HookType;

/**
 * A utility for extracting the annotated info about an action type.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
public final class ActionHelper {

    private final static Logger log = Logger.getLogger(ActionHelper.class);

    /**
     * Returns the combined hook type bit mask for the action type
     *
     * @param action the action
     * @return the hook type bit mask for the action
     */
    public final static int hookTypeBitMask(ActionType action) {

        Field field = null;
        Exception ex = null;
        try {
            field = ActionType.class.getDeclaredField(action.name());
            // Work-around for bug #4533479 in java
            field.setAccessible(true);
        } catch (SecurityException e) {
            ex = e;
        } catch (NoSuchFieldException e) {
            ex = e;
        } finally {
            if (ex != null) {
                String msg = "Caught exception while trying to access "
                        + "ActionType of " + action.name() + ":";
                log.error(msg);
                throw new G9ClientFrameworkException(ex);
            }
        }

        ActionTypeInfo actionTypeInfo = null;
        if (field != null) {
            actionTypeInfo = field.getAnnotation(ActionTypeInfo.class);
        }

        if (actionTypeInfo == null) {
            // can happen if enum is missing annotation...
            // In that case - 0 is the best we can do.
            return 0;
        }

        int bitMask = 0;
        HookType[] hookTypes = actionTypeInfo.value();
        for (HookType h : hookTypes) {
            bitMask = bitMask | h.MASK;
        }

        return bitMask;
    }

    /**
     * Check if the action type is a displayable action
     *
     * @param action the action to test
     * @return {@code true} if this is a displayable action
     */
    public static boolean isDisplayableAction(ActionType action) {
        return (hookTypeBitMask(action) & HookType.DISPLAYABLE.MASK) != 0;
    }

    /**
     * Check if the action type is an obtainable action
     *
     * @param action the action to test
     * @return {@code true} if this is an obtainable action
     */
    public static boolean isObtainableAction(ActionType action) {
        return (hookTypeBitMask(action) & HookType.OBTAINABLE.MASK) != 0;
    }

    /**
     * Check if the action is a checkable action
     *
     * @param action the action to test
     * @return {@code true} if this is a checkable action
     */
    public static boolean isCheckableAction(ActionType action) {
        return (hookTypeBitMask(action) & HookType.CHECKABLE.MASK) != 0;
    }

    /**
     * Check if the action is a parameterized action
     * @param action the action to test
     * @return {@code true} if this is a parameterized action
     */
    public static boolean isParameterizedAction(ActionType action) {
        return (hookTypeBitMask(action) & HookType.PARAMETERIZED.MASK) != 0;
    }

    /**
     * Check if the action is a hookable action
     *
     * @param action the action to test
     * @return {@code true} if this is a hookable action
     */
    public static boolean isHookableAction(ActionType action) {
        return (hookTypeBitMask(action) & HookType.HOOKABLE.MASK) != 0;
    }

    /**
     * Check if the action is a clearable action
     *
     * @param action the action to test
     * @return {@code true} if this is a clearable action
     */
    public static boolean isClearableAction(ActionType action) {
        return (hookTypeBitMask(action) & HookType.CLEARABLE.MASK) != 0;
    }

    /**
     * Check if an action is a gui action
     *
     * @param action the action to check
     * @return {@code true} if the action is a gui action
     */
    public static boolean isGuiAction(ActionType action) {
        switch (action) {
            case CLEAROBJECT: // fall through
            case ENABLE: // fall through
            case DISABLE: // fall through
            case SHOW: // fall through
            case HIDE: // fall through
            case FOCUS: // fall through
            case UNSELECT: // fall through
            case EXPAND: // fall through
            case COLLAPSE: // fall through
            case SELECT:
                return true;
            default:
                return false;
        }
    }

    /**
     * Check if an action is a grape action
     *
     * @param action the action to check
     * @return {@code true} if the action is a grape action
     */
    public static boolean isGrapeAction(ActionType action) {
        switch (action) {
            case FIND: // fall through
            case FINDALL: // fall through
            case INSERT: // fall through
            case UPDATE: // fall through
            case SAVE: // fall through
            case DELETE: // fall through
                return true;
            default:
                return false;
        }
    }

}
