/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.action;

import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.Callable;

import no.esito.util.BeanID;
import no.g9.client.core.controller.DialogController;

/**
 * Run action methods, possibly in dedicated thread.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
@BeanID("actionMethodRunner")
public interface ActionMethodRunner {
    
    /**
     * The fixed part of the tread name.
     * Each thread created by action method runners are named according to the
     * following pattern:
     * <br><em>fixed part</em>_<em>prefix</em>_<em>count</em>
     * <p>This string is the fixed part.
     */
    String THREAD_NAME_FIXED_PART = "jVine_";
    
    /**
     * Set the name of the threads that will be created by this runner.
     * If the name does not end with an underscore character ('_'), an 
     * underscore character will be appended. The actual thread name to be used
     * will consist of the prefix supplied by this method, combined with a 
     * thread count number.
     * 
     * @param threadName the name of the thread.
     */
    void setThreadNamePrefix(String threadName);

    /**
     * Perform the specified action method.
     * 
     * @param actionMethod the action method to run.
     */
    void doMethod(Runnable actionMethod);
    
    /**
     * Invoke the specified runnable on the gui thread.
     * @param guiMethod the runnable to be run by the gui thread.
     * @throws InvocationTargetException exceptions thrown by the runnable are 
     * wrapped in an InvocationTargetException and re-thrown.
     */
    void invokeOnGui(Runnable guiMethod) throws InvocationTargetException;
    
    /**
     * Invoke the specified callable on the gui thread.
     * @param <V> the return value
     * @param guiMethod the callable to be invoked by the gui thread.
     * @return the callable's return value
     * @throws InvocationTargetException exceptions thrown by the callable are 
     * wrapped in an InvocationTargetException and re-thrown.
     */
    <V> V invokeOnGui(Callable<V> guiMethod) throws InvocationTargetException;
    
    /**
     * Shuts down the action method runner.
     */
    void shutDown();
    
    /**
     * Get the action queue used to coordinate the gui thread and the working
     * thread (created by this runner).
     * @return the gui-coordinating queue.
     */
    ActionQueue getActionQueue();
    
    /**
     * Release the gui-thread (that is waiting on the coordinating action queue).
     */
    void releaseGui();
   
    /**
     * Set the dialog controller of this action method runner.
     * @param controller the dialog controller.
     */
    void setDialogController(DialogController controller);
    

}
