/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.action;

import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.Callable;

/**
 * A queue to synchronize handing of actions and results between two threads.
 * The queue manages two threads, a <em>posting queue</em>, and a working queue. The
 * <em>posting queue</em> posts a task (by invoking either {@link #perform(Callable)} or
 * {@link #perform(Runnable)}) and waits while the other thread executes the task.
 * The other working thread registers at the {@link #ready()} method when it is
 * ready to execute tasks. 
 */
public interface ActionQueue {

    /**
     * Post the callable task to the queue, and wait for the other thread
     * perform it.
     * 
     * @param <V> the return type of the task.
     * @param task the callable task to perform on the other thread.
     * @return the result of invoking the task.
     * @throws InvocationTargetException if the other thread throws an exception
     *             while performing the task, it is wrapped in an
     *             invocationTargetException and re-thrown.
     */
    public <V> V perform(Callable<V> task) throws InvocationTargetException;

    /**
     * Post the runnable task to the queue, and wait for the other thread to
     * perform it.
     * 
     * @param task the runnable task to perform on the other thread.
     * @throws InvocationTargetException if the other thread throws an exception
     *             while performing the task, it is wrapped in an
     *             invocationTargetException and re-thrown.
     */
    public void perform(Runnable task) throws InvocationTargetException;

    /**
     * The worker registers at this method when it is ready to perform tasks.
     * This method blocks until another thread releases the worker by invoking
     * the release method. 
     */
    public void ready();

    /**
     * Releases the working thread
     */
    public void release();

}
