/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.communication;

import java.io.IOException;
import java.io.StringWriter;

import no.g9.client.core.controller.DialogCallBackValue;
import no.g9.exception.G9BaseException;
import no.g9.message.MessageSystem;

import com.fasterxml.jackson.core.JsonGenerationException;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * A transport class used for sending a DialogCallBackValue as return data
 * from an external dialog.
 */
public class ExternalDialogCallBackValue {

    private String sourceName;
    private Object value;
    private String messageId;
    private Object[] messageArgs;
    private Integer opCode;
    private DialogCallBackValue.Flag flagValue;

    /**
     * Empty constructor, used by Jackson.
     */
    public ExternalDialogCallBackValue() {
        // EMPTY
    }

    /**
     * Create a new object based on the given DialogCallBackValue.
     *
     * @param dialogCallBackValue the dialog call back value to "copy"
     */
    public ExternalDialogCallBackValue(DialogCallBackValue dialogCallBackValue) {
        if (dialogCallBackValue.getSource() != null) {
            sourceName = dialogCallBackValue.getSource().getDialogConstant().getInternalName();
        }
        if (dialogCallBackValue.getValue() != null) {
            value = dialogCallBackValue.getValue();
        }
        if (dialogCallBackValue.getMessage() != null) {
            setMessageId(dialogCallBackValue.getMessage().getMessageID());
            setMessageArgs(dialogCallBackValue.getMessage().getArgs());
        }
        opCode = dialogCallBackValue.getOpCode();
        flagValue = dialogCallBackValue.getFlagValue();
    }

    /**
     * Convert this object to a DialogCallBackValue.
     *
     * Note that the source (dialog controller) is not set, as the external dialog
     * has no dialog controller object in this application partition. Normally,
     * the source will be set to the application window dialog controller.
     *
     * @return a new DialogCallBackValue based on this object
     */
    public DialogCallBackValue asDialogCallBackValue() {
        DialogCallBackValue dialogCallBackValue = new DialogCallBackValue(null, value, opCode);
        dialogCallBackValue.setFlagValue(flagValue);
        if (messageId != null) {
            dialogCallBackValue.setMessage(MessageSystem.getMessageFactory().getMessage(messageId, messageArgs));
        }
        return dialogCallBackValue;
    }

    /**
     * Convert the given object to a JSON representation.
     *
     * @param externalDialogCallBackValue the object to convert
     * @return the JSON representation of the given object
     */
    public static String toJson(ExternalDialogCallBackValue externalDialogCallBackValue) {
        ObjectMapper mapper = new ObjectMapper();
        StringWriter writer = new StringWriter();

        try {
            mapper.writeValue(writer, externalDialogCallBackValue);
            return writer.getBuffer().toString();
        } catch (JsonGenerationException e) {
            throw new G9BaseException(e);
        } catch (JsonMappingException e) {
            throw new G9BaseException(e);
        } catch (IOException e) {
            throw new G9BaseException(e);
        }
    }

    /**
     * Convert the given JSON representation to an ExternalDialogCallBackValue object.
     *
     * @param jsonRepresentation the JSON representation of the given object
     * @return a new ExternalDialogCallBackValue object created from JSON
     */
    public static ExternalDialogCallBackValue fromJson(String jsonRepresentation) {
        if (jsonRepresentation == null) {
            throw new IllegalArgumentException("The JSON representation can not be null");
        }
        ObjectMapper mapper = new ObjectMapper();
        try {
            ExternalDialogCallBackValue cbValue = mapper.readValue(jsonRepresentation, ExternalDialogCallBackValue.class);
            return cbValue;
        } catch (JsonParseException e) {
            throw new G9BaseException(e);
        } catch (JsonMappingException e) {
            throw new G9BaseException(e);
        } catch (IOException e) {
            throw new G9BaseException(e);
        }
    }

    /**
     * @return the sourceName
     */
    public String getSourceName() {
        return sourceName;
    }

    /**
     * @param sourceName the sourceName to set
     */
    public void setSourceName(String sourceName) {
        this.sourceName = sourceName;
    }

    /**
     * @return the value
     */
    public Object getValue() {
        return value;
    }

    /**
     * @param value the value to set
     */
    public void setValue(Object value) {
        this.value = value;
    }

    /**
     * @return the messageId
     */
    public String getMessageId() {
        return messageId;
    }

    /**
     * @param messageId the messageId to set
     */
    public void setMessageId(String messageId) {
        this.messageId = messageId;
    }

    /**
     * @return the messageArgs
     */
    public Object[] getMessageArgs() {
        return messageArgs.clone();
    }

    /**
     * @param messageArgs the messageArgs to set
     */
    public void setMessageArgs(Object[] messageArgs) {
        this.messageArgs = messageArgs.clone();
    }

    /**
     * @return the opCode
     */
    public Integer getOpCode() {
        return opCode;
    }

    /**
     * @param opCode the opCode to set
     */
    public void setOpCode(Integer opCode) {
        this.opCode = opCode;
    }

    /**
     * @return the flagValue
     */
    public DialogCallBackValue.Flag getFlagValue() {
        return flagValue;
    }

    /**
     * @param flagValue the flagValue to set
     */
    public void setFlagValue(DialogCallBackValue.Flag flagValue) {
        this.flagValue = flagValue;
    }

}
