/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.controller;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import no.g9.os.AttributeConstant;
import no.g9.os.RoleConstant;

/**
 * Stores all field values for a given domain instance and a flag.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
public final class FieldData {

    /**
     * The role of the instance represented by this class.
     */
    private final RoleConstant role;

    /**
     * The instance flag.
     */
    private InstanceFlag flag;

    /**
     * The field map, maps the (generated) enum constant to the corresponding
     * field value.
     */
    private final Map<AttributeConstant, Object> fieldMap = Collections
            .synchronizedMap(new HashMap<AttributeConstant, Object>());

    /**
     * Constructs a new instance field data object for the specified object
     * selection node.
     * 
     * @param role the object selection node of this instance field data.
     */
    public FieldData(RoleConstant role) {
        this.role = role;
    }

    /**
     * Gets the object selection node of this instance.
     * 
     * @return the object selection node of this instance.
     */
    public final RoleConstant getRole() {
        return role;
    }

    /**
     * Gets the value for the specified field.
     * 
     * @param fieldIdentifier The (generated) enum denoting the role attribute
     * @return the field value.
     */
    public Object getFieldValue(AttributeConstant fieldIdentifier) {
        return fieldMap.get(fieldIdentifier);
    }

    /**
     * @param fieldIdentifier The (generated) enum denoting the role attribute
     * @param fieldValue The value of the specified field.
     * @return the field value.
     */
    public Object setFieldValue(AttributeConstant fieldIdentifier,
            Object fieldValue) {
        return fieldMap.put(fieldIdentifier, fieldValue);
    }

    /**
     * Set the instance flag for this set of field data.
     * 
     * @param flag the new instance flag.
     */
    public void setFlag(InstanceFlag flag) {
        this.flag = flag;
    }

    /**
     * Check if there is data in this field data instance.
     * 
     * @return true if there is data or it is not marked for delete.
     */
    public boolean hasData() {
        return this.flag != InstanceFlag.DELETE && !fieldMap.isEmpty();
    }

    /**
     * Returns the set of field entries contained in this field data.
     * 
     * @return the entries in this field data collection.
     */
    public Set<Entry<AttributeConstant, Object>> entries() {
        return fieldMap.entrySet();
    }

    /**
     * Get the instance flag for this set of field data.
     * 
     * @return the instance flag.
     */
    public InstanceFlag getFlag() {
        return flag;
    }

    @Override
    public String toString() {
        return this.getClass().getSimpleName() + " (" + role + ") " + entries();
    }
    
    

}