/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.model;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import no.esito.jvine.controller.OSNode;
import no.esito.log.Logger;
import no.g9.client.core.controller.RoleState;
import no.g9.os.AttributeConstant;
import no.g9.os.Key;
import no.g9.os.KeyTool;
import no.g9.os.RelationCardinality;
import no.g9.os.RelationType;

/**
 * Implements a standard tree node.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 *
 * @param <T> The domain type of this tree node.
 */
public class TreeNodeImpl<T> implements TreeNode<T> {

    /** The logger */
    private final static Logger log = Logger.getLogger(TreeNodeImpl.class);

    /** The current instance */
    private T current;

    /** The parent tree node */
    private final TreeNode<?> parent;

    /** The object selection node of this tree node */
    private final OSNode<T> osNode;

    /**
     * The collection of (domain) instances that can be reached from this node.
     */
    private final Collection<TreeNode<?>> children;

    /**
     * Constructs a new TreeNode with the specified tree node parent and object
     * selection node.
     *
     * @param parent the parent of this tree node
     * @param osNode the object selection role of this node.
     */
    public TreeNodeImpl(TreeNode<?> parent, OSNode<T> osNode) {

        if (log.isTraceEnabled()) {
            log.trace("Creating tree node for " + osNode + " with parent "
                    + parent);
        }

        this.parent = parent;
        this.osNode = osNode;
        children = new ArrayList<TreeNode<?>>();
    }

    @Override
    public TreeNode<?> getParent() {

        return parent;
    }

    @Override
    public Collection<TreeNode<?>> getChildren() {
        return children;
    }

    @Override
    public void addChild(TreeNode<?> child) {
        children.add(child);
        if (log.isTraceEnabled()) {
            log.trace(this + " added child " + child);
        }

    }

    @Override
    public Collection<OSNode<?>> setCurrentInstance(T current) {

        if (log.isDebugEnabled()) {
            log.debug(this + " setting " + current + " as current instance,"
                    + " and propagating change to children.");
        }

        updateCurrent(current);

        setClean();

        return changed();
    }

    @Override
    public Collection<OSNode<?>> setCurrentRootInstances(Collection<T> instances) {
        if (log.isDebugEnabled()) {
            log.debug("Registering current on root's sentinel.");
        }

        if (getOSNode().getRelationType() != RelationType.ROOT) {
            String msg = getOSNode().toString()
                    + " is not a root node."
                    + " Only root nodes can set a collection of current instances.";
            throw new UnsupportedOperationException(msg);
        }

        ((TreeNodeSentinel) parent).addRootInstances(osNode, instances);

        setCleared();
        return changed();

    }

    /**
     * @return a list of changed nodes.
     */
    private List<OSNode<?>> changed() {
        List<OSNode<?>> changed = new LinkedList<OSNode<?>>();
        changed.add(osNode);

        for (TreeNode<?> child : getChildren()) {

            changed.addAll(child.computeCurrent());
        }
        return changed;
    }





    @Override
    public Collection<OSNode<?>> computeCurrent() {

        RelationCardinality cardinality = getOSNode().getCardinality();

        if (cardinality == RelationCardinality.MANY) {
            if (log.isTraceEnabled()) {
                log.trace(this + " is many-related. Clearing current and all "
                        + " children.");
            }
             Collection<OSNode<?>> cleared = clear(false);
             setCleared();
             return cleared;
        }

        Collection<T> instances = getInstances();

        if (instances.size() > 1) {
            String msg = "Encounterend more than one instance"
                    + " when getting instances for the " + getOSNode()
                    + " role";

            throw new RuntimeException(msg);
        }

        if (instances.isEmpty()) {
            updateCurrent(null);
            setClean();
        } else {
            for (T tmp : instances) {
               updateCurrent(tmp);

                if (log.isDebugEnabled()) {
                    log.debug(this + " current selected " + current);
                }

                if (current != null) {
                    setClean();
                } else {
                    setCleared();
                }
            }
        }

        List<OSNode<?>> changed = new LinkedList<OSNode<?>>();
        changed.add(getOSNode());

        for (TreeNode<?> child : getChildren()) {
            changed.addAll(child.computeCurrent());
        }

        return changed;

    }

    @Override
    public T getCurrentInstance() {
        return current;
    }

    @Override
    public Object getChildRelation(OSNode<?> child) {
        if (getCurrentInstance() == null) {

            if (log.isDebugEnabled()) {
                log.debug(this + " cannot get relation to " + child
                        + " since current parent is null");
            }

            return null;
        }

        return getOSNode().getRelation(getCurrentInstance(),
                child.getRoleConstant());

    }

    @Override
    @SuppressWarnings("unchecked")
    public Collection<T> getInstances() {
        if (log.isTraceEnabled()) {
            log.trace(this + " getting instances");
        }

        Object parentToThis = getParent().getChildRelation(getOSNode());

        boolean isRoot = getOSNode().getRelationCardinality() == RelationCardinality.ROOT;
        if (isRoot) {
            if (parentToThis == null) {
                parentToThis = getCurrentInstance();
            }
        }

        Collection<T> instances = Collections.EMPTY_SET;
        if (parentToThis instanceof Collection) {
            instances = (Collection<T>) parentToThis;
            if (instances instanceof Set<?>) {
                Set<T> tmp = new LinkedHashSet<T>();
                tmp.addAll(instances);
                instances = tmp;
                if (!isRoot) {
                    Object currentParent = getParent().getCurrentInstance();
                    if (currentParent != null) {
                        getOSNode().getParent().setRelation(currentParent, instances, getOSNode().getRoleConstant());
                    }
                }

            }

            if (log.isTraceEnabled()) {
                log.trace("Relation is a collection, "
                        + "no further action is needed.");
            }


        } else if (parentToThis != null) {

            if (log.isTraceEnabled()) {
                log.trace("Relation is a single object. "
                        + "Creating a collection to store it in.");
            }

            instances = new LinkedHashSet<T>(1);
            instances.add(getOSNode().castToType(parentToThis));

        } else if (getCurrentInstance() != null) {
            instances = new LinkedHashSet<T>(1);
            instances.add(getCurrentInstance());
        }

        if (log.isTraceEnabled()) {
            log.trace(this + " when asked for instances returns " + instances);
        }

        return instances;
    }



    @Override
    public OSNode<T> getOSNode() {
        return osNode;
    }

    @Override
    public Collection<OSNode<?>> clearCurrent(boolean intercept) {
        if (log.isTraceEnabled()) {
            log.trace(this + " clearing node.");
        }
        Collection<OSNode<?>> changedNodes = new LinkedHashSet<OSNode<?>>();

        changedNodes.add(this.getOSNode());
        updateCurrent(null);

        for (TreeNode<?> child : getChildren()) {
            changedNodes.addAll(child.clear(intercept));
        }

        return changedNodes;
    }

    @Override
    public Collection<OSNode<?>> clear(boolean intercept) {
        if (log.isTraceEnabled()) {
            log.trace(this + " clearing node.");
        }
        Collection<OSNode<?>> changedNodes = new LinkedHashSet<OSNode<?>>();

        changedNodes.add(this.getOSNode());
        updateCurrent(null);

        if (log.isTraceEnabled()) {
            log.trace(this + " clearing current");
        }

        if (getOSNode().getRelationType() == RelationType.ROOT) {
            if (log.isTraceEnabled()) {
                log.trace(this + " clearing the sentinel.");
            }
            ((TreeNodeSentinel) parent).clearRootInstance(getOSNode());
        }

        for (TreeNode<?> child : getChildren()) {
            changedNodes.addAll(child.clear(intercept));
        }

        if (log.isTraceEnabled()) {
            log.trace(this + " cleared nodes are: " + changedNodes);
        }

        setCleared();

        return changedNodes;

    }

    /**
     *
     */
    private void setCleared() {
        getOSNode().setState(RoleState.CLEARED);
        getOSNode().resetState();
    }

    @Override
    public Collection<OSNode<?>> removeCurrent() {
        updateCurrent(null);
        return Arrays.asList(new OSNode<?>[] { getOSNode() });
    }

    @Override
    public String toString() {
        return "TreeNode [" + getOSNode() + "]";
    }

    private void setClean() {
        getOSNode().setState(RoleState.CLEAN);
        getOSNode().resetState();
    }

    private void updateCurrent(T newCurrent) {
        if (log.isDebugEnabled()) {
            log.debug("Setting current to: " + newCurrent);
        }
        this.current = newCurrent;

        if (getOSNode().isMany()) {
            Collection<T> instances = getInstances();
            if (newCurrent != null && instances != null) {

                if (log.isTraceEnabled()) {
                    log.trace("Adding instance to complete set of instances.");
                }
                removeInstance(instances, newCurrent);
                instances.add(newCurrent);

                if (log.isTraceEnabled()) {
                    log.trace("Instances: " + instances);
                }

            }
        } else if (!getOSNode().isRoot()){
            Object currentParent = getParent().getCurrentInstance();
            if (currentParent != null) {
                getParent().getOSNode().setRelation(currentParent, newCurrent, getOSNode().getRoleConstant());
            }
        }

    }

    private void removeInstance(Collection<T> instances, T instance) {        
        if (!instances.remove(instance)) {
            // The instance was not found using the equals method, try other keys if using a generated key
            List<Key> keys = getOSNode().getKeys();
            if (!keys.isEmpty()) {
                Key equalsKey = keys.get(0);
                if (equalsKey.getAttributes().length == 1) {

                    // Try other unique keys
                    for (T possibleInstance : instances) {
                        if (getOSNode().matchUniqueKeys(possibleInstance, instance, false)) {
                            instances.remove(possibleInstance);
                            return;
                        }
                    }
                    
                    // Try matching all fields except the one in equalsKey
                    // This is only done if the value of the equals attribute is undefined
                    AttributeConstant equalsAttribute = equalsKey.getAttributes()[0];
                    for (T possibleInstance : instances) {
                        Object equalsValue = getOSNode().getValue(possibleInstance, equalsAttribute);
                        if (!KeyTool.isDefined(equalsValue) && equalsUsingAttributes(getOSNode().getAttributeConstants(), equalsAttribute, possibleInstance, instance)) {
                            instances.remove(possibleInstance);
                            return;
                        }
                    }
                }
            }
        }
    }

    private boolean equalsUsingAttributes(AttributeConstant[] attributes, AttributeConstant excludedAttribute, T anObject, T anOtherObject) {
        if (log.isTraceEnabled()) {
            log.trace(this + " equals using attributes " + Arrays.toString(attributes));
        }
        if (anObject == anOtherObject) {
            return true;
        }
        if (anObject == null) {
            return false;
        }
        boolean foundMatch = true;
        for (int i = 0; i < attributes.length && foundMatch; i++) {
            if (attributes[i] != excludedAttribute) {
                Object value1 = getOSNode().getValue(anObject, attributes[i]);
                Object value2 = getOSNode().getValue(anOtherObject, attributes[i]);
                if (value1 != null && value2 != null) {
                    foundMatch = KeyTool.isDefined(value1) && KeyTool.isDefined(value2);
                    // KeyTool also checks for null reference.
                    if (foundMatch) {
                        foundMatch = value1.equals(value2);
                    }
                }
                if (log.isTraceEnabled()) {
                    log.trace("Comparing values in " + attributes[i] + ", value1: " + value1 + ", value2: " + value2
                            + ", equals so far: " + foundMatch);
                }
            }
        }
        return foundMatch;
    }

}