/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

import java.util.HashMap;
import java.util.Map;

import no.g9.client.core.controller.ApplicationController;
import no.g9.client.core.controller.DialogController;
import no.g9.support.ActionType;
import no.g9.support.action.ActionTarget;

/**
 * This class is responsible for creating actions and managing hooks.
 */
public class ActionFactory {

    /** The map from hook key (target and type) to hook */
    private Map<ActionHookKey, ActionHookList<?>> hookMap =
            new HashMap<ActionHookKey, ActionHookList<?>>();

    /**
     * Creates an action that corresponds to the specified type and target.
     *
     * @param <T> The target class type.
     * @param actionType the type of action
     * @param target the action target
     * @param targetType the target type class
     * @param actionTask the action task
     * @param dialogController the action's dialog controller
     * @return the G9 action
     */
    @SuppressWarnings("unchecked")
    public <T> G9Action<T> getAction(ActionType actionType, ActionTarget target,
            Class<T> targetType, ActionTask<T> actionTask,
            DialogController dialogController) {

        G9Action<T> action = new G9Action<T>(actionType, target, actionTask,
                            targetType, dialogController);

        ActionHookKey hookKey = new ActionHookKey(actionType, target);
        ActionHookList<T> actionHookList = (ActionHookList<T>) hookMap.get(hookKey);

        action.setActionHookList(actionHookList);

        return action;
    }

    /**
     * Create the specified invoke action that will bind parameters and invoke
     * the remote service.
     *
     * @param rst the remote service target.
     * @param dialogController the dialog controller
     * @return the invoke action that invokes the remote service.
     */
    @SuppressWarnings("unchecked")
    public G9Action<?> getInvokeAction(RemoteServiceTarget rst,
            DialogController dialogController) {
        G9Action<Object> action =
                new G9Action<Object>(ActionType.INVOKE, rst,
                        (ActionTask<Object>) rst.getActionTask(), Object.class,
                        dialogController);
        ActionHookKey hookKey = new ActionHookKey(ActionType.INVOKE, rst.getService());
        ActionHookList<Object> actionHooks =
                (ActionHookList<Object>) hookMap.get(hookKey);
        action.setActionHookList(actionHooks);
        return action;
    }

    /**
     * Returns a G9Action for the specified application controller.
     *
     * @param <T> the action target type
     * @param actionType the action type
     * @param target the action target
     * @param targetType the action target type
     * @param actionTask the action task
     * @param applicationController the application controller
     * @return the G9 action.
     */
    @SuppressWarnings("unchecked")
    public <T> G9Action<T> getAction(ActionType actionType, ActionTarget target,
            Class<T> targetType, ActionTask<T> actionTask,
            ApplicationController applicationController) {
        G9Action<T> action =
                new G9Action<T>(actionType, target, actionTask, targetType,
                        applicationController);
        ActionHookKey hookKey = new ActionHookKey(actionType, target);
        ActionHookList<T> actionHooks = (ActionHookList<T>) hookMap.get(hookKey);
        action.setActionHookList(actionHooks);
        return action;
    }

    /**
     * Registers a hook for the given target and type.
     *
     * @param <T> The target type
     * @param actionTarget the action target
     * @param actionType the action type
     * @param hook an hook for the specified action
     */
    @SuppressWarnings("unchecked")
    public <T> void registerHook(ActionTarget actionTarget, ActionType actionType, ActionHook<T> hook) {
        ActionHookKey key = new ActionHookKey(actionType, actionTarget);
        ActionHookList<T> hooks = (ActionHookList<T>) hookMap.get(key);
        if (hooks == null) {
            hooks = new ActionHookList<T>();
            hookMap.put(key, hooks);
        }
        hooks.getHooks().add(hook);
    }

    /**
     * Get the list of action hooks for the specified action target and type.
     *
     * @param actionTarget the target of the action
     * @param actionType the action type
     * @return the action hooks registered for the specified action target and type
     */
    public ActionHookList<?> getActionHookList(ActionTarget actionTarget, ActionType actionType) {
        ActionHookKey key = new ActionHookKey(actionType, actionTarget);
        return hookMap.get(key);
    }

    /**
     * The key class used to identify a hook
     */
    private static class ActionHookKey {
        /** The action type */
        private final ActionType actionType;
        /** The action target */
        private final Object actionTarget;
        /** The hash code for this action hook key - calculated in constructor. */
        private final int hashCode;

        /**
         * Constructs a new action hook key
         *
         * @param actionType the action type
         * @param actionTarget the action target
         */
        ActionHookKey(ActionType actionType, Object actionTarget) {
            this.actionType = actionType;
            this.actionTarget = actionTarget;
            int caclHash = 17;
            caclHash = caclHash * 37 + actionType.hashCode();
            caclHash = caclHash * 37 + actionTarget.hashCode();
            hashCode = caclHash;

        }

        @Override
        public boolean equals(Object obj) {
            // According to Effective Java

            if (this == obj) {
                return true;
            }

            if (!(obj instanceof ActionHookKey)) {
                return false;
            }

            ActionHookKey other = (ActionHookKey) obj;

            return actionType.equals(other.actionType)
                    && actionTarget.equals(other.actionTarget);
        }

        @Override
        public int hashCode() {
            return hashCode;
        }

        @Override
        public String toString() {
            return "[" + actionType + " " + actionTarget + "]";
        }
    }

}