/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

/**
 * The base action hook class
 * 
 * @param <T> The target type
 */
public abstract class ActionHook<T> implements Hookable<T> {

    /** The current action of this hook */
    private transient G9Action<T> currentAction;

    /**
     * Gets the current g9 action. Synchronized on "this".
     * 
     * @return The g9 action associated with this hook.
     */
    public final synchronized G9Action<T> getCurrentAction() {

        return currentAction;
    }
    
    /**
     * Get the event that triggered the current action. Synchronized on "this".
     * @return the triggering event
     */
    public final synchronized EventContext getEvent() {
        return currentAction.getEvent();
    }

    /**
     * Set the current G9 action. If this action hook already has a current
     * action (which is running), this method will block until the current
     * action is finished, before setting the new, current action.
     * 
     * @param currentAction the action to associate with this hook
     * @throws InterruptedException if interrupted while blocking.
     */
    synchronized void setCurrentAction(G9Action<T> currentAction)
            throws InterruptedException {
        while (this.currentAction != null
                && !this.currentAction.equals(currentAction)) {
            wait();
        }
        this.currentAction = currentAction;
        notify();
    }

    /**
     * Removes the current G9 action. If the specified action is not the
     * current action this method will block.
     * 
     * @param action the action to remove (must be the current one)
     * @throws InterruptedException if interrupted while blocking.
     */
    synchronized void removeCurrentAction(G9Action<T> action)
            throws InterruptedException {
        while (this.currentAction != action) {
            wait();
        }
        this.currentAction = null;
        notify();
    }

    /**
     * Returns the string representation of this action hook, containing the
     * current action name and target info. Used for debugging.
     */
    @Override
    public String toString() {
        return getClass().getName() + " " + getCurrentAction().toString();
    }

}