/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

import no.g9.client.core.controller.ApplicationController;
import no.g9.message.Message;

/**
 * The action hook adapter offers an empty implementation of the basic hook
 * methods. Application programmers should extend this adapter (or one of it's
 * sub-classes) and override the hook methods they are interested in.
 * 
 * @param <T> The action target type
 */
public abstract class ActionHookAdapter<T> extends ActionHook<T> {
    
    /**
     * Cancel the ongoing action. 
     * @see G9Action#cancel()
     * @throws IllegalStateException if the action cannot be cancelled.
     */
    protected void cancelAction() {
        G9Action<T> currentAction = getCurrentAction();
        if (currentAction != null) {
            currentAction.cancel();
        } else {
            String msg = "Action has finished execution. Cancelling not possible.";
            throw new IllegalStateException(msg);
        }
    }
    
    /**
     * Get the application controller of the ongoing action, or null if 
     * there is no current, ongoing action.
     * @return the application controller.
     */
    protected final ApplicationController getApplicationController() {
        G9Action<T> currentAction = getCurrentAction();
        return currentAction != null ? currentAction.getApplicationController()
                : null;
    }
    
    @Override
    public void initialized() {
        // Empty.
    }

    @Override
    public void performed(T result) {
        // Empty.
    }

    @Override
    public void succeeded() {
        // Empty.
    }

    @Override
    public Message failed(Throwable cause, Message message) {
        if (message != null) {
            return message;
        }
        return getCurrentAction().getDefaultFailedMessage(cause);
    }

    @Override
    public void cancelled() {
        // Empty.
    }

    @Override
    public void finished() {
        // Empty.
    }
}