/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;

import no.esito.jvine.action.CancelException;
import no.esito.jvine.action.HookMethod;
import no.esito.jvine.controller.JVineController;
import no.esito.log.Logger;
import no.g9.client.core.controller.DialogController;
import no.g9.support.ActionType;
import no.g9.support.action.ActionTarget;

/**
 * The checkable action is an action that will perform a dialog check.
 *
 * @param <V> The return type of this action
 */
public class CheckableAction<V> extends G9Action<V> {

    /** The checkable hooks (if any) */
    private List<Checkable> checkableHooks = new ArrayList<Checkable>();

    /** The logger */
    private static Logger log = Logger.getLogger(CheckableAction.class);

    /**
     * Constructs a new checkable g9 action.
     *
     * @param actionType the action type
     * @param actionTarget the action target
     * @param actionTask the callable implementing the action task
     * @param resultClass the target class *
     * @param dialogController the dialog controller that initiated this action
     */
    protected CheckableAction(ActionType actionType, ActionTarget actionTarget,
            ActionTask<V> actionTask, Class<V> resultClass,
            DialogController dialogController) {

        super(actionType, actionTarget, actionTask, resultClass,
                dialogController);

    }

    @Override
    protected void setActionHookList(ActionHookList<V> actionHookList) {
        super.setActionHookList(actionHookList);
        if (actionHookList != null) {
            checkableHooks = actionHookList.getCheckableHooks();
        }
    }

    /**
     * Returns the list of checkable hooks.
     *
     * @return the list of checkable hooks, possibly an empty list
     */
    @Override
    List<Checkable> getCheckableHooks() {
        return checkableHooks;
    }

    /**
     * The checking state, invokes the check task, the hook and returns the
     * result
     */
    protected Callable<Void> checking = new Callable<Void>() {

        @Override
        public Void call() throws Exception {
            if (!CANCELLED) {
                ActionType checkActionType = getCheckActionType();

                ActionTask<Boolean> checkActionTask = JVineController
                        .getInstance(getController()).getCheckDialogTask(
                                checkActionType, getActionTarget());
                checkActionTask.FLAG = getFlag();
                Boolean checkResult = checkActionTask.call();

                if (checkResult == null) {
                    checkResult = Boolean.FALSE;
                    log.info(CheckableAction.this + " check action failed. "
                            + "Using default value false");
                }

                if (!checked(checkResult).booleanValue()) {
                    cancel();
                }
                log.info(CheckableAction.this + " checked");
            }
            return null;

        }

        @SuppressWarnings("deprecation")
        private ActionType getCheckActionType() {
            switch (getActionType()) {
                case FIND: // fall through

                case FINDALL: // fall through

                case INVOKE:
                    return ActionType.CHECK_FIND;

                case INSERT:
                	return ActionType.INSERT;

                case UPDATE:
                	return ActionType.UPDATE;

                case SAVE:
                    return ActionType.SAVE;

                case DELETE:
                    return ActionType.CHECK_DELETE;

                case CLOSE:
                    return ActionType.CHECK_CLOSE;
                case CLEAR:
                    return ActionType.CHECK_CLEAR;
                case OPEN:
                    return ActionType.OPEN;
                default:
                    return null;
            }
        }

    };

    @Override
    public V call() throws Exception {
        log.info("Starting execution of " + this);
        V result = null;
        try {
//            initializing.call();
            checking.call();
            try {
//                result = performing.call();
            } catch (CancelException ce) {
                log.info(this + " cancelled while performing.");
                if (log.isTraceEnabled()) {
                    log.trace("Cancel trace:", ce);
                }
                cancel();
            }
//            succeeding.call();
//            cancelling.call();
        } catch (Exception e) {
            failed(e);
            log.info(this + " failed", e);
        } finally {
//            finishing.call();
            cleanUpHook();
        }

        return result;
    }

    /**
     * The checked hook. Invoked before the performed hook.
     *
     * @param checkResult the check result
     * @return the result from the checked hook
     * @exception Exception re-throws exception from hook method
     */
    Boolean checked(final Boolean checkResult) throws Exception {
        if (shouldInvokeHook() && !getCheckableHooks().isEmpty()) {
            String methodName = "checked";
            Class<?>[] params = {
                Boolean.class
            };
            ThreadType threadType = getThreadType(methodName, params);
            HookMethod<Boolean> checkHook = new HookMethod<Boolean>(methodName) {
                @Override
                public Boolean call() throws Exception {
                    Boolean result = checkResult;
                    for (Checkable hook : getCheckableHooks()) {
                        result = hook.checked(result);
                    }
                    return result;
                }
            };
            return hookInvoker.execute(getApplicationController(), threadType,
                    checkHook);
        }

        return checkResult;
    }

}