/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

import no.g9.client.core.view.ListRow;

/**
 * Class used to communicate an event from the view to the controller. An event
 * method object consists of a method name (mandatory), the source of the event
 * (typically a gui-widget) and an event (e.g. a gui-event).
 */
public final class EventContext {

    /** String denoting the method */
    private String methodName;

    /** The event source, e.g. the component/widget/thing that created the event */
    private Object source;
    
    /** The event */
    private Object event;

    /** The ListRow in which the event occurred */
    private ListRow listRow;

    /**
     * Constructs a new Trigger object with the specified method name,
     * <code>null</code> source, <code>null</code> event and
     * <code>null</code> listRow.
     * 
     * @param methodName the name of the method that should be invoked.
     */
    public EventContext(String methodName) {
        this(methodName, null, null, null);
    }

    /**
     * Constructs a new Trigger object with the specified method name and source,
     * and <code>null</code> event and <code>null</code> listRow.
     * 
     * @param methodName the name of the method that should be invoked.
     * @param source the source of the event.
     */
    public EventContext(String methodName, Object source) {
        this(methodName, source, null, null);
    }

    /**
     * Constructs a new Trigger object with the specified method name, source
     * and event, and <code>null</code> listRow.
     * 
     * @param methodName the name of the method that should be invoked.
     * @param source the source of the event (e.g. the widget)
     * @param event the event
     */
    public EventContext(String methodName, Object source, Object event) {
        this(methodName, source, event, null);
    }

    /**
     * Constructs a new Trigger object. All other constructors delegates to
     * this.
     * 
     * @param methodName the name of the method that should be invoked.
     * @param source the source of the event (e.g. the widget)
     * @param event the event
     * @param listRow the ListRow in which the event occurred.
     * @throws IllegalArgumentException if <code>methodName</code> is
     *             <code>null</code>.
     */
    public EventContext(String methodName, Object source, Object event, ListRow listRow) {
        if (methodName == null) {
            throw new IllegalArgumentException("Method name cannot be null");
        }
        this.methodName = methodName;
        this.source = source;
        this.event = event;
        this.setListRow(listRow);
    }

    /**
     * Gets the method name.
     * 
     * @return the name of the method to invoke.
     */
    public String getMethodName() {
        return methodName;
    }

    /**
     * Gets the source of the event. Typically, the source is target-specific,
     * e.g. the view-component/widget that originated the event.
     * 
     * @return the source of the event.
     */
    public Object getSource() {
        return source;
    }

    /**
     * Gets the event that originated this trigger object. Typically, this will
     * be target-specific, as different targets have their own event types.
     * 
     * @return the event object, target specific.
     */
    public Object getEvent() {
        return event;
    }

    /**
     * Returns the string representation of this EventMethod. The returned
     * string contains the method name, the event object and the event source.
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return "Method: " + getMethodName() + ", event: " + event + ", source:"
                + source + ", listRow:" + listRow;
    }

    /**
     * Sets the ListRow in which the event occurred.
     * 
     * @param listRow
     *            the listRow to set.
     */
    public void setListRow(ListRow listRow) {
        this.listRow = listRow;
    }

    /**
     * @return the listRow
     */
    public ListRow getListRow() {
        return listRow;
    }

}
