/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

import no.esito.jvine.action.CancelException;
import no.esito.jvine.action.HookMethod;
import no.esito.jvine.controller.JVineAppController;
import no.esito.jvine.controller.JVineController;
import no.esito.jvine.controller.OSNode;
import no.esito.jvine.model.CurrentRoleObject;
import no.esito.jvine.view.MessageUtil;
import no.esito.log.Logger;
import no.esito.util.ServiceLoader;
import no.g9.client.core.controller.ApplicationController;
import no.g9.client.core.controller.DialogController;
import no.g9.exception.G9BaseException;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;
import no.g9.message.ReplySetType;
import no.g9.os.RoleConstant;
import no.g9.support.ActionType;
import no.g9.support.ActionTypeInfo;
import no.g9.support.HookType;
import no.g9.support.action.ActionTarget;

/**
 * The base g9 action class.
 *
 * @param <V> The target class this action operates on.
 */
@SuppressWarnings("rawtypes")
public class G9Action<V> implements Callable<V> {

    /**
     *
     */
    final MessageUtil messageUtil = ServiceLoader
            .getService(MessageUtil.class);

    /**
     * @return MessageUtil
     */
    public MessageUtil getMessageUtil() {
    	return messageUtil;
    }

    /** Map containing the default thread type for the hook-methods. */
    private static Map<String, ThreadType> defaultAnnotations = new HashMap<String, ThreadType>();

    private EventContext event;

    /** Used in dialog open events. */
    private Object actionSetupValue;

    /** General purpose flag */
    private Boolean FLAG = Boolean.FALSE;

    /**
     * Get the general purpose FLAG value.
     * @return the flag value
     * @see #getFlag()
     */
    public Boolean getFlag() {
        return FLAG;
    }

    /**
     * Set a general purpose flag. This flag is used by the following actions:
     * <ul>
     * <li>OPEN - if set to true, the action will attempt to always open a
     * <em>new</em> dialog.
     * </ul>
     *
     * @param flagValue the value of the flag.
     */
    public void setFlag(Boolean flagValue) {
        FLAG = flagValue;
    }

    /**
     * Get the event that triggered this action.
     *
     * @return the triggering event
     */
    public EventContext getEvent() {
        return event;
    }

    /**
     * Set the event that triggered this action.
     *
     * @param event the triggering event
     */
    public void setEvent(EventContext event) {
        this.event = event;
    }

    // Set up the map of default annotations using reflection.
    static {
        Class<?>[] interfaces = { Hookable.class, Checkable.class,
                Obtainable.class, Displayable.class };

        for (Class<?> clazz : interfaces) {
            for (Method m : clazz.getDeclaredMethods()) {
                ThreadInfo annotation = m.getAnnotation(ThreadInfo.class);
                if (annotation != null) {
                    defaultAnnotations.put(m.getName(), annotation.value());
                }
            }
        }

    }

    /**
     * The hook invoker
     */
    protected static final HookInvoker hookInvoker = ServiceLoader
            .getService(HookInvoker.class);

    /**
     * @return HookInvoker
     */
    public HookInvoker getHookInvoker() {
    	return hookInvoker;
    }

    /** The list of action hooks */
    private ActionHookList<V> actionHookList;

    /**
     * The action target
     */
    private ActionTarget actionTarget;

    /**
     * The type of action
     */
    private ActionType actionType;

    /** The actual task to perform */
    protected ActionTask<V> actionTask;

    /** The logger. */
    private static Logger log = Logger.getLogger(G9Action.class);

    /**
     * @return Logger
     */
    public static synchronized Logger getLog() {
		return log;
	}

	/**
	 * @param log Logger
	 */
	public static synchronized void setLog(Logger log) {
		G9Action.log = log;
	}

	/** The result class type */
    private Class<V> resultClass;

    /**
     * @return resultClass
     */
    public Class<V> getResultClas() {
    	return resultClass;
    }

    /** The dialog controller */
    private DialogController dialogController;

    /**
     * @return DialogController
     */
    public DialogController getDialogController() {
    	return dialogController;
    }

    /**
     * The ApplicationController
     */
    ApplicationController applicationController;

    private Message DEFAULT_FAILED_MSG;

    /**
     * Test whether hooks are turned on or off.
     *
     * @return <code>true</code> if the hook mechanism is turned on.
     */
    protected final boolean isInvokeHooksOn() {
        JVineAppController appCtrl = JVineAppController
                .getInstance(applicationController);
        boolean applicationHooks = appCtrl.isInvokeHooksOn();
        if (log.isDebugEnabled() && !applicationHooks) {
            log.debug(this + " all hooks are switched off!");
        }
        if (getController() == null) {
            return applicationHooks;
        }
        JVineController dlgCtrl = JVineController.getInstance(getController());
        boolean dlgHooks = dlgCtrl.isInvokeHooksOn();
        if (log.isDebugEnabled() && !dlgHooks && applicationHooks) {
            log.debug(this + " hooks are turned off!");
        }
        return applicationHooks && dlgHooks;
    }

    /**
     * Constructs a new g9 action
     *
     * @param actionType the action type
     * @param actionTarget the action target
     * @param actionTask the callable implementing the action task
     * @param resultClass the target class
     * @param dialogController the dialog controller that initiated this action
     */
    protected G9Action(ActionType actionType, ActionTarget actionTarget,
            ActionTask<V> actionTask, Class<V> resultClass,
            DialogController dialogController) {
        this(actionType, actionTarget, actionTask, resultClass);
        this.dialogController = dialogController;
        this.applicationController = dialogController
                .getApplicationController();
    }

    /**
     * Constructs a new g9 action
     *
     * @param actionType the action type
     * @param actionTarget the action target
     * @param actionTask the callable implementing the action task
     * @param resultClass the target class
     * @param applicationController the dialog controller that initiated this
     *            action
     */
    protected G9Action(ActionType actionType, ActionTarget actionTarget,
            ActionTask<V> actionTask, Class<V> resultClass,
            ApplicationController applicationController) {
        this(actionType, actionTarget, actionTask, resultClass);
        this.dialogController = null;
        this.applicationController = applicationController;

    }

    private G9Action(ActionType actionType, ActionTarget actionTarget,
            ActionTask<V> actionTask, Class<V> resultClass) {
        this.actionType = actionType;
        this.actionTarget = actionTarget;
        this.actionTask = actionTask;
        this.resultClass = resultClass;
    }

    /**
     * Get the default failed action message.
     *
     * @param cause the throwable that caused the failure
     *
     * @return the default failed message.
     */
    synchronized Message getDefaultFailedMessage(Throwable cause) {
        cause = unwrap(cause);
        if (cause instanceof G9BaseException) {
            G9BaseException gbe = (G9BaseException) cause;
            Message errMsg = gbe.getErrMsg();
            if (errMsg != null) {
                ReplySetType validReplies = errMsg.getValidReplies();
                if (validReplies == null
                        || validReplies.equals(ReplySetType.REPLSET_NONE)) {
                    errMsg.setValidReplies(ReplySetType.REPLSET_OK);
                }
                return errMsg;
            }
        }
        if (DEFAULT_FAILED_MSG == null) {
            DEFAULT_FAILED_MSG = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.CF_FAILED_TO_EXECUTE_ACTION, this);
        }
        return DEFAULT_FAILED_MSG;
    }

    /**
     * Returns the dialog controller that initiated this action
     *
     * @return the dialog controller this action "belongs" to.
     */
    public final DialogController getController() {
        return dialogController;
    }

    /**
     * Gets the application controller that initiated this action
     *
     * @return the application controller this action "belongs" to.
     */
    public final ApplicationController getApplicationController() {
        return applicationController;
    }

    /**
     * Gets the callable action task of this action
     *
     * @return the action task
     */
    protected ActionTask<V> getActionTask() {
        return actionTask;
    }

    /**
     * Flag - true if action is cancelled.
     */
    protected volatile boolean CANCELLED = false;

    /**
     * Check if this action is cancelled.
     * <p>
     * Synchronizes on "this".
     *
     * @return {@code true} if this action is cancelled
     */
    public synchronized boolean isCancelled() {
        return CANCELLED;
    }

    /**
     * Cancel the action.
     * <p>
     * Synchronizes on "this".
     */
    public synchronized void cancel() {
        log.info("Cancelling " + this);
        CANCELLED = true;

    }

    /**
     * Get the action setup value.
     *
     * @return action setup value
     * @see #setActionSetupValue(Object)
     */
    public Object getActionSetupValue() {
        return actionSetupValue;
    }

    /**
     * Set the action setup value. The action setup value will be used as a
     * parameter for actions that support parameters.
     * <p>
     * <b>Note:</b> Although the type of the <code>value</code> parameter is
     * <code>Object</code> one must make sure that the actual type matches the
     * expected parameter type; e.g. for the <em>Open</em> action, the parameter
     * type is <code>no.g9.client.core.controller.DialogSetupValue</code>.
     *
     * @param setupValue the action setup parameter
     */
    public void setActionSetupValue(Object setupValue) {
        actionSetupValue = setupValue;
    }

    /**
     * Returns the checkable hooks.
     *
     * @return a list of the checkable hooks, possibly an empty list
     */
    List<Checkable> getCheckableHooks() {
        return getActionHookList().getCheckableHooks();
    }

    /**
     * Returns the obtainable hooks.
     *
     * @return a list of the obtainable hooks, possibly an empty list
     */
    List<Obtainable> getObtainableHooks() {
        return getActionHookList().getObtainableHooks();
    }

    /**
     * Returns the displayable hooks.
     *
     * @return a list of the displayable hooks, possibly an empty list
     */
    List<Displayable> getDisplayableHooks() {
        return getActionHookList().getDisplayableHooks();
    }

    /**
     * @return boolean
     */
    boolean isSaveAction() {
        switch (getActionType()) {
            case SAVE: // fall through
            case INSERT: // fall through
            case UPDATE: // fall through
                return true;
            default:
                return false;
        }

    }

    private List<CurrentRoleObject> currentInstanceList;

    /**
     * @return currentInstanceList
     */
    protected List<CurrentRoleObject> getCurrentInstanceList() {
    	return currentInstanceList;
    }

    private void saveCurrentInstances() {
        currentInstanceList = new ArrayList<CurrentRoleObject>();
        String target = getActionTarget().toString();
        RoleConstant roleConst = getController().getOSConst(target);

        OSNode<?> targetNode = JVineController.getInstance(getController())
                .getOSNode(roleConst);
        currentInstanceList = targetNode.getCurrentObjectList();
        return;
    }


    private boolean hasActionType(HookType hook) throws SecurityException, NoSuchFieldException {
        	Annotation[] annotations = ActionType.class.getField(getActionType().name()).getAnnotations();
        	HookType[] val = null;
        	for (Annotation an : annotations) {
        	    if (an instanceof ActionTypeInfo) {
        	        ActionTypeInfo info = (ActionTypeInfo)an;
        	        val = info.value();
        	        for (HookType hookType : val) {
        	            if (hookType == hook) {
        	                return true;
        	            }
        	        }
        	    }
			}
        	return false;
    }

    /**
     * {@inheritDoc} Performs the action and invokes hooks. After the hook as
     * <em>finished</em> the registered hook is removed, and the connection from
     * the hook to this action is removed.
     *
     * @return the result of performing this action.
     */
    @Override
    public V call() throws Exception {
        log.info("Starting execution of " + this);
        V result = null;
        try {

        	Initializing<V> initializing = new Initializing<V>(this);
            initializing.call();

            if (hasActionType(HookType.CHECKABLE) && !isCancelled()) {
            	Checking<V> checking = new Checking<V>(this);
            	checking.call();
			}
            boolean keepCurrentInfo = isSaveAction();
            if (keepCurrentInfo && !isCancelled()) {
                saveCurrentInstances();
            }

            if (hasActionType(HookType.OBTAINABLE) && !isCancelled()) {
            	Obtaining<V> obtaining = new Obtaining<V>(this);
            	Object obtainResult = obtaining.call();
            	getActionTask().setTaskObject(obtainResult);
            }

            if (!isCancelled()) {
                Performing<V> performing = new Performing<V>(this);
                try {
                    result = performing.call();
                } catch (CancelException ce) {
                    log.info(this + " cancelled while performing.");
                    if (log.isTraceEnabled()) {
                        log.trace("Cancel trace:", ce);
                    }
                    cancel();
                }
            }

            if (hasActionType(HookType.CLEARABLE) && !isCancelled()) {
            	Clearing<V> clearing = new Clearing<V>(this);
            	clearing.call();
            }

            if (hasActionType(HookType.DISPLAYABLE) && !isCancelled()) {
            	Display<V> displaying = new Display<V>(this, getActionTask().getTaskObject());
            	displaying.call();
            }

            if (!isCancelled()) {
                Succeeding<V> succeeding = new Succeeding<V>(this);
                succeeding.call();
            }

            if (isCancelled()) {
                Cancelling<V> cancelling = new Cancelling<V>(this);
                cancelling.call();
            }
        } catch (Exception e) {
            failed(e);
        } finally {
        	Finishing<V> finishing = new Finishing<V>(this);
            finishing.call();
            cleanUpHook();
        }

        return result;
    }

    /**
     * Decouples the relation between this action and the action hook. This
     * method blocks until the action execution has finished.
     *
     * @throws InterruptedException if interrupted while waiting for the action
     *             to finish.
     */
    protected void cleanUpHook() throws InterruptedException {
        if (hasHook()) {
            if (log.isDebugEnabled()) {
                log.debug(this + " cleaning up hook");
            }
            getActionHookList().removeCurrentAction(this);

            setActionHookList(null);
        }
    }

    /**
     * Gets the action target
     *
     * @return the enumeration denoting the target
     */
    public ActionTarget getActionTarget() {
        return actionTarget;
    }

    /**
     * Sets the action target
     *
     * @param actionTarget the action target enum
     */
    void setActionTarget(ActionTarget actionTarget) {
        this.actionTarget = actionTarget;
    }

    /**
     * Returns the list of action hooks used by this action.
     *
     * @return the action hook list
     */
    protected ActionHookList<V> getActionHookList() {
        return actionHookList;
    }

    /**
     * Sets the list of action hooks to be used by this action.
     *
     * @param actionHookList the action hook list of this action
     */
    protected void setActionHookList(ActionHookList<V> actionHookList) {
        this.actionHookList = actionHookList;
    }

    /**
     * The failed hook, invoked if an exception is raised during the execution
     * of the action. This hook is invoked on a dedicated worker thread by default.
     * Use the annotation {@link ThreadInfo} to control which thread this method is
     * invoked on.
     *
     * @param cause The throwable that caused the execution of the action to
     *            abort.
     * @return Void - e.g. null
     * @exception Exception re-throws exception from hook method
     */
    Void failed(final Throwable cause) throws Exception {
        log.warn("Failed to execute " + this, cause);
        Message msg = getDefaultFailedMessage(cause);
        if (shouldInvokeHook()) {
            String methodName = "failed";
            ThreadType threadType = getThreadType(methodName, Throwable.class);
            HookMethod<Message> invocation = new HookMethod<Message>(methodName) {

                @Override
                public Message call() {
                    Message failed = null;
                    try {
                        for (ActionHook<?> hook : getActionHookList().getHooks()) {
                            failed = hook.failed(cause, failed);
                        }
                    } catch (Exception e) {
                        log.warn("Failed hook threw exception. Falling back to default error message", e);
                        failed = getDefaultFailedMessage(cause);
                    }
                    return failed;
                }
            };

            msg = hookInvoker.execute(applicationController, threadType,
                    invocation);
        }
        if (msg != null) {
            dialogController.dispatchMessage(msg);
        }
        return null;
    }

    // /////////////////////////////////////////////////////////////////
    // /
    // / UTILITIES
    // /
    // /////////////////////////////////////////////////////////////////

    /**
     * Utility method, returns the thread type the method is annotated with.
     *
     * @param methodName the declared method name
     * @param params the method parameters
     * @return the thread type for a the specified hook method
     */
    protected ThreadType getThreadType(String methodName, Class... params) {
        ThreadType type = ThreadType.GUI;
        Exception ex = null;
        try {
            ThreadInfo annotation = null;
            if (actionHookList != null) {
                annotation = actionHookList.getHooks().get(0).getClass().getMethod(methodName, params).getAnnotation(ThreadInfo.class);
            }
            if (annotation == null) {
                type = defaultAnnotations.get(methodName);
            } else {
                type = annotation.value();
            }
        } catch (SecurityException e) {
            ex = e;
        } catch (NoSuchMethodException e) {
            type = defaultAnnotations.get(methodName);
        }
        if (ex != null) {
            String msg = "Caught exception while trying to infere thread info on hook method";
            throw new G9ClientFrameworkException(msg, ex);
        }
        return type;
    }

    /**
     * Check if this action has an associated hook.
     *
     * @return true if this action has a registered hook.
     */
    protected boolean hasHook() {
        if (actionHookList == null) {
            return false;
        }
        return !actionHookList.getHooks().isEmpty();
    }

    /**
     * Test if hook should be invoked. This method tests that:
     * <ol>
     * <li>The hook mechanism is turned on.
     * <li>This action has a registered hook
     * </ol>
     * If both conditions are true, hooks should be invoked.
     *
     * @return <code>true</code> if hooks should be invoked.
     */
    protected boolean shouldInvokeHook() {
        return isInvokeHooksOn() && hasHook();
    }

    @Override
    public String toString() {
        return actionType + " " + actionTarget;
    }

    /**
     * Returns the action type of this action
     *
     * @return the action type of this action
     */
    public ActionType getActionType() {
        return actionType;
    }

    /**
     * Gets the result class
     *
     * @return the result class
     */
    protected Class<V> getResultClass() {
        return resultClass;
    }

    private Throwable unwrap(Throwable t) {
        Throwable unwrapped = unwrapToG9Base(t);
        if (unwrapped instanceof G9BaseException) {
            return unwrapped;
        }
        return t;
    }

    private Throwable unwrapToG9Base(Throwable t) {
        if (t instanceof G9BaseException) {
            return t;
        }

        if (t.getCause() != null) {
            return unwrapToG9Base(t.getCause());
        } else if (t instanceof InvocationTargetException) {
            InvocationTargetException it = (InvocationTargetException) t;
            if (it.getTargetException() != null) {
                return unwrapToG9Base(t.getCause());
            }
        }

        return t;
    }
}