/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.action;

import no.g9.message.Message;

/**
 * Defines the hook methods of a g9 action. The methods defined by this
 * interface are intended for callback from the g9 action classes. An action
 * has a sequence of states (or phases) which translate to the methods defined
 * by this interface.
 *
 * @param <T> The target type
 */
public interface Hookable<T> {

    /**
     * The initialized hook. This hook is invoked immediately before the action
     * is started.
     * <p>
     * If applicable, this hook is invoked on a dedicated worker thread by
     * default. Use the annotation {@link ThreadInfo} to control which thread
     * this method is invoked on.
     */
    @ThreadInfo(ThreadType.WORKER)
    public void initialized();

    /**
     * The performed hook, invoked after {@link #initialized()} unless the
     * action has failed or the action is canceled.
     * <p>
     * If applicable, this hook is invoked on a dedicated worker thread by
     * default. Use the annotation {@link ThreadInfo} to control which thread
     * this method is invoked on.
     *
     * @param result The result of invoking the action
     */
    @ThreadInfo(ThreadType.WORKER)
    public void performed(T result);

    /**
     * The succeeded hook, invoked after {@link #performed(Object)} unless the
     * action failed or the action was canceled.
     * <p>
     * If applicable, this hook is invoked on a dedicated worker thread by
     * default. Use the annotation {@link ThreadInfo} to control which thread
     * this method is invoked on.
     */
    @ThreadInfo(ThreadType.WORKER)
    public void succeeded();

    /**
     * The failed hook, invoked if an exception is raised during the execution
     * of the action.
     * <p>
     * The returned message will be displayed to the user (<code>null</code>
     * means no display).
     * <p>
     * If applicable, this hook is invoked on a dedicated worker thread by
     * default. Use the annotation {@link ThreadInfo} to control which thread
     * this method is invoked on.
     *
     * @param cause The throwable that caused the execution of the action to
     *            abort
     * @param message The message object from a previously executed failed hook, or <code>null</code>
     * @return The message that should be displayed in order to inform the user
     *         about the failure
     */
    @ThreadInfo(ThreadType.WORKER)
    public Message failed(Throwable cause, Message message);

    /**
     * The canceled hook, invoked if the action is canceled before or during the
     * execution of the action.
     * <p>
     * If applicable, this hook is invoked on a dedicated worker thread by
     * default. Use the annotation {@link ThreadInfo} to control which thread
     * this method is invoked on.
     */
    @ThreadInfo(ThreadType.WORKER)
    public void cancelled();

    /**
     * This finished hook, unconditionally invoked as the last method in an
     * action life cycle.
     * <p>
     * If applicable, this hook is invoked on a dedicated worker thread by
     * default. Use the annotation {@link ThreadInfo} to control which thread
     * this method is invoked on.
     */
    @ThreadInfo(ThreadType.WORKER)
    public void finished();

}