/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.controller;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import no.esito.jvine.action.ThreadManager;
import no.esito.jvine.communication.ActionMessageCompilation;
import no.esito.jvine.communication.CallBackWrapper;
import no.esito.jvine.communication.ExternalDialogCallBackValue;
import no.esito.jvine.communication.SystemMessagePipe;
import no.esito.jvine.communication.SystemMessageUtils;
import no.esito.jvine.controller.DialogInteractionBroker;
import no.esito.jvine.controller.JVineAppController;
import no.esito.jvine.controller.JVineApplicationController;
import no.esito.jvine.controller.JVineController;
import no.esito.jvine.view.AbstractApplicationView;
import no.esito.log.Logger;
import no.esito.util.ServiceLoader;
import no.g9.client.core.action.ActionFactory;
import no.g9.client.core.action.ActionHook;
import no.g9.client.core.action.ActionHookList;
import no.g9.client.core.action.EventContext;
import no.g9.client.core.communication.CommunicationError;
import no.g9.client.core.communication.G9ActionPayload;
import no.g9.client.core.communication.MessageReceiver;
import no.g9.client.core.communication.SystemMessage;
import no.g9.client.core.communication.SystemMessageContext;
import no.g9.client.core.view.ApplicationView;
import no.g9.client.core.view.DialogView;
import no.g9.client.core.view.ViewModel;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageReply;
import no.g9.message.MessageReplyType;
import no.g9.message.MessageSystem;
import no.g9.service.G9Spring;
import no.g9.service.JGrapeService;
import no.g9.support.ActionType;
import no.g9.support.ClientContext;
import no.g9.support.action.ActionTarget;

import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.util.StringUtils;

/**
 * The application controller. Controls various aspects of the application.
 */
public abstract class ApplicationController extends JVineApplicationController implements SystemMessagePipe {

    /**
     * The client context
     */
    private final ClientContext clientContext;

    /**
     * The service proxy.
     */
    private JGrapeService serviceProxy;

    /** The application name */
    private final String applicationName;

    private ApplicationView applicationView;

    private final ActionFactory actionFactory = new ActionFactory();

    /** The logger */
    private final Logger log = Logger.getLogger(ApplicationController.class);

    /** Controls the dialog creation and cache of open instances **/
    private DialogInstanceManager dialogInstanceManager = null;

    /** A map of the pending call backs, used on return from external dialogs. */
    private Map<String, CallBackWrapper> pendingCallBacks = new HashMap<String, CallBackWrapper>();

    /**
     * Getter
     * @return the DialogInstanceManager
     */
    protected DialogInstanceManager getDialogInstanceManager(){
    	return dialogInstanceManager;
    }

    /**
     * Constructs a new application controller
     *
     * @param applicationName
     *            the name of the application controller.
     */
    public ApplicationController(String applicationName) {
        super();
        clientContext = new ClientContext();
        clientContext.setApplicationName(applicationName);
        this.applicationName = applicationName;
        this.dialogInstanceManager = new DialogInstanceManager(this);
    }

    /**
     * Get the dialog constant for the specified dialog.
     *
     * @param <T>
     *            The expected dialog controller type.
     * @param dialogInternalName
     *            the dialog name as it is defined in the bean id.
     * @return the dialog constant.
     * @throws ClassCastException
     *             if the returned type does not match the expected type.
     */
    @SuppressWarnings("unchecked")
    public final synchronized <T extends DialogConstant> T getDialogConst(String dialogInternalName) {
        return (T) dialogInstanceManager.getDialogConstant(dialogInternalName);
    }

    /**
     * Gets the current instance number, i.e. the dialog on top of the stack.
     *
     * @param dialogConstant
     *            the dialog constant
     * @return the current instance number, <code>null</code> if no active dialog found for this
     *         dialog constant
     */
    public final DialogInstance getCurrentDialogInstance(DialogConstant dialogConstant) {
        return dialogInstanceManager.getCurrentDialogInstance(dialogConstant);
    }

    /**
     * Sets the active dialog instance.
     *
     * @param dialogInstanceKey
     *            the key identifying the dialog instance
     * @return the dialog controller identified by the dialogInstanceKey,
     *         <code>null</code> if no instance found with the given id
     */
    public final DialogController setActiveDialogInstance(DialogInstance dialogInstanceKey) {
        return dialogInstanceManager.setActiveDialogInstance(dialogInstanceKey);
    }

    /**
     * Removes the dialog instance identified by the instance key from the
     * stack.
     *
     * <strong>WARNING:</strong> Although this method is public, it should not be
     * treated as part of the public API, as it might change in incompatible
     * ways between releases (even patches).
     *
     * <p>
     * Use {@link ApplicationController#closeDialogInstance(DialogInstance)} to close a dialog.
     *
     * @param dialogInstance
     *            the dialog instance key of the dialog to remove.
     */
    public void removeDialogInstance(DialogInstance dialogInstance) {
        dialogInstanceManager.removeDialogInstance(dialogInstance);
    }

    /**
     * Close dialog instance identified by the <code>dialogInstance</code> key.
     *
     * @param dialogInstance the dialog instance to close
     * @return true, if successful
     */
    public synchronized boolean closeDialogInstance(DialogInstance dialogInstance) {
        DialogController dialogController = dialogInstanceManager.getDialogController(dialogInstance);
        boolean checkClose = JVineController.getInstance(dialogController).checkClose();
        if (!checkClose) {
            String msgID = CRuntimeMsg.CC_CHECK_OPEN_MSG;
            String dialogTitle = getApplicationView().getDialogTitle(dialogInstance);
            MessageReply reply = dialogController.dispatchMessage(msgID,
                    dialogTitle);
            checkClose = MessageReplyType.REPLY_OK.equals(reply);
        }
        if (checkClose) {
            AbstractApplicationView applicationView = (AbstractApplicationView) getApplicationView();
            applicationView.removeInstance(dialogInstance);
            return true;
        }
        return false;
    }


    /**
     * Checks for available dialog instance.
     *
     * @param dialogConstant
     *            the dialog constant
     * @return true if there is one or more available dialog instances for the
     *         given dialog
     */
    public boolean hasAvailableDialogInstance(DialogConstant dialogConstant) {
        return dialogInstanceManager.hasAvailableDialogInstance(dialogConstant);
    }

    /**
     * Gets a list of active dialog instances.
     *
     * @param dialogConstant
     *            the dialog constant identifying the dialog
     * @return the list of active dialog instances, an empty list if no active
     *         dialogs of this type
     */
    public List<DialogInstance> getActiveDialogInstances(DialogConstant dialogConstant) {
        return dialogInstanceManager.getActiveDialogInstances(dialogConstant);
    }

    /**
     * Get a list of open dialog instances, ordered by when the instance was created (oldest first).
     * @param dialogConstant constant identifying the dialog
     * @return list of open dialog instances or empty list if no instances is opened
     */
    public List<DialogInstance> getOpenDialogInstanceList(DialogConstant dialogConstant) {
        return dialogInstanceManager.getOpenDialogList(dialogConstant);
    }

    /**
     * Get a list of dialog controllers for all open dialogs.
     *
     * @return list of all open dialog controllers
     */
    public List<DialogController> getOpenDialogs() {
    	return dialogInstanceManager.getOpenDialogList();
    }

    /**
     * Registers the controller.<br>
     * <strong>Warning: this method should not be used by developers.</strong>
     *
     * @param instanceNumber instance number
     * @param newController controller
     */
    public void registerController(int instanceNumber, DialogController newController) {
        dialogInstanceManager.registerController(instanceNumber, newController);
    }

    /**
     * Get the current dialog controller for the specified dialog.
     *
     * @param <T>
     *            The expected dialog controller type
     * @param dialogConstant
     *            the dialog constant denoting the dialog controller.
     * @return the cached or a new dialog controller
     * @throws ClassCastException
     *             if the returned type does not match the expected dialog
     *             controller type.
     */
    @SuppressWarnings("unchecked")
    public final synchronized <T extends DialogController> T getDialogController(DialogConstant dialogConstant) {
        if (log.isTraceEnabled()) {
            log.trace("Getting current dialog controller for: " + dialogConstant);
        }
        return (T) dialogInstanceManager.getDialogController(dialogConstant);
    }

    /**
     * Gets the dialog controller identified by the dialog instance key. This
     * method does nothing with the internal stack.
     *
     * <strong>WARNING:</strong> Although this method is public, it should not
     * be treated as part of the public API, as it might change in incompatible
     * ways between releases (even patches).
     *
     * @param <T>
     *            the generic type of the controller.
     * @param dialogInstance
     *            the dialog instance
     * @return the dialog controller, <code>null</code> if not found.
     */
    @SuppressWarnings("unchecked")
    public final synchronized <T extends DialogController> T getDialogController(DialogInstance dialogInstance) {
        if (log.isTraceEnabled()) {
            log.trace("Getting dialog controller for: " + dialogInstance);
        }
        return (T) dialogInstanceManager.getDialogController(dialogInstance);
    }


    /**
     * Creates a new dialog instance and pushes it on top of the stack.
     *
     * <strong>WARNING:</strong> Although this method is public, it should not
     * be treated as part of the public API, as it might change in incompatible
     * ways between releases (even patches).
     *
     * @param dialogConstant
     *            the dialog constant
     * @param instanceNumber
     *            the instance number, set to -1 to get the next available
     *            instanceNumber
     * @return the dialog instance
     * @throws InstanceNumberOutOfBoundsException
     *             when no more dialogs of this dialog can be created.
     * @throws IllegalStateException
     *             when instanceNumber refers to an instance that already exists
     */
    public DialogInstance createDialogInstance(DialogConstant dialogConstant, int instanceNumber) {
        return dialogInstanceManager.createDialogInstance(dialogConstant, instanceNumber);
    }

    /**
     * Gets the application name
     *
     * @return the application name
     */
    public String getApplicationName() {
        return applicationName;
    }

    /**
     * Gets the application ID.
     *
     * The application ID is the same as the application name,
     * with a lower case first letter.
     *
     * @return the application ID
     */
    public String getApplicationId() {
        return StringUtils.uncapitalize(getApplicationName());
    }

    /**
     * Returns the client context
     *
     * @return the client context
     */
    public ClientContext getClientContext() {
        return clientContext;
    }

    /**
     * Returns the jgrape service
     *
     * @return the jgrape service proxy
     */
    public synchronized JGrapeService getServiceProxy() {
        if (log.isTraceEnabled()) {
            log.trace("Getting JGrapeService");
        }
        if (serviceProxy == null) {
            if (log.isDebugEnabled()) {
                log.debug("Creating a new instance of JGrapeService");
            }
            serviceProxy = ServiceLoader.getService(JGrapeService.class);
        }
        return serviceProxy;
    }

    /**
     * Gets the application's view.
     *
     * @return the application view.
     */
    public final ApplicationView getApplicationView() {
        return applicationView;
    }

    /**
     * Gets the specified dialog view.
     *
     * @param <T>
     *            the dialog view type
     * @param dialogConstant
     *            the (generated) constant denoting the dialog.
     * @return the dialog view.
     */
    @SuppressWarnings("unchecked")
    public final <T extends DialogView> T getDialogView(DialogConstant dialogConstant) {
        return (T) getApplicationView().getDialogView(dialogConstant);
    }

    /**
     * Sets the application's view.
     *
     * @param applicationView
     *            the application view.
     */
    public final void setApplicationView(ApplicationView applicationView) {
        this.applicationView = applicationView;
    }

    /**
     * Performs the specified action.
     *
     * @param action the action to perform.
     * @param target the target of the action.
     * @param flag a general purpose flag
     */
    public final void performAction(ActionType action, DialogConstant target, boolean flag) {

        if (target instanceof ExternalDialogConstant) {
            performExternalAction(action, target);
        } else {
            handleCloseAction(action, target);
            getApplicationView().performAction(action, target, flag);
        }
    }

    /**
     * Performs the specified action. The action was initiated from an external source.
     *
     * @param action the action to perform.
     * @param target the target of the action.
     * @param flag a general purpose flag
     */
    public final void performActionFromExternal(ActionType action, DialogConstant target, boolean flag) {

        if (target instanceof ExternalDialogConstant) {
            performExternalAction(action, target);
        } else {
            handleCloseAction(action, target);
            ((AbstractApplicationView)getApplicationView()).performAction(action, target, flag, true);
        }
    }

    private void handleCloseAction(ActionType action, DialogConstant target) {
        if (action == ActionType.CLOSE) {
            DialogController dialogController = getDialogController(target);
            JVineController.getInstance(dialogController).closeDialogController();
            getApplicationView().performAction(ActionType.CLEAROBJECT, target, false);
        }
    }

    private void performExternalAction(ActionType action, DialogConstant target) {
        G9ActionPayload actionPayload = new G9ActionPayload(action, "");
        if (actionPayload.isValid) {
            SystemMessage message = SystemMessageUtils.createSystemMessage("", target, actionPayload.code());
            ExternalDialogConstant externalTarget=(ExternalDialogConstant)target;

            ActionMessageCompilation actionMessageCompilation= new ActionMessageCompilation(message, actionPayload, this);
            externalTarget.setReactor(actionMessageCompilation);
        }
    }

    /**
     * Performs the specified action.
     *
     * @param action
     *            the action to perform.
     * @param target
     *            the target of the action.
     */
    public final void performAction(final ActionType action, final DialogConstant target) {
        performAction(action, target, false);
    }

    /**
     * Register the specified action hook
     *
     * @param actionTarget
     *            the action target
     * @param actionType
     *            the action type
     * @param hook
     *            the action hook that should be invoked when the action runs.
     */
    public final void registerHook(ActionTarget actionTarget, ActionType actionType, ActionHook<?> hook) {
        actionFactory.registerHook(actionTarget, actionType, hook);
    }

    /**
     * Get the specified action hooks.
     *
     * @param actionTarget the action target
     * @param actionType the action type
     * @return the list of hooks registered with the specified action and target.
     */
    protected final ActionHookList<?> getActionHookList(ActionTarget actionTarget, ActionType actionType) {
        return actionFactory.getActionHookList(actionTarget, actionType);
    }

    /**
     * Get the ViewModel for the given dialog.
     *
     * @param dialog
     *            the dialog which "owns" the ViewModel.
     * @return the ViewModel for the given dialog.
     * @see ApplicationView#getViewModel(DialogConstant)
     */
    public final ViewModel getViewModel(DialogConstant dialog) {
        return getApplicationView().getViewModel(dialog);
    }

    /**
     * Get the ViewModel for the given dialog instance.
     *
     * @param instance the dialog instance which "owns" the ViewModel
     * @return the ViewModel for the given dialog instance
     */
    final ViewModel getViewModel(DialogInstance instance) {
        AbstractApplicationView appView = (AbstractApplicationView) getApplicationView();
        return appView.getViewModel(instance);
    }

    /**
     * Invokes the specified task on the gui thread. If the invoking thread is a
     * worker thread, the runnable is put on the GUI executing queue. Otherwise
     * (the invoking thread <em>is</em> a GUI thread), the Runnable's
     * <code>run</code> method is simply invoked as is.
     *
     * @param task
     *            the task to perform.
     * @throws InvocationTargetException
     *             exceptions that occurred during invocation of the runnable
     *             are wrapped and re-thrown.
     */
    public final void invokeOnGui(Runnable task)
            throws InvocationTargetException {
        if (ThreadManager.isWorkerThread()) {
            JVineAppController.getInstance(this).getActionQueue().perform(task);
        } else {
            task.run();
        }
    }

    /**
     * Return the dialog constant of the application window.
     *
     * @param <T>
     *            the actual type of the dialog constant.
     * @return the dialog constant representing the application window.
     */
    public abstract <T extends DialogConstant> T getApplicationDialogConstant();

    /**
     * Forward a message towards the receiver.
     * If the receiver is this application, locate a handler to receive the message.
     *
     * @param codedMessage The coded message
     */
    public void forward(String codedMessage) {
        SystemMessage decodedMessage = new SystemMessage(codedMessage);
        forward(decodedMessage);

    }

    /**
     * Forward a message towards the receiver.
     * If the receiver is this application, locate a handler to receive the message.
     *
     * @param decodedMessage The decoded message
     */
    @Override
    public void forward(SystemMessage decodedMessage) {
        if (getApplicationName().equalsIgnoreCase(decodedMessage.receiver)) {

            // First, try user defined ports
            Map<String, MessageReceiver> handlers;
            try {
                handlers = G9Spring.getBean("gva.ports.userdefined");
            } catch (Exception e) {
                handlers = new HashMap<String, MessageReceiver>();
            }
            MessageReceiver handler = handlers.get(decodedMessage.port);
            if (handler != null) {
                final ApplicationController thisApplicationController = this;
                handler.receive(decodedMessage, new SystemMessageContext(){
                    @Override
                    public ApplicationController getApplicationController() {
                        return thisApplicationController;
                    }
                });
                return;
            }

            // Second, try the call back port
            if (SystemMessage.CALLBACK_PORT.equals(decodedMessage.port)) {
                dispatchCallBack(decodedMessage);
                return;
            }

            // Third, try the error handler port
            if (SystemMessage.ERROR_PORT.equals(decodedMessage.port)) {
                CommunicationError communicationError = new CommunicationError(decodedMessage.payload);
                handleCommunicationError(communicationError);
                return;
            }

            // Lastly, try the dialogs as a port
            try {
                DialogConstant dialogConstant = getDialogConst(decodedMessage.port);
                if (dialogConstant != null) {
                    forwardToDialog(decodedMessage, dialogConstant);
                    return;
                }
            } catch (NoSuchBeanDefinitionException e) {
                // No worry, just throw away the message
            }
        } else {
            forwardToExternal(decodedMessage);
            return;
        }
        log.info("No handler for message \"" + decodedMessage.code() + "\". Ignoring it.");
    }

    /**
     * This method is called when there was a problem sending a message to an external application.
     * Override this method in a subclass if you want to do more than logging the error.
     *
     * @param commErr an object containing error information
     */
    public void handleCommunicationError(final CommunicationError commErr) {
        Object[] args = { commErr.failingMessage.code(), commErr.statusCode, commErr.description };
        Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.COM_SENDING_FAILED, args);
        MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
        forwardToExternal(SystemMessage.ALERT_TEMPLATE.payload(msg.getMessageText()));
    }

    /**
     * Add the given call back to the map of pending call backs for this application.
     * The map key for the call back is returned.
     *
     * Note: For internal use only!
     *
     * @param actionInvoker the dialog which invoked the action
     * @param callee the dialog which is the action target
     * @param callBack the call back
     * @return the map key
     */
    public String addCallBack(DialogController actionInvoker, DialogConstant callee, CallBack callBack) {
        String key = actionInvoker.getDialogConstant().getInternalName() + "_" + actionInvoker.getDialogInstanceNumber();
        pendingCallBacks.put(key, new CallBackWrapper(actionInvoker.getDialogConstant(), callee, callBack));
        return key;
    }

    private void dispatchCallBack(SystemMessage callBackMessage) {
        SystemMessage tmpMsg = new SystemMessage(callBackMessage.payload);
        String key = tmpMsg.receiver;
        CallBackWrapper wrapper = pendingCallBacks.remove(key);
        if (wrapper != null) {
            DialogController caller = getDialogController(wrapper.actionInvoker);
            DialogConstant callee = wrapper.callee;
            ExternalDialogCallBackValue ecbValue = ExternalDialogCallBackValue.fromJson(tmpMsg.payload);
            DialogCallBackValue callBackValue = ecbValue.asDialogCallBackValue();
            DialogConstant applicationDialogConstant = getApplicationDialogConstant();
			callBackValue.setSource(getDialogController(applicationDialogConstant));
            DialogInteractionBroker broker = DialogInteractionBroker.getInstance();
            DialogCallBackValue returnValue = broker.doCallback(callBackValue, caller, callee);
            if (wrapper.callBack != null) {
                wrapper.callBack.callBack(returnValue);
            }
        }
    }

    /**
     * Get a CallBack object which will forward the call back value to the given
     * receiver/port.
     *
     * @param callBackMessage the return port message
     * @return the CallBack object
     */
    public CallBack getExternalCallBack(final SystemMessage callBackMessage) {
        return new CallBack() {

            @Override
            public void callBack(DialogCallBackValue callBackValue) {
                String cbPayload = ExternalDialogCallBackValue.toJson(new ExternalDialogCallBackValue(callBackValue));
                SystemMessage forwardMessage = callBackMessage.payload(callBackMessage.payload + "//" + cbPayload);
                if (log.isTraceEnabled()) {
                    log.trace("External call back, message: " + forwardMessage);
                }
                forwardToExternal(forwardMessage);
            }

        };
    }

    /**
     * Try forwarding a message to a dialog.
     *
     * @param decodedMessage the message to forward
     * @param dialogConstant the receiving dialog
     */
    protected void forwardToDialog(SystemMessage decodedMessage, DialogConstant dialogConstant) {

        EventContext eventMethod = new EventContext(SystemMessage.MESSAGE_ACTION_PERFORMER, "", decodedMessage);

        DialogConstant applicationDialogConstant = getApplicationDialogConstant();
		DialogController applicationDialog = getDialogController(applicationDialogConstant);
        JVineController.getInstance(applicationDialog).dispatch(eventMethod);
    }

    /**
     * Forward a message towards an external receiver. The next link in the chain is the applicationDialog.
     *
     * @param decodedMessage the message to forward
     */
    protected void forwardToExternal(SystemMessage decodedMessage) {
        ((AbstractApplicationView)getApplicationView()).forward(decodedMessage);
    }

    /**
     * Get the current application controller for this thread.
     *
     * @return the current application controller
     */
    public static final ApplicationController getCurrentApplicationController() {
    	return JVineApplicationController.getCurrentApplicationController();
    }

}
