/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.controller;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

import no.esito.jvine.controller.DialogInstanceKey;
import no.esito.jvine.controller.DialogInteractionBroker;
import no.esito.jvine.controller.JVineController;
import no.esito.jvine.controller.OSNode;
import no.esito.jvine.controller.StateController;
import no.esito.jvine.view.AbstractApplicationView;
import no.esito.log.Logger;
import no.g9.client.core.action.ActionHook;
import no.g9.client.core.action.ActionHookList;
import no.g9.client.core.action.CheckType;
import no.g9.client.core.validator.ValidateContext;
import no.g9.client.core.validator.ValidationPolicy.Policy;
import no.g9.client.core.validator.ValidationResult;
import no.g9.client.core.view.DialogView;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.ViewModel;
import no.g9.client.core.view.table.TableModel;
import no.g9.client.spreadsheet.SpreadsheetExporter;
import no.g9.exception.G9ClientException;
import no.g9.message.DispatcherContext;
import no.g9.message.Message;
import no.g9.message.MessageDispatcher;
import no.g9.message.MessageReply;
import no.g9.message.MessageSystem;
import no.g9.os.AttributeConstant;
import no.g9.os.OSRole;
import no.g9.os.RoleConstant;
import no.g9.support.ActionType;
import no.g9.support.Visitor;
import no.g9.support.action.ActionTarget;

/**
 * The abstract dialog control.
 */
public abstract class DialogController extends StateController {

    private static final Logger log = Logger.getLogger(DialogController.class);

    /** The application controller */
    private ApplicationController applicationController;

    /** The callback */
    private CallBack callBack;

    /** The setup value */
    private DialogSetupValue<?> setupValue;

    /** The spreadsheet exporter */
    private SpreadsheetExporter spreadsheetExporter;

    private final JVineController jController;

    /** The dialog name */
    public final String dialogName;


    /**
     * Constructs a new DialogControl with the specified dialog name.
     *
     * @param dialogName the name of the dialog.
     * @param applicationController the application controller
     */
    public DialogController(final String dialogName,
            final ApplicationController applicationController) {
        super(dialogName);
        if (dialogName == null) {
            throw new IllegalArgumentException("Dialog name can not be null");
        }
        this.dialogName = dialogName;
        this.applicationController = applicationController;
        jController = JVineController.getInstance(this);
    }

    /**
     * Gets the changed field attributes for the specified role.
     *
     * @param role the role in question
     * @return a collection of the roles changed attributes.
     */
    public final Collection<AttributeConstant> getChangedAttributes(
            RoleConstant role) {
        return getViewModel().getChangedAttributes(role);
    }

    /**
     * Sets the specified field value.
     *
     * @param attribute constant denoting the field.
     * @param value the value to display.
     */
    public final void setFieldValue(AttributeConstant attribute, Object value) {
        for (DialogObjectConstant doc : getViewModel().getAttributeFields(
                attribute)) {
            setFieldValue(doc, value);
        }
    }

    /**
     * Sets the specified field value and changed status.
     *
     * @param attribute constant denoting the field.
     * @param value the value to display.
     * @param changed the changed status of the field.
     */
    public final void setFieldValue(AttributeConstant attribute, Object value,
            boolean changed) {
        for (DialogObjectConstant doc : getViewModel().getAttributeFields(
                attribute)) {
            setFieldValue(doc, value, changed);
        }
    }

    /**
     * Set the application controller for this dialog
     *
     * @param applicationController the application controller
     */
    final void setApplicationController(
            ApplicationController applicationController) {
        this.applicationController = applicationController;
        init();
    }

    /**
     * Invoked after the dialog controller has been fully initialized.
     */
    public void init() {
        // empty default implementation.
    }

    /**
     * Invoked each time the dialog is opened. The setupValue is initialized by
     * hooks.
     *
     * @param setupValue the setup value.
     */
    public void setup(DialogSetupValue<?> setupValue) {
        // empty default implementation
    }

    /**
     * Gets the dialog view class of this dialog.
     *
     * @param <T> The dialog view type
     * @return the dialog view.
     */
    @SuppressWarnings("unchecked")
    public final synchronized <T extends DialogView> T getDialogView() {
        return (T) JVineController.getInstance(this).getDialogView();
    }

    /**
     * Get the ViewModel for the this dialog.
     *
     * @return the ViewModel for this dialog.
     */
    @Override
    public final synchronized ViewModel getViewModel() {
        DialogInstance instance = new DialogInstanceKey(getDialogConstant(), getDialogInstanceNumber());
        return getApplicationController().getViewModel(instance);
    }

    /**
     * Gets the object selection role.
     *
     * @param <T> The expected return type, e.g. OSRole&lt;?&gt;.
     * @param role the constant denoting the role.
     * @return the OSRole of said role.
     * @throws ClassCastException if the returned type does not match the
     *             expected type.
     * @throws IllegalArgumentException if the specified role is not found among
     *             this dialog's object selection.
     */
    @SuppressWarnings("unchecked")
    public final <T extends OSRole<?>> T getOSRole(RoleConstant role) {
        return (T) jController.getOSNode(role);
    }

    /**
     * Gets the specified displayed field value (converted from to model).
     *
     * @param <T> The expected field value type
     * @param dataItem the constant denoting the view data item.
     * @return the displayed field value
     * @throws ClassCastException if the actual field value type is not
     *             assignable to the expected field value type.
     */
    @SuppressWarnings("unchecked")
    public final <T> T getFieldValue(DialogObjectConstant dataItem) {
        return (T) getViewModel().getFieldValue(dataItem);
    }

    /**
     * Gets the specified displayed attribute value.
     *
     * @param <T> The expected field value type
     * @param attribute the constant denoting the attribute.
     * @return the displayed field value
     * @throws IllegalArgumentException if no data item corresponds to the
     *             specified attribute.
     * @throws ClassCastException if the actual field value type is not
     *             assignable to the expected field value type.
     */
    public final <T> T getFieldValue(AttributeConstant attribute) {
        for (DialogObjectConstant doc : getViewModel().getAttributeFields(
                attribute)) {
            return this.<T> getFieldValue(doc);
        }

        String msg = "Missing dialog object for attribute " + attribute;
        throw new IllegalArgumentException(msg);
    }

    /**
     * Set the specified field value.
     *
     * @param field dialog object constant denoting the view's data item.
     * @param value the value to display.
     */
    public final void setFieldValue(DialogObjectConstant field, Object value) {
        getViewModel().setFieldValue(field, value);
    }

    /**
     * Set the specified field value and field changed status. If the
     * <code>changed</code> parameter is <code>true</code> the field will be
     * considered <em>changed</em> (as if changed by a user). Otherwise the
     * field will be considered <em>unchanged</em>.
     *
     * @param field the dialog object constant denoting the view's data item.
     * @param fieldValue the value to display
     * @param changed if <code>true</code>
     */
    public final void setFieldValue(DialogObjectConstant field,
            Object fieldValue, boolean changed) {
        getViewModel().setFieldValue(field, fieldValue);
        getViewModel().setChanged(field, changed);
    }

    @Override
    public String toString() {
        return "DialogController: " + dialogName;
    }

    /**
     * Register the specified action hook.
     *
     * @param actionTarget the action target
     * @param actionType the action type
     * @param hook an action hook that should be invoked when the action runs
     */
    public final void registerHook(ActionTarget actionTarget, ActionType actionType, ActionHook<?> hook) {
        jController.getActionFactory().registerHook(actionTarget, actionType, hook);
    }

    /**
     * Get the list of action hooks for the specified action target and type.
     *
     * @param actionTarget the target of the action
     * @param actionType the action type
     * @return the action hooks registered for the specified action target and type
     */
    public ActionHookList<?> getActionHookList(ActionTarget actionTarget, ActionType actionType) {
        return jController.getActionFactory().getActionHookList(actionTarget, actionType);
    }

    /**
     * Clears the specified role and all children.
     *
     * @param role the role to clear.
     * @param intercept if true, clear intercept will be invoked during clear
     */
    public final void clear(RoleConstant role, boolean intercept) {
        if (log.isDebugEnabled()) {
            log.debug(this + " clearing role: " + role);
        }
        Collection<OSNode<?>> cleared = jController.getOSNode(role).clear(
                intercept);
        Collection<OSNode<?>> includeNodes = jController
                .getOneRelatedChildren(role);
        includeNodes.add(jController.getOSNode(role));
        jController.reportToView(cleared, includeNodes, null);
    }

    /**
     * Clears the current instance of the specified role and propagates the
     * change to children.
     *
     * @param role the role to clear.
     * @param intercept if true, clear intecept will be invoked during clear.
     */
    public final void clearCurrent(RoleConstant role, boolean intercept) {
        if (log.isDebugEnabled()) {
            log.debug(this + " clearing current of role: " + role);
            Collection<OSNode<?>> cleared = jController.getOSNode(role)
                    .clearCurrent(intercept);
            Collection<OSNode<?>> includeNodes = jController
                    .getOneRelatedChildren(role);
            includeNodes.add(jController.getOSNode(role));
            jController.reportToView(cleared, includeNodes, null);
        }
    }

    /**
     * Clears the specified role and children but keeps the data belonging to
     * the key fields. All fields belonging to the specified role except the key
     * fields are cleared. Child roles are completely cleared.
     *
     * @param role the role to clear.
     */
    public final void clearKeepKeys(RoleConstant role) {
        if (log.isDebugEnabled()) {
            log.debug(this + " clear keep keys for role: " + role);
        }
        Collection<OSNode<?>> cleared = jController.getOSNode(role)
                .clearKeepKeys();
        Collection<OSNode<?>> oneRelatedChildren = jController
                .getOneRelatedChildren(role);
        jController.reportToView(cleared, oneRelatedChildren, null);
    }

    /**
     * Get the application controller for this dialog
     *
     * @return the application controller
     */
    @Override
    public final ApplicationController getApplicationController() {
        return applicationController;
    }

    /**
     * Set the call back used to perform dialog interaction between dialogs.
     *
     * @param callBack the call back
     */
    public final void setCallBack(CallBack callBack) {
        this.callBack = callBack;
    }

    /**
	 * @return the current setupValue for the dialog
	 */
	public DialogSetupValue<?> getSetupValue() {
		return setupValue;
	}

	/**
	 * Set a new setupValue for the dialog.
	 *
	 * @param setupValue the setupValue to set
	 */
	public void setSetupValue(DialogSetupValue<?> setupValue) {
		this.setupValue = setupValue;
	}

	/**
     * Test if any fields of the specified role is changed in the gui.
     *
     * @param role the role to test.
     * @return <code>true</code> if at least one field value of the specified
     *         role is dirty.
     * @see ViewModel#getChangedAttributes(RoleConstant)
     */
    public final boolean isChanged(RoleConstant role) {
        ViewModel viewModel = getViewModel();
        boolean isChanged = !viewModel.getChangedAttributes(role).isEmpty();
        if (log.isTraceEnabled()) {
            log.trace("Changed attributes of " + role + " are: "
                    + viewModel.getChangedAttributes(role));
        }
        return isChanged;
    }

    /**
     * Test if there are any current conversion errors for the specified role
     *
     * @param role the role to test for
     * @return <code>true</code> if any fields in the specified role has a
     *         conversion error.
     */
    public final boolean hasConversionErrors(RoleConstant role) {
        return jController.getOSNode(role).hasConversionErrors();
    }

    /**
     * Get a map of dialog objects and conversion error messages for the
     * specified role.
     *
     * @param role the role to get messages for
     * @return all messages relating to conversion errors on the role
     */
    public final Map<DialogObjectConstant, Collection<?>> getConversionMessages(
            RoleConstant role) {
        return jController.getOSNode(role).getConversionContextMessages();
    }

    /**
     * Returns the name of the object selection.
     *
     * @return the object selection name
     */
    public abstract String getObjectSelectionName();

    /**
     * Returns the list of all role constants for the object selection
     * used in the dialog.
     *
     * @return all role constants
     */
    public abstract List<RoleConstant> getRoleConstants();

    /**
     * Returns the constant denoting the role. The role must be part of this
     * dialog's object selection.
     *
     * @param <T> The expected role constant type.
     * @param role the name of the role
     * @return the role constant
     * @throws ClassCastException if the returned type does not match the
     *             expected type.
     */
    public abstract <T extends RoleConstant> T getOSConst(String role);

    /**
     * Perform the call back to the invoking dialog.
     *
     * @param value an object to transfer back to the invoking dialog
     */
    public final void doCallBack(Object value) {
        this.doCallBack(value, null);
    }

    /**
     * Perform the call back to the invoking dialog.
     *
     * @param callBackValue the call back value to return to the invoking
     *            dialog's call back.
     */
    public final void doCallBack(DialogCallBackValue callBackValue) {
        JVineController jCtrl = JVineController.getInstance(this);

        DialogController caller = jCtrl.getCaller();
        DialogInteractionBroker broker = DialogInteractionBroker.getInstance();
        DialogCallBackValue returnValue = broker.doCallback(callBackValue, caller, this.getDialogConstant());

        if (callBack != null) {
            callBack.callBack(returnValue);
        }
        jCtrl.setCaller(null);
        callBack = null;
    }

    /**
     * Perform the call back to the invoking dialog
     *
     * @param value an object to transfer back to the invoking dialog
     * @param opCode the opCode to transfer back to the invoking dialog
     */
    public final void doCallBack(Object value, Integer opCode) {
        DialogCallBackValue callBackValue = new DialogCallBackValue(this,
                value, opCode);
        doCallBack(callBackValue);
    }

    /**
     * Registers an interceptor. The interceptorType defines which action type
     * the interceptor applies to, typically FIND, SAVE or DELETE.
     *
     * @param checkType the action type this interceptor intercepts.
     * @param interceptor the interceptor to register.
     */
    public final void registerInterceptor(CheckType checkType,
            Interceptor interceptor) {
        OSNode<?> node = jController.getOSNode(interceptor.getRole());
        node.addInterceptor(checkType, interceptor);
    }

    /**
     * Perform the specified check change on the specified role.
     *
     * @param role the target role of the check change.
     * @param checkType the check type to perform
     * @return <code>true</code> if the specified role and all child roles are
     *         unchanged.
     */
    public final boolean checkChange(RoleConstant role, CheckType checkType) {
        OSNode<?> node = jController.getOSNode(role);
        return node.checkChanged(checkType, role).getCheckResult();
    }

    /**
     * Create a new instance of the specified root and update it with the view's
     * field values.
     *
     * @param <T> The domain object type.
     * @param role the role to peek at.
     * @return an new instance of the specified role with values from the view.
     * @throws G9ClientException if either convert of validate failed during
     *             the update of the domain object from the view.
     * @throws ClassCastException if the returned type does not match the
     *             expected type.
     */
    @SuppressWarnings("unchecked")
    public final <T> T peek(RoleConstant role) {
        Object instance = jController.getOSNode(role).peek();
        jController.checkValidationAndConvert(true);
        return (T) instance;
    }

    /**
     * Displays the fields of the specified instance.
     *
     * @param role the role of the instance to display.
     * @param instance the instance to display.
     */
	public final void display(RoleConstant role, Object instance) {
		display(role, instance, false);
	}

    /**
     * Get all the displayed instances of the specified role.
     *
     * @param <T> the expected domain type of the role
     * @param role the constant denoting the role
     * @return a collection of the displayed instances.
     * @throws ClassCastException if the actual domain type does not match the
     *             expected domain type of the specified role.
     */
    public final <T> Collection<T> getAllInstances(RoleConstant role) {
        OSNode<T> node = jController.getOSNode(role);
        return jController.getAllInstances(node);
    }

    /**
     * Displays all fields of the specified instance.
     *
     * @param role the role of the instance to display.
     * @param instance the instance to display.
     * @param changed if <code>true</code> the view's corresponding data items
     *            will be considered <em>changed</em>, otherwise
     *            <em>unchanged</em>.
     */
    public final void display(RoleConstant role, Object instance, boolean changed) {
        Collection<DialogObjectConstant> roleFields = getViewModel().getRoleFields(role);
        for (DialogObjectConstant field : roleFields) {
            AttributeConstant attribute = field.getAttribute();
            Object value = getOSRole(role).getValue(instance, attribute);
            setFieldValue(field, value, changed);
        }
    }

    /**
     * Sets the current instance and recursively updates the view's model of
     * instances to display.
     *
     * @param role the role of the new current instance
     * @param instance the instance to set as the new current instance.
     */
    public final void setCurrentInstance(RoleConstant role, Object instance) {
        OSNode<?> node = jController.getOSNode(role);
        Collection<OSNode<?>> changedNodes = node.setCurrentInstance(instance);
        Collection<OSNode<?>> includeNodes = new ArrayList<OSNode<?>>();
        includeNodes.add(node);
        jController.reportToView(changedNodes, includeNodes, null);
    }

    /**
     * Sets the current instances for a "many" root role, and recursively updates
     * the view's model of instances to display.
     *
     * @param <T> the expected domain type
     * @param role the root role for the instances
     * @param instances the collection of instances to set
     */
    public final <T> void setCurrentInstances(RoleConstant role, Collection<T> instances) {
        OSNode<T> node = jController.getOSNode(role);
        Collection<OSNode<?>> changedNodes = node.setCurrentInstances(instances);
        Collection<OSNode<?>> includeNodes = new ArrayList<OSNode<?>>();
        includeNodes.add(node);
        jController.reportToView(changedNodes, includeNodes, null);
        /* Sorting and filtering.*/
        TableModel<ListRow> tableModel = getDialogView().getViewModel().getTableModel(role);
        if (tableModel != null) {
            tableModel.filterTableView();
            tableModel.sortTableView();
        }
    }
    
    /**
     * Gets the current instance of the specified role.
     *
     * @param <T> the expected domain type
     * @param role the constant denoting the role.
     * @return the current displayed instance.
     * @throws ClassCastException if the expected domain type is not assignable
     *             to the actual domain type of the object selection role.
     */
    @SuppressWarnings("unchecked")
    public final <T> T getCurrentInstance(RoleConstant role) {
        OSNode<T> node = (OSNode<T>) jController.getOSNode(role);
        return node.getCurrentInstance();
    }

    /**
     * Get the state of the specified object selection role.
     *
     * @param role the constant denoting the object selection role
     * @return the role's state.
     */
    public final RoleState getState(RoleConstant role) {
        return jController.getOSNode(role).getState();
    }

    /**
     * Set the state of the specified object selection role.
     *
     * @param role the constant denoting the object selection role.
     * @param state the new state to set.
     */
    public final void setState(RoleConstant role, RoleState state) {
        jController.getOSNode(role).setState(state);
    }

    /**
     * Returns a string containing the role states of each role in this dialog's
     * object selection. This is mainly intended for short term debugging
     * purposes. The returned string consists of one line per object selection
     * role listing the role's name, the role's state and the list of changed
     * attributes.
     *
     * @return a listing of all role's state.
     */
    public final String getRoleStates() {
        return new RoleStateVisitor().getRoleStates();
    }

    private class RoleStateVisitor implements Visitor<OSRole<?>> {
        private String result = "";

        @Override
        public void visit(OSRole<?> visitable) {
            RoleConstant roleConstant = visitable.getRoleConstant();
            RoleState state = getState(roleConstant);
            result += visitable.toString() + ": " + state + ", "
                    + getChangedAttributes(roleConstant) + "\n";
        }

        String getRoleStates() {
            OSNode<?>[] rootNodes = jController.getRootNodes();
            for (OSNode<?> node : rootNodes) {
                node.visitBranch(this);
            }
            int lastNewLine = result.lastIndexOf("\n");
            return result.substring(0, lastNewLine);
        }

    }

    /**
     * Invoke the specified runnable on the gui thread.
     *
     * @param guiMethod the runnable to be run by the gui thread.
     * @throws InvocationTargetException exceptions thrown by the runnable are
     *             wrapped in an InvocationTargetException and re-thrown.
     */
    public final void invokeOnGui(Runnable guiMethod)
            throws InvocationTargetException {
        jController.invokeOnGui(guiMethod);
    }

    /**
     * Invoke the specified callable on the gui thread.
     *
     * @param <V> the return value
     * @param guiMethod the callable to be invoked by the gui thread.
     * @return the callable's return value
     * @throws InvocationTargetException exceptions thrown by the callable are
     *             wrapped in an InvocationTargetException and re-thrown.
     */
    public final <V> V invokeOnGui(Callable<V> guiMethod)
            throws InvocationTargetException {
        return jController.invokeOnGui(guiMethod);
    }

    /**
     * Run all validators with the specified policy for the specified role.
     *
     * @param osRole role to validate
     *
     * @param validationPolicy policy to use when finding validators
     * @return map of all validation results (both succeeded and failed) and
     *         corresponding validation context
     */
    public final Map<ValidationResult, ValidateContext> validate(
            RoleConstant osRole, Policy validationPolicy) {
        OSNode<Object> osNode = jController.getOSNode(osRole);
        return osNode.validate(null, null, validationPolicy);
    }

    /**
     * Hook method invoked when on change validation fails. This method should
     * be overridden in concrete dialog controllers which can handle validation
     * failures at own discretion. The returned map should contain validation
     * failures that are to be handled by the G9 runtime.
     *
     * @param failedResult map of failed validation results and corresponding
     *            context
     * @return map of unprocessed failed validations
     */
    public Map<ValidationResult, ValidateContext> failedOnChangeValidator(
            Map<ValidationResult, ValidateContext> failedResult) {
        return failedResult;
    }

    /**
     * Get the message context.
     *
     * @return a message context suitable for use in a jVine interactor setting.
     */
    public DispatcherContext getDispatcherContext() {
        AbstractApplicationView appView = (AbstractApplicationView) getApplicationController()
                .getApplicationView();
        return appView.getDispatcherContext();
    }

    @Override
    public MessageDispatcher getMessageDispatcher() {
    	return MessageSystem.getMessageDispatcher(getDispatcherContext());
    }

    /**
     * Dispatch the message with specified messageID and arguments.
     *
     * @param messageID string identifying the message
     * @param args message arguments
     * @return the message reply (if applicable)
     * @see MessageDispatcher#dispatch(String, Object...)
     */
    public MessageReply dispatchMessage(String messageID, Object... args) {
        return getMessageDispatcher().dispatch(messageID, args);
    }

    /**
     * Dispatch the specified message.
     *
     * @param message the message to dispatch
     * @return the message reply (if applicable)
     * @see MessageDispatcher#dispatch(Message)
     */
    public MessageReply dispatchMessage(Message message) {
        return getMessageDispatcher().dispatch(message);
    }

    /**
     * Dispatch the message with specified messageID, exception and arguments.
     *
     * @param messageID string identifying the message
     * @param exception the exception that caused the message to be dispatched
     * @param args message arguments
     * @return the message reply (if applicable)
     * @see MessageDispatcher#dispatch(String, Throwable, Object...)
     */
    public MessageReply dispatchMessage(String messageID, Throwable exception,
            Object... args) {
        return getMessageDispatcher().dispatch(messageID, exception, args);
    }

    /**
     * Gives a map of role constants and object selection roles.
     *
     * @return a map from role constant to object selection role.
     */
    public abstract Map<RoleConstant, OSRole<?>> getRoleMap();

	/**
	 * @return The current spreadsheet exporter
	 */
	public SpreadsheetExporter getSpreadsheetExporter() {
		return spreadsheetExporter;
	}

	/**
	 * Set the current spreadsheet exporter.
	 *
	 * @param spreadsheetExporter the new current spreadsheet exporter
	 */
	public void setSpreadsheetExporter(SpreadsheetExporter spreadsheetExporter) {
		this.spreadsheetExporter = spreadsheetExporter;
	}

}
