/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.controller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import no.esito.jvine.action.ActionMethodRunner;
import no.esito.jvine.controller.DialogInstanceKey;
import no.esito.jvine.controller.DialogKey;
import no.esito.jvine.controller.JVineController;
import no.esito.log.Logger;
import no.esito.util.ServiceLoader;
import no.g9.service.G9Spring;

import org.springframework.util.Assert;

/**
 * A manager for handling of multiple dialog instances. <br>
 *
 * <strong>WARNING:</strong> Although this class is public, it should not be treated as part of the public API, as it might
 * change in incompatible ways between releases (even patches).
 *
 */
public class DialogInstanceManager {

    /** Suffix for the dialog controllers' bean id */
    private static final String CONTROLLER = "Controller";

    /** Suffix for the dialog constants' bean id */
    private static final String CONSTANT = "Constant";

    /** The logger */
    private Logger log = Logger.getLogger(DialogInstanceManager.class);

    /** The application controller in use **/
    private ApplicationController applicationController;

    /**
     * Maps dialog object constants to the corresponding dialog controller.
     */
    private Map<DialogKey, Stack<DialogController>> dialogMap = Collections
            .synchronizedMap(new HashMap<DialogKey, Stack<DialogController>>());

    /**
     * Maps dialog object constants to the corresponding list of dialog controllers.
     */
    private Map<DialogKey, List<DialogController>> dialogLists = Collections
            .synchronizedMap(new HashMap<DialogKey, List<DialogController>>());

    /**
     * Instantiates a new dialog instance manager.
     *
     * @param applicationController the application controller
     */
    public DialogInstanceManager(ApplicationController applicationController) {
        super();
        Assert.notNull(applicationController, "applicationController must not be null");
        this.applicationController = applicationController;
    }

    /**
     * Get a new instance of the specified dialog controller (spring configured bean, configured as a prototype).
     *
     * @param <T> the actual class of the dialog controller.
     * @param beanID the Spring bean id.
     * @return the dialog controller.
     */
    @SuppressWarnings("unchecked")
    synchronized <T extends DialogController> T createDialogControllerInternal(String beanID) {
        T dialogController = (T) getDialogControllerBeanInternal(beanID);

        return dialogController;
    }

    /**
     * Gets the current dialog controller.
     *
     * @param <T> the generic type of the dialog
     * @param dialogConstant the dialog constant
     * @return the dialog controller refactor this
     */
    @SuppressWarnings("unchecked")
    synchronized <T extends DialogController> T getDialogController(DialogConstant dialogConstant) {
        Stack<DialogController> dcStack = dialogMap.get(new DialogKey(dialogConstant));
        if (dcStack != null && !dcStack.isEmpty()) {
            DialogController firstController = dcStack.peek();
            if (firstController != null) {
                if (log.isTraceEnabled()) {
                    log.trace("Got first controller from stack, instance " + firstController.getDialogInstanceNumber() + " of "
                            + dialogConstant.getInternalName() + ", g9Name " + dialogConstant.getG9Name());
                }
                return (T) firstController;
            }
        }

        // first instance of this dialog
        DialogController newController = createAndRegisterDialogController(dialogConstant.getInternalName(), 1);
        return (T) newController;
    }

    /**
     * Creates and returns a new instance of a given dialog controller and pushes it onto the stack.
     *
     * @param <T> the generic type
     * @param dialogConstant the dialog constant
     * @return the new dialog controller
     * @throws InstanceNumberOutOfBoundsException when no more dialogs of this dialog can be created.
     */
    @SuppressWarnings("unchecked")
    synchronized <T extends DialogController> T createDialogController(DialogConstant dialogConstant) {
        return (T) createDialogController(dialogConstant, -1);
    }

    private synchronized <T extends DialogController> T createDialogController(DialogConstant dialogConstant, int instanceNumber) {
        DialogController newController = createDialogControllerInternal(dialogConstant.getInternalName());
        return this.<T>registerController(instanceNumber, newController);
    }

    /**
     * @param <T> The DialogController subclass to return
     * @param instanceNumber instanceNumber
     * @param newController controller
     * @return new controller
     */
    @SuppressWarnings("unchecked")
    <T extends DialogController> T registerController(int instanceNumber, DialogController newController) {
    	DialogConstant dialogConstant=newController.getDialogConstant();
		int dialogInstanceNumber = instanceNumber > 0 ? instanceNumber : getNextAvailableInstanceNumber(dialogConstant);
        if (!validateInstanceNumber(dialogConstant, dialogInstanceNumber)) {
            throw new InstanceNumberOutOfBoundsException("Cannot create new instance with instanceNumber "
                    + dialogInstanceNumber);
        }
        newController.setDialogInstanceNumber(dialogInstanceNumber);
        registerDialog(newController);
        return (T) newController;
	}

    /**
     * Creates a new dialog intance and pushes it on top of the stack.
     *
     * @param dialogConstant the dialog constant
     * @return the dialog instance
     * @param instanceNumber the instance number, set to -1 to get the next available instanceNumber
     * @throws InstanceNumberOutOfBoundsException when no more dialogs of this dialog can be created.
     * @throws IllegalStateException when instanceNumber refers to an instance that already exists
     */
    DialogInstance createDialogInstance(DialogConstant dialogConstant, int instanceNumber) {
        Assert.notNull(dialogConstant, "dialogConstant must not be null");
        if (instanceNumber > 0) {
            if (checkForInstanceInUse(dialogConstant, instanceNumber)) {
                throw new IllegalStateException("Instance " + instanceNumber + " of " + dialogConstant + " is in use");
            }
        }
        DialogController dialogController = createDialogController(dialogConstant, instanceNumber);
        return new DialogInstanceKey(dialogConstant, dialogController.getDialogInstanceNumber());
    }

    /**
     * Get the dialog constant for the specified dialog.
     *
     * @param dialogInternalName the dialog name as it is defined in the bean id.
     * @return the dialog constant.
     * @throws ClassCastException if the returned type does not match the expected type.
     */
    synchronized DialogConstant getDialogConstant(String dialogInternalName) {
        return getDialogConstantBeanInternal(dialogInternalName);
    }

    /**
     * Sets the active dialog instance.
     *
     * @param dialogInstanceKey the key identifying the dialog instance
     * @return the dialog controller identified by the dialogInstanceKey, <code>null</code> if no instance found with the given
     *         id
     */
    synchronized DialogController setActiveDialogInstance(DialogInstance dialogInstanceKey) {
        Assert.notNull(dialogInstanceKey, "dialogInstanceKey must not be null");
        Stack<DialogController> dcStack = dialogMap.get(new DialogKey(dialogInstanceKey.getDialogConstant()));
        if (dcStack == null || dcStack.isEmpty()) {
            return null;
        }
        DialogController newNumberOne = locateInstance(dialogInstanceKey, dcStack);
        if (newNumberOne != null) {
            dcStack.remove(newNumberOne);
            dcStack.push(newNumberOne);
        }

        return newNumberOne;
    }

    /**
     * Gets the dialog controller identified by the instance key. No manipulation of the controller stack is performed.
     *
     * @param dialogInstanceKey the dialog instance key
     * @return the dialog controller, <code>null</code> if none found.
     */
    synchronized DialogController getDialogController(DialogInstance dialogInstanceKey) {
        Assert.notNull(dialogInstanceKey, "dialogInstanceKey must not be null");
        Stack<DialogController> dcStack = dialogMap.get(new DialogKey(dialogInstanceKey.getDialogConstant()));
        if (dcStack == null || dcStack.isEmpty()) {
            return null;
        }
        for (DialogController dialogController : dcStack) {
            if (dialogInstanceKey.getDialogInstanceNumber() == dialogController.getDialogInstanceNumber()) {
                return dialogController;
            }
        }
        return null;
    }

    /**
     * Wraps the static bean factory for dialog controller in this method, to ease testing
     *
     * @param dialogInternalName the internal name of the dialog
     * @return the bean internal
     */
    @SuppressWarnings("cast")
    DialogController getDialogControllerBeanInternal(String dialogInternalName) {
        String beanID = dialogInternalName + DialogInstanceManager.CONTROLLER;
        if (log.isDebugEnabled()) {
            log.debug("Creating new controller using beanID " + beanID);
        }

        DialogController dialogController = (DialogController) G9Spring.getBean(DialogController.class, beanID);
        return dialogController;
    }

    /**
     * Wraps the static bean factory for dialog constants in this method, to ease testing
     *
     * @param dialogInternalName the internal name of the dialog
     *
     * @return the dialog constant
     */
    @SuppressWarnings("cast" )
    DialogConstant getDialogConstantBeanInternal(String dialogInternalName) {
        String beanID = dialogInternalName + DialogInstanceManager.CONSTANT;
        if (log.isTraceEnabled()) {
            log.trace("Getting dialog constant using beanID " + beanID);
        }
        DialogConstant dialogConstant = (DialogConstant) G9Spring.getBean(DialogConstant.class, beanID);
        return dialogConstant;
    }

    /**
     * Wraps the static bean factory in this method, to ease testing
     *
     * @param <T> the generic type
     * @param clazz the clazz
     * @return the bean internal
     */
    <T> T getBeanInternal(Class<T> clazz) {
        return ServiceLoader.getService(clazz);
    }

    /**
     * Gets the next available instance number for a given dialog.
     *
     * @param dialogConstant the dialog constant
     * @return the next available instance number
     * @throws InstanceNumberOutOfBoundsException when no more dialogs of this dialog can be created.
     */
    synchronized int getNextAvailableInstanceNumber(DialogConstant dialogConstant) {
        Assert.notNull(dialogConstant, "dialogConstant must not be null");
        int start = 1;
        DialogKey dialogKey = new DialogKey(dialogConstant);
        Stack<DialogController> dialogControllers = dialogMap.get(dialogKey);
        if (dialogControllers != null && !dialogControllers.isEmpty()) {
            List<DialogController> sortedList = new ArrayList<DialogController>(dialogControllers);
            Collections.sort(sortedList, new Comparator<DialogController>() {
                @Override
                public int compare(DialogController o1, DialogController o2) {
                    return o1.getDialogInstanceNumber() - o2.getDialogInstanceNumber();
                }
            });
            for (DialogController dialogController : sortedList) {
                if (dialogController.getDialogInstanceNumber() == start) {
                    start++;
                }
            }
        }
        return start;
    }

    /**
     * Removes the dialog instance identified by the instance key from the stack.
     *
     * @param dialogInstanceKey the dialog instance key of the dialog to remove.
     */
    synchronized void removeDialogInstance(DialogInstance dialogInstanceKey) {
        Assert.notNull(dialogInstanceKey, "dialogInstanceKey must not be null");
        removeFromStack(dialogInstanceKey);
        removeFromList(dialogInstanceKey);

    }

    private void removeFromStack(DialogInstance dialogInstanceKey) {
        Stack<DialogController> dcStack = dialogMap.get(new DialogKey(dialogInstanceKey.getDialogConstant()));
        if (dcStack == null || dcStack.isEmpty()) {
            return;
        }
        DialogController instanceToClose = locateInstance(dialogInstanceKey, dcStack);
        if (instanceToClose != null) {
            dcStack.remove(instanceToClose);
        }
    }

    private synchronized void removeFromList(DialogInstance dialogInstanceKey) {
        DialogConstant dialogConstant = dialogInstanceKey.getDialogConstant();
        List<DialogController> dcList = dialogLists.get(new DialogKey(dialogConstant));
        if (dcList == null || dcList.isEmpty()) {
            return;
        }
        DialogController instanceToClose = locateInstance(dialogInstanceKey, dcList);
        if (instanceToClose != null) {
            dcList.remove(instanceToClose);
        }
    }

    private DialogController locateInstance(DialogInstance dialogInstanceKey, List<DialogController> dcStack) {
        DialogController instanceToClose = null;
        for (DialogController dialogController : dcStack) {
            if (dialogInstanceKey.getDialogInstanceNumber() == dialogController.getDialogInstanceNumber()) {
                instanceToClose = dialogController;
                break;
            }
        }
        return instanceToClose;
    }

    /**
     * Checks for available dialog instance.
     *
     * @param dialogConstant the dialog constant
     * @return true if there is one or more available dialog instances for the given dialog
     */
    boolean hasAvailableDialogInstance(DialogConstant dialogConstant) {
        Assert.notNull(dialogConstant, "dialogConstant must not be null");
        int nextAvailableInstanceNumber = getNextAvailableInstanceNumber(dialogConstant);
        boolean isValid = validateInstanceNumber(dialogConstant, nextAvailableInstanceNumber);
        if (log.isTraceEnabled()) {
            log.trace("hasAvailableDialogInstances: " + dialogConstant.getG9Name() + ", max: "
                    + dialogConstant.getMaximumNumberOfInstances() + ", next: " + nextAvailableInstanceNumber + ", isValid: "
                    + isValid);
        }
        return isValid;
    }

    /**
     * Gets a list of active dialog instances.
     *
     * @param dialogConstant the dialog constant identifying the dialog
     * @return the list of active dialog instances, an empty list if no active dialogs of this type
     */
    synchronized List<DialogInstance> getActiveDialogInstances(DialogConstant dialogConstant) {
        Stack<DialogController> dcStack = dialogMap.get(new DialogKey(dialogConstant));
        if (dcStack == null || dcStack.isEmpty()) {
            return Collections.emptyList();
        }
        return createDialogInstanceList(dcStack);
    }

    /**
     * Get a list of open dialog instances for a dialog.
     * The list is ordered in the sequence the dialog instances were opened.
     *
     * @param dialogConstant generated constant denoting the dialog
     * @return list of open dialog instances
     */
    synchronized List<DialogInstance> getOpenDialogList(DialogConstant dialogConstant) {
        List<DialogController> dcList = dialogLists.get(new DialogKey(dialogConstant));
        if (dcList == null || dcList.isEmpty()) {
            return Collections.emptyList();
        }
        return createDialogInstanceList(dcList);

    }

    /**
     * Get a list of all open dialog controllers.
     * The list is ordered in the sequence the dialog instances were opened.
     *
     * @return list of open dialog instances
     */
    synchronized List<DialogController> getOpenDialogList() {
    	List<DialogController> dcList = new ArrayList<>();
    	for (DialogKey dk : dialogLists.keySet()) {
    		dcList.addAll(dialogLists.get(dk));
    	}
    	return dcList;
    }
    
    private List<DialogInstance> createDialogInstanceList(List<DialogController> dcList) {
        List<DialogInstance> activeDialogInstances = new ArrayList<DialogInstance>();
        for (DialogController dialogController : dcList) {
            activeDialogInstances.add(createDialogInstanceKey(dialogController));
        }
        return Collections.unmodifiableList(activeDialogInstances);
    }

    private DialogInstance createDialogInstanceKey(DialogController controller) {
        Assert.notNull(controller, "controller must not be null");
        return new DialogInstanceKey(controller.getDialogConstant(), controller.getDialogInstanceNumber());
    }

    private DialogController createAndRegisterDialogController(String internalName, int instanceNumber) {
        DialogController newController = createDialogControllerInternal(internalName);
        newController.setDialogInstanceNumber(instanceNumber);
        if (log.isTraceEnabled()) {
            log.trace("Creating new controller for instance " + newController.getDialogInstanceNumber() + " of " + internalName);
        }

        registerDialog(newController);
        return newController;
    }

    /**
     * Gets the current instance number, i.e. the dialog on top of the stack.
     *
     * @param dialogConstant the dialog constant
     * @return the current dialog instance, <code>null</code> if no dialog found for this dialog constant
     */
    synchronized DialogInstance getCurrentDialogInstance(DialogConstant dialogConstant) {
        Assert.notNull(dialogConstant, "dialogConstant must not be null");
        Stack<DialogController> dcStack = dialogMap.get(new DialogKey(dialogConstant));
        if (dcStack != null && !dcStack.isEmpty()) {
            return new DialogInstanceKey(dialogConstant, dcStack.peek().getDialogInstanceNumber());
        }
        return null;
    }

    private synchronized void registerDialog(DialogController controller) {
        ActionMethodRunner actionMethodRunner = getBeanInternal(ActionMethodRunner.class);
        JVineController.getInstance(controller).setActionMethodRunner(actionMethodRunner);
        actionMethodRunner.setDialogController(controller);
        addToStack(controller);
        addToList(controller);
        controller.setApplicationController(this.applicationController);
    }

    private synchronized void addToStack(DialogController controller) {
        JVineController jvc = JVineController.getInstance(controller);
        DialogKey dialogKey = jvc.getDialogKey();
        Stack<DialogController> stack = dialogMap.get(dialogKey);

        if (stack == null) {
            stack = new Stack<DialogController>();
            dialogMap.put(dialogKey, stack);
        }

        stack.push(controller);
    }

    private synchronized void addToList(DialogController controller) {
        JVineController jvc = JVineController.getInstance(controller);
        DialogKey dialogKey = jvc.getDialogKey();
        List<DialogController> list = dialogLists.get(dialogKey);

        if (list == null) {
            list = new LinkedList<DialogController>();
            dialogLists.put(dialogKey, list);
        }

        list.add(controller);
    }

    private boolean validateInstanceNumber(DialogConstant dialogConstant, int dialogInstanceNumber) {
        if (dialogConstant.getMaximumNumberOfInstances() <= 0) {
            return dialogInstanceNumber == 1;
        }
        return dialogInstanceNumber <= dialogConstant.getMaximumNumberOfInstances();

    }

    private boolean checkForInstanceInUse(DialogConstant dialogConstant, int dialogInstanceNumber) {
        Stack<DialogController> dcStack = dialogMap.get(new DialogKey(dialogConstant));
        if (dcStack == null || dcStack.isEmpty()) {
            return false;
        }

        for (DialogController dialogController : dcStack) {
            if (dialogController.getDialogInstanceNumber() == dialogInstanceNumber) {
                return true;
            }
        }
        return false;

    }

}
