/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.validator;

import no.esito.log.Logger;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.util.MessageTextHelper;
import no.g9.client.core.validator.ValidationPolicy.Policy;
import no.g9.message.CRuntimeMsg;


/**
 * The mandatory validator is a {@link Policy#ON_SAVE form} validator. This
 * validator is used to test that a field <em>has</em> a value. For
 * <code>String</code>s, this means that the field must contain at least one
 * non-blank character. For all other types, it must be something other than
 * <code>null</code>. This validator does not specify a validation policy.
 */
public class MandatoryValidator implements FieldValidator {
    private static final Logger log = Logger.getLogger(MandatoryValidator.class);
    
    
	/**
	 * Validates a candidate value. The validation fails if either
	 * <ul>
	 * <li>the value is a blank or empty string
	 * <li>the value is <code>null</code>
	 * </ul>
     * All other values are accepted as legal values by this validator.
	 * @param value candidate value to validate
	 * @param context validate context 
	 * @return the result of the validation
	 */
	@Override
	public ValidationResult validate(Object value, ValidateContext context) {
	    DialogObjectConstant field = context.getDialogObjectConstant();
	    if (log.isTraceEnabled()) {
	        log.trace("Validating " + field);
	    }

	    Object[] args = new Object[]{MessageTextHelper.getLabelTextForField(context)};
	    ValidationResult failed = new ValidationResult(false, CRuntimeMsg.CF_MANDATORY_FAILED, args);

		// First case - empty or blank string...
		if (value instanceof String) {
			String sValue = ((String) value).trim();
			if (sValue.isEmpty()) {
				return failed;
			}
		}

		// Second case - value is null
		if (value == null) {
			return failed;
		}

		// All else is good...
		return ValidationResult.DEFAULT_OK_RESULT;
	}

	
}
