/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.validator;

/**
 * A field validator that always return ok (default) or fail for all
 * validations.
 *
 * Example of use:
 * {@code
 * <bean id="trueValidator"
 *     class="no.g9.client.core.validator.NopValidator">
 *     <property name="result" value="false" />
 * </bean>
 *
 * This bean supports the following properties:
 * <li> result: A boolean that tells if validations should succeed or not.
 * <li> messageNumber: Specifies message number to return on all validations.
 * <li> messageArgs: An array that specifies message arguments to return.}
 *
 *  By default, this validator will return true for all validations.
 */
public class NopValidator implements FieldValidator {

    private boolean result = true;
    private String messageID = ValidationResult.DEFAULT_FAIL_MESSAGE;
    private Object[] messageArgs = null;
    private ValidationResult validationResult;

    /**
     * Default constructor.
     */
    public NopValidator() {
        validationResult = new ValidationResult(result, messageID, messageArgs);
    }

    /**
     * Sets the result to return for validation queries.
     * @param result the result to set
     */
    public void setResult(boolean result) {
        this.result = result;
        validationResult = new ValidationResult(result, messageID, messageArgs);
    }

    /**
     * Sets message number to return on failed validations.
     * @param messageID The message id to return on failed validations
     */
    public void setMessageID(String messageID) {
        this.messageID = messageID;
        validationResult = new ValidationResult(result, messageID, messageArgs);
    }

    /**
     * Sets message text to return on failed validations.
     * @param messageArgs The message text to return on failed validations
     */
    public void setMessageArgs(Object[] messageArgs) {
        this.messageArgs = (messageArgs != null) ? messageArgs.clone() : null;
        validationResult = new ValidationResult(result, messageID, this.messageArgs);
    }

    @Override
    public ValidationResult validate(Object value, ValidateContext context) {
        return validationResult;
    }

}
