/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.validator;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A validator that validates against a regexp pattern.
 * This validator will match the String value of candidate values against a set
 * of regular expressions. If any of them match, the candidate is valid. If
 * we get no match, the result is negative.
 *
 * Example: How to configure a validator that checks if the candidate value
 * consists of at least 6 characters.
 *
 * {@code
 * <bean id="passwordValidator"
 *     class="no.g9.client.core.validator.RegexpValidator">
 *     <property name="patterns" value=".{6,}" />
 * </bean>}
 *
 * This bean supports the following properties:
 * <ul>
 * <li> patterns: An array of regular expression strings to validate against.
 *      @see java.util.regex.Pattern
 * <li> pattern: A single pattern given as string.
 * <li> failMessageNumber: Directs what message number to return on validation
 *      fail.
 * <li> failMessageArgs: An array that specifies fail message arguments
 * </ul>
 *
 *
 */
public class RegexpValidator implements FieldValidator {

    private Pattern[] patterns;
    private String faileMessageID = ValidationResult.DEFAULT_FAIL_MESSAGE;
    private Object[] failMessageArgs = null;

    /**
     * Sets patterns to validate against.
     * @param patterns The regexp patterns to validate against
     */
    public void setPatterns(String[] patterns) {
        this.patterns = new Pattern[patterns.length];
        for (int i = 0; i < patterns.length; i++) {
            this.patterns[i] = Pattern.compile(patterns[i]);
        }
    }

    /**
     * Sets a single pattern to validate against.
     * @param pattern The regexp pattern to validate against
     */
    public void setPattern(String pattern) {
        this.patterns = new Pattern[] {Pattern.compile(pattern)};
    }

    /**
     * Sets message number to return on failed validations.
     * @param messageID The message number to return on failed validations
     */
    public void setFailMessageNumber(String messageID) {
        this.faileMessageID = messageID;
    }

    /**
     * Sets message text to return on failed validations.
     * @param messageArgs The message text to return on failed validations
     */
    public void setFailMessageArgs(Object[] messageArgs) {
        this.failMessageArgs = (messageArgs != null) ? messageArgs.clone() : null;
    }

    @Override
    public ValidationResult validate(Object value, ValidateContext context) {
        if (value == null) return ValidationResult.DEFAULT_OK_RESULT;
        String s = value.toString();
        for (Pattern pattern : patterns) {
            Matcher matcher = pattern.matcher(s);
            if (matcher.matches()) {
                return ValidationResult.DEFAULT_OK_RESULT;
            }
        }
        // If here, we had no match
        return new ValidationResult(false, faileMessageID, failMessageArgs);
    }

}
