/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.validator;

import java.util.Arrays;

import com.google.common.base.Objects;

import no.g9.message.CRuntimeMsg;

/**
 * Validation result holder.
 */
public class ValidationResult {

	/** Default message number for validation ok */
	public final static String DEFAULT_OK_MESSAGE = null;

	/** Default message number for validation fail */
	public final static String DEFAULT_FAIL_MESSAGE = CRuntimeMsg.CF_VALIDATION_FAILED;

	/** Default ok validation result */
	public final static ValidationResult DEFAULT_OK_RESULT = new ValidationResult(
			true);

	/** Default fail validation result */
	public final static ValidationResult DEFAULT_FAIL_RESULT = new ValidationResult(
			false, DEFAULT_FAIL_MESSAGE, null);

	// Validation result
	private final boolean result;

	// Message number of validation message
	private final String messageID;

	// Message arguments of validation message
	private final Object[] msgArgs;

	/**
	 * Creates a new ValidateResult.
	 *
	 * @param result
	 *            The result of the validation, <code>true</code> for passed
	 *            validation, <code>false</code> for failed validation.
	 * @param msgID
	 *            Message number of validation message (if any)
	 * @param msgArgs
	 *            Message arguments of validation result (if any)
	 */
	public ValidationResult(boolean result, String msgID, Object[] msgArgs) {
		this.result = result;
		this.messageID = msgID;
		this.msgArgs = (msgArgs != null) ? msgArgs.clone() : null;
	}

	/**
	 * Creates a new ValidateResult.
	 *
	 * @param result
	 *            The result of the validation, <code>true</code> for passed
	 *            validation, <code>false</code> for failed validation.
	 */
	public ValidationResult(boolean result) {
		this.result = result;
		this.messageID = result ? DEFAULT_OK_MESSAGE : DEFAULT_FAIL_MESSAGE;
		this.msgArgs = null;
	}

	/**
	 * Returns the validation result. <code>true</code> means that the
	 * validation succeeded.
	 *
	 * @return the validation result
	 */
	public boolean succeeded() {
		return result;
	}

	/**
	 * Returns the validation result. <code>true</code> means that the
	 * validation succeeded.
	 *
	 * @return the validation result
	 * @deprecated Use {@link #succeeded} instead
	 */
	@Deprecated
	public boolean getResult() {
		return result;
	}

	/**
	 * Returns the message number of the validation message.
	 *
	 * @return the message number of the validation message
	 */
	public String getMsgNumber() {
		return messageID;
	}

	/**
	 * Returns the message arguments of the validation message.
	 *
	 * @return the message arguments of the validation message
	 */
	public Object[] getMsgArgs() {
		return (msgArgs != null) ? msgArgs.clone() : null;
	}

    @Override
    public String toString() {
        return "ValidationResult [result=" + result + ", messageID="
                + messageID + ", msgArgs=" + Arrays.toString(msgArgs) + "]";
    }
    
    @Override
    public boolean equals(Object obj) {
        if (obj == null) return false;
        if (this.getClass() != obj.getClass()) return false;
        final ValidationResult other= (ValidationResult) obj;
        return Objects.equal(this.result, other.result) &&
                Objects.equal(this.messageID, other.messageID) &&
                Objects.equal(Arrays.toString(this.msgArgs), Arrays.toString(this.msgArgs));
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(this.result, this.messageID, Arrays.toString(this.msgArgs));
    }

}
