/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.spreadsheet;

import no.esito.util.BeanID;
import no.esito.util.G9PropertyName;

import org.apache.poi.ss.usermodel.Workbook;

/**
 * A common interface for workbook providers.
 * It also has the WORKBOOK_FORMAT enum which enumerates the available workbook formats.
 * <br>
 * The implementation to be used at runtime can be controlled
 * by configuring a "workbookProvider" Spring bean, or by setting
 * the "WorkbookProvider" property in the "g9.config.properties" file.
 * <br>
 * The default is to use <code>no.g9.client.spreadsheet.DefaultWorkbookProvider</code>.
 */
@BeanID("workbookProvider")
@G9PropertyName("WorkbookProvider")
public interface WorkbookProvider {

	/**
	 * Create a new workbook in the given format.
	 *
	 * @param bookFormat The workbook format to use
	 * @return a new Workbook object in the given format
	 */
	Workbook createWorkbook(WORKBOOK_FORMAT bookFormat);

	/**
	 * Enumerator to indicate workbook format.
	 */
	enum WORKBOOK_FORMAT {
		/** Old school Excel format. */
		XLS(".xls", "application/vnd.ms-excel"),
		/** Excel format 2007 onwards. */
		XLSX(".xlsx", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");

		private String suffix;
		private String mimeType;

		private WORKBOOK_FORMAT(String suffix, String mimeType) {
			this.suffix = suffix;
			this.mimeType = mimeType;
		}

		/**
		 * Gets the suffix value.
		 *
		 * @return The suffix
		 */
		public String getSuffix() {
			return suffix;
		}

		/**
		 * Gets the mimeType value.
		 *
		 * @return The mimeType
		 */
		public String getMimeType() {
			return mimeType;
		}

	}

}
