/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.awt.AWTKeyStroke;
import java.awt.Component;
import java.awt.KeyboardFocusManager;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.text.ParseException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.swing.JRootPane;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.KeyStroke;
import javax.swing.UIManager;
import javax.swing.event.EventListenerList;
import javax.swing.text.Document;

import no.g9.client.event.G9ValueChangedEvent;
import no.g9.client.event.G9ValueChangedListener;
import no.g9.client.event.G9ValueState;
import no.g9.client.support.EnumeratorDocument;
import no.g9.client.support.G9Document;
import no.g9.client.support.G9FieldValue;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.*;
import no.g9.support.TypeTool;


/**
 * In addition to being a JTextArea, this class implements the G9ValueState
 * interface.
 */
public class G9TextArea extends JTextArea implements G9ValueState, G9FieldValue {

    /** The old value of this component, used to trigger value changed */
    private Object oldVal;

    /**
     * The initial value of this component, used to find out if this component
     * is changed.
     */
    private Object initialValue;

    /** Bug fix flag */
    private boolean bugFix;

    /** Editable flag */
    private boolean editable = true;

    /** Enabled flag */
    private boolean enabled = true;

    /** Select all on focus flag */
    private boolean selectAllOnFocus = false;

    /** List of value changed listeners */
    protected EventListenerList vcListeners = new EventListenerList();

    /**
     * Constructs a new TextArea. A default model is set, the initial string is
     * null, and rows/columns are set to 0.
     */
    public G9TextArea() {
        this(null, null, 0, 0);
    }

    /**
     * Constructs a new empty TextArea with the specified number of rows and
     * columns. A default model is created, and the initial string is null.
     *
     * @param rows the number of rows &gt;= 0
     * @param columns the number of columns &gt;= 0
     * @exception IllegalArgumentException if the rows or columns arguments are
     *                negative.
     */
    public G9TextArea(int rows, int columns) {
        this(null, null, rows, columns);
    }

    /**
     * Constructs a new TextArea with the specified text displayed. A default
     * model is created and rows/columns are set to 0.
     *
     * @param text the text to be displayed, or null
     */
    public G9TextArea(String text) {
        this(null, text, 0, 0);
    }

    /**
     * Constructs a new TextArea with the specified text and number of rows and
     * columns. A default model is created.
     *
     * @param text the text to be displayed, or null
     * @param rows the number of rows &gt;= 0
     * @param columns the number of columns &gt;= 0
     * @exception IllegalArgumentException if the rows or columns arguments are
     *                negative.
     */
    public G9TextArea(String text, int rows, int columns) {
        this(null, text, rows, columns);
    }

    /**
     * Constructs a new JTextArea with the given document model, and defaults
     * for all of the other arguments (null, 0, 0).
     *
     * @param doc the model to use
     */
    public G9TextArea(Document doc) {
        this(doc, null, 0, 0);
    }

    /**
     * Constructs a new JTextArea with the specified number of rows and columns,
     * and the given model. All of the constructors feed through this
     * constructor.
     *
     * @param doc the model to use, or create a default one if null
     * @param text the text to be displayed, null if none
     * @param rows the number of rows &gt;= 0
     * @param columns the number of columns &gt;= 0
     * @exception IllegalArgumentException if the rows or columns arguments are
     *                negative.
     */
    public G9TextArea(Document doc, String text, int rows, int columns) {
        super(doc, text, rows, columns);
        initialValue = text;
        setUpFocusTraversalKeys();

        addFocusListener(new FocusAdapter() {
            @Override public void focusGained(FocusEvent e) {
                ((G9Document) getDocument()).setInputMode(true);
                if (menuOrToolbar(e.getOppositeComponent())) {
                    return;
                }
                oldVal = getValue();
                if (isSelectAllOnFocus()) {
                    selectAll();
                }
            }

            @Override public void focusLost(FocusEvent e) {
                if (menuOrToolbar(e.getOppositeComponent())) {
                    return;
                }
                Object value = getValue();
                if ((oldVal != null ^ value != null)
                    || oldVal != null && !oldVal.equals(value)) {
                    fireG9ValueChangedEvent(
                            G9ValueChangedEvent.VALUE_CHANGED, oldVal,
                            value);
                }
                ((G9Document) getDocument()).setInputMode(false);
                setCaretPosition(0);
            }

            /**
             * Check if the component loosing focus is a menu or toolbar
             * component
             *
             * @param opposite the component that lost focus
             * @return <code>true</code> if the component is a menu or toolbar
             *         component
             */
            private boolean menuOrToolbar(Component opposite) {
                return opposite == null
                        || opposite instanceof JRootPane
                        || (opposite instanceof G9Button && ((G9Button) opposite)
                                .isInToolbar());

            }
        });

        editable = true;
        enabled = true;
        bugFix = TypeTool.isVmVersion1_4();

    }

    @Override
    public void addValueChangedListener(G9ValueChangedListener listener) {
        vcListeners.add(G9ValueChangedListener.class, listener);
    }

    /**
     * Fires a g9 value changed event.
     *
     * @param id the type of event
     * @param oldValue the old value
     * @param newValue the new value
     */
    protected void fireG9ValueChangedEvent(int id, Object oldValue,
            Object newValue) {
        Object[] listeners = vcListeners.getListenerList();
        G9ValueChangedEvent e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (e == null) {
                e = new G9ValueChangedEvent(this, id, oldValue,
                        newValue);
            }
            if (listeners[i] == G9ValueChangedListener.class) {
                switch (e.getID()) {
                case G9ValueChangedEvent.VALUE_CHANGED:
                    ((G9ValueChangedListener) listeners[i + 1]).valueChanged(e);
                    break;
                default:
                    break;
                }
            }
        }
    }

    @Override
    public Object getValue() {
        Object value = null;
        Document doc = getDocument();
        if (doc instanceof G9Document) {
            G9Document gDoc = (G9Document) doc;
            value = gDoc.getValue();
        } else {
            value = getText(); // Fallback
        }
        return value;
    }

    @Override
    public void setValue(Object value) {
        Document doc = getDocument();
        if (doc instanceof G9Document) {
            ((G9Document) doc).setValue(value);
        } else { // Fallback.
            setText(String.valueOf(value));
        }
        setCaretPosition(0);
    }

    /**
     * @see no.g9.client.support.G9FieldValue#format()
     */
    @Override
    public String format() {
        String str = "";
        Document doc = getDocument();
        if (doc instanceof G9Document) {
            G9Document gDoc = (G9Document) doc;
            str = gDoc.format();
        }
        return  str;
    }

    @Override
    public Object parse(String formattedString) {
        Object o = formattedString;
        Document doc = getDocument();
        if (doc instanceof G9Document) {
            G9Document gDoc = (G9Document) doc;
            try {
                o = gDoc.parse(formattedString);
            }
            catch (ParseException e) {
                Object[] args = { getClass(), getName(), formattedString };
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CF_DOCUMENT_PARSEEXCEPTION, args);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(e, msg);
            }
        }
        return o;
    }

    /**
     * Parses the object
     *
     * @param o the object to parse
     * @return the object
     */
    public Object parse(Object o) {
        return o;
    }

    @Override
    public void resetState() {
        oldVal = getValue();
        initialValue = oldVal;
    }

    @Override
    public boolean isChanged() {

        // both null
        if (initialValue == null && getValue() == null) {
            return false;
        }
        // one null the other not.
        if (initialValue == null ^ getValue() == null) {
            return true;
        }

        // neither is null, make sure both values have equal formatting
        // before comparing
        Object formattedInitial = ((G9Document) getDocument())
                .transform(initialValue);
        Object formattedCurrent = ((G9Document) getDocument())
                .transform(getValue());
        return (formattedCurrent == null) ? formattedInitial != null : !formattedCurrent.equals(formattedInitial);
    }

    @Override
    public void setText(String t) {
        Document doc = getDocument();
        if (doc instanceof EnumeratorDocument) {
            setValue(((EnumeratorDocument) doc).toValue(t));
        } else {
            super.setText(t);
        }

    }

    @Override
    public void display(Object value) {
        Document doc = getDocument();
        if (doc instanceof EnumeratorDocument && value instanceof String) {
            value = ((EnumeratorDocument) doc).toValue(value);
        }
        setValue(value);
    }

    @Override
    public Object getInitialValue() {
        return initialValue;
    }

    @Override
    public void setInitialValue(Object value) {
        Document doc = getDocument();
        if (doc instanceof EnumeratorDocument && (value instanceof String || value instanceof Integer)) {
            value = ((EnumeratorDocument) doc).toValue(value);
        }
        initialValue = value;
        oldVal = initialValue;
    }

    @Override
    public void setEnabled(boolean enable) {
        super.setEnabled(enable);
        if (!bugFix) {
            return;
        }
        enabled = enable;
        if (!enable || editable) {
            super.setEditable(enable);
        }

        if (enable) {
            setUpFocusTraversalKeys();
        }

    }

    @Override
    public void setEditable(boolean editable) {
        this.editable = editable;
        super.setEditable(editable);
        setEnabled(enabled);
    }

    @Override
    public void setVisible(boolean show) {
    	Component parent = getParent();
    	while (parent != null && !(parent instanceof JScrollPane)) {
    		parent = parent.getParent();
    	}
    	if (parent != null) {
    		parent.setVisible(show);
    	} else {
    		super.setVisible(show);
    	}
    }

    /**
     * If the selectAllOnFocus property is <code>true</code> the displayed text
     * is selected each time this component gains focus.
     *
     * @return Returns the selectAllOnFocus.
     */
    public boolean isSelectAllOnFocus() {
        return selectAllOnFocus;
    }

    /**
     * If the selectAllOnFocus property is <code>true</code> the displayed text
     * is selected each time this component gains focus.
     *
     * @param selectAllOnFocus The selectAllOnFocus property.
     */
    public void setSelectAllOnFocus(boolean selectAllOnFocus) {
        this.selectAllOnFocus = selectAllOnFocus;
    }

    /**
     * Set up tab and shift tab key as forward / backward key in Windows L&F,
     */
    private void setUpFocusTraversalKeys() {
        if (UIManager.getLookAndFeel().getID().equals("Windows")) {
            Set<AWTKeyStroke> unmodifiable = getFocusTraversalKeys(KeyboardFocusManager.FORWARD_TRAVERSAL_KEYS);
            Set<AWTKeyStroke> forwardTraversalKeys = new HashSet<AWTKeyStroke>();
            KeyStroke tabKey = KeyStroke.getKeyStroke(KeyEvent.VK_TAB, 0);
            forwardTraversalKeys.add(tabKey);
            Iterator<AWTKeyStroke> it = unmodifiable.iterator();
            while (it.hasNext()) {
                AWTKeyStroke o = it.next();
                forwardTraversalKeys.add(o);
            }
            setFocusTraversalKeys(KeyboardFocusManager.FORWARD_TRAVERSAL_KEYS, forwardTraversalKeys);

            unmodifiable = getFocusTraversalKeys(KeyboardFocusManager.BACKWARD_TRAVERSAL_KEYS);
            Set<AWTKeyStroke> backwardTraversalKeys = new HashSet<AWTKeyStroke>();
            KeyStroke shiftTabKey = KeyStroke.getKeyStroke(KeyEvent.VK_TAB, InputEvent.SHIFT_DOWN_MASK);
            backwardTraversalKeys.add(shiftTabKey);
            it = unmodifiable.iterator();
            while (it.hasNext()) {
                AWTKeyStroke o = it.next();
                backwardTraversalKeys.add(o);
            }
            setFocusTraversalKeys(KeyboardFocusManager.BACKWARD_TRAVERSAL_KEYS, backwardTraversalKeys);
        }
    }

}
