/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component.menu;

import java.awt.Component;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.JMenu;
import javax.swing.JMenuBar;

import no.g9.client.support.G9DialogFrame;

/**
 * Extends JMenuBar with the capability of merging Menu bars using the
 * {@link #mergeMenues(G9DialogFrame)}. Merging is performed according to
 * the rules specified in the "Genova 8.0-Sysdul Client/Server Architecture GUI
 * Runtime Environment" manual.
 */
public class G9MenuBar extends JMenuBar {

    /** Because of serialization */
    private static final long serialVersionUID = 1L;

    /** The file menu of this menu bar */
    private G9Menu fileMenu;

    /** The edit menu of this menu bar */
    private G9Menu editMenu;

    /** The view menu of this menu bar */
    private G9Menu viewMenu;

    /** The window menu of this menu bar */
    private G9Menu windowMenu;

    /** The help menu of this menu bar */
    private G9Menu helpMenu;

    /** All user/(custom) menus, key = name, value = menu */
    private Map<String, Component> customNameToMenu = new HashMap<String, Component>();

    /**
     * The menus of this menu (, including file, edit, view, window, help menu
     */
    private List<Component> myComponents;

    /** The parent frame of this menu bar. Never <code>null</code> */
    private G9DialogFrame parentFrame;

    /** Modality state of this menu bar */
    private boolean isModal = false;

    /**
     * Default constructor. Creates a new G9MenuBar
     * 
     * @param parentFrame the "owning" dialog of this menu bar
     */
    public G9MenuBar(G9DialogFrame parentFrame) {
        super();

        if (parentFrame == null)
            throw new IllegalArgumentException(
                    "Parent frame can not be null");

        this.parentFrame = parentFrame;
    }

    /**
     * Adds a menu to this menu bar.
     * 
     * @param menu the menu to add
     * @return the added menu
     */
    public synchronized JMenu addG9Menu(JMenu menu) {
        return add(menu);
    }

    /**
     * Returns this menu parent frame
     * 
     * @return this menu parent frame
     */
    public G9DialogFrame getParentFrame() {
        return this.parentFrame;
    }

    /**
     * Do not use this to add a menu to this menu bar! <br>
     * Use {@link #addG9Menu(JMenu)} <br>
     * Adds a menu to this menu bar.
     * 
     * @param c the menu to add
     * @return the added menu
     */
    @Override
    public synchronized JMenu add(JMenu c) {
        if (c instanceof G9Menu) {
            G9Menu menu = (G9Menu) c;
            switch (menu.getMenuType()) {
                case G9Menu.CUSTOM_MENU:
                    if (!customNameToMenu.containsKey(c.getName())) {
                        customNameToMenu.put(c.getName(), c);
                    }
                    break;
                case G9Menu.FILE_MENU:
                    fileMenu = menu;
                    break;
                case G9Menu.EDIT_MENU:
                    editMenu = menu;
                    break;
                case G9Menu.VIEW_MENU:
                    viewMenu = menu;
                    break;
                case G9Menu.WINDOW_MENU:
                    windowMenu = menu;
                    break;
                case G9Menu.HELP_MENU:
                    helpMenu = menu;
                    break;
                default:
                    break;
            }
        }

        if (myComponents == null) {
            myComponents = new ArrayList<Component>();
        }
        myComponents.add(c);

        return super.add(c);
    }

    /**
     * Merges the menu bar from the given dialog with this menu bar.
     * 
     * @param dialog the dialog that contains the menu bar to merge with this
     *            menu bar.
     */
    @SuppressWarnings("deprecation")
    public synchronized void mergeMenues(G9DialogFrame dialog) {
        Component[] menuComps = null;
        if (dialog != null && dialog.getG9MenuBar() != null) {
            menuComps = dialog.getG9MenuBar().getG9Components();
        }
        mergeMenues(menuComps, dialog != null ? dialog.isModal() : false);
        validate();
        repaint();
    }

    /**
     * Merges the component array with menu items into this menu.
     * 
     * @param menues array of menu items to merge
     * @param modal flag indicating if the dialog is modal
     */
    private synchronized void mergeMenues(Component[] menues,
            boolean modal) {

        synchronized (getTreeLock()) {

            removeAll();

            Component[] my = getG9Components();

            int mergePos = myComponents != null ? myComponents.size() : 0;
            if (windowMenu != null) {
                mergePos = myComponents.indexOf(windowMenu);
            } else if (helpMenu != null) {
                mergePos = myComponents.indexOf(helpMenu);
            }

            for (int i = 0; i < mergePos; i++) {
                add(my[i]);
            }

            Set<G9Menu> mergedMenus = new HashSet<G9Menu>();

            if (menues != null) {
                for (int i = 0; i < menues.length; i++) {
                    if (menues[i] instanceof G9Menu) {
                        G9Menu gm = (G9Menu) menues[i];
                        boolean merged = false;
                        switch (gm.getMenuType()) {
                            case G9Menu.CUSTOM_MENU:
                                G9Menu custom = (G9Menu) customNameToMenu
                                        .get(gm.getName());
                                if (custom != null) {
                                    custom.setText(gm.getText());
                                    custom.mergeMenu(gm);
                                    mergedMenus.add(custom);
                                    merged = true;
                                }
                                break;

                            case G9Menu.FILE_MENU:
                                if (fileMenu != null) {
                                    fileMenu.setText(gm.getText());
                                    fileMenu.mergeMenu(gm);
                                    mergedMenus.add(fileMenu);
                                    merged = true;
                                }
                                break;

                            case G9Menu.EDIT_MENU:
                                if (editMenu != null) {
                                    editMenu.setText(gm.getText());
                                    editMenu.mergeMenu(gm);
                                    mergedMenus.add(editMenu);
                                    merged = true;
                                }
                                break;

                            case G9Menu.VIEW_MENU:
                                if (viewMenu != null) {
                                    viewMenu.setText(gm.getText());
                                    viewMenu.mergeMenu(gm);
                                    mergedMenus.add(viewMenu);
                                    merged = true;
                                }
                                break;

                            case G9Menu.WINDOW_MENU:
                                if (windowMenu != null) {
                                    windowMenu.setText(gm.getText());
                                    windowMenu.mergeMenu(gm);
                                    mergedMenus.add(windowMenu);
                                    merged = true;
                                }
                                break;

                            case G9Menu.HELP_MENU:
                                if (helpMenu != null) {
                                    helpMenu.setText(gm.getText());
                                    helpMenu.mergeMenu(gm);
                                    mergedMenus.add(helpMenu);
                                    merged = true;
                                }
                                break;
                            default:
                                break;

                        }
                        if (!merged) {
                            add(menues[i]);
                        }

                    } else {
                        add(menues[i]);
                    }
                }
            }

            if (my != null) {
                for (int i = mergePos; i < my.length; i++) {
                    add(my[i]);
                }

                // Remove any previously merged menues from menus
                // on this menubar not allready merged above.
                for (int i = 0; i < my.length; i++) {
                    if (!mergedMenus.contains(my[i])) {
                        ((G9Menu) my[i]).mergeMenu(null);
                    }
                }
            }

            setModal(modal);

        }
    }

    /**
     * Enable or disable all menubar menus and their menuitems
     * 
     * @param enable if <code>true</code> menus are
     */
    public synchronized void setEnabledWholeMenuBar(boolean enable) {
        if (myComponents == null) {
            return;
        }
        Iterator<Component> it = myComponents.iterator();
        while (it.hasNext()) {
            Object menu = it.next();
            if (menu instanceof G9Menu) {
                ((G9Menu) menu).setEnabledWholeMenu(enable);
            }
        }
    }

    /**
     * Enable or disable the modal status of this menu bar. Note that modality
     * is only meaningfull for the application main frame menu bar. For other
     * internal dialogs, the behavior is to enable or disable the internal
     * dialogs own menus.
     * 
     * @param isModal Puts this menu bar into modal mode if <code>true</code>
     *            otherwise the menu bar is put back to normal mode.
     */
    public synchronized void setModal(boolean isModal) {
        if (isModal == this.isModal)
            return;

        this.isModal = isModal;

        if (myComponents == null)
            return;

        Iterator<Component> it = myComponents.iterator();
        while (it.hasNext()) {
            Object menu = it.next();
            if (menu instanceof G9Menu) {
                ((G9Menu) menu).setModal(isModal);
            }
        }
    }

    /**
     * Return <code>true</code> if this menubar is modal otherwise
     * <code>false</code>.
     * 
     * @return Returns <code>true</code> if this menubar is modal otherwise
     *         <code>false</code>.
     */
    public synchronized boolean isModal() {
        return isModal;
    }

    /**
     * If set to <code>true</code> then consecutive separators are reduced to
     * one in all g9 menus added to this menubar.
     * 
     * @param remove The flag to set.
     */
    public synchronized void setRemoveConsecutiveSeparators(boolean remove) {
        for (int i = 0; i < getMenuCount(); i++) {
            if (getMenu(i) instanceof G9Menu) {
                ((G9Menu) getMenu(i))
                        .setRemoveConsecutiveSeparators(remove);
            }
        }
    }

    /**
     * Return the menus of this menu bar
     * 
     * @return the menus if this menu bar
     */
    public synchronized Component[] getG9Components() {

        if (myComponents == null) {
            return null;
        }
        Component[] retVal = new Component[myComponents.size()];
        for (int i = 0; i < retVal.length; i++) {
            retVal[i] = myComponents.get(i);
        }
        return retVal;
    }

    /**
     * Get the menu of given type.
     * 
     * @param menuType (missing javadoc)
     * @param customName - only used if <code>menuType</code> is
     *            <code>WINDOW_MENU</code>
     * @return menu if it exists;
     */
    public synchronized G9Menu getMenu(int menuType, String customName) {
        G9Menu menu = null;
        switch (menuType) {
            case G9Menu.CUSTOM_MENU:
                menu = (G9Menu) customNameToMenu.get(customName);
                break;
            case G9Menu.FILE_MENU:
                menu = fileMenu;
                break;
            case G9Menu.EDIT_MENU:
                menu = editMenu;
                break;
            case G9Menu.VIEW_MENU:
                menu = viewMenu;
                break;
            case G9Menu.WINDOW_MENU:
                menu = windowMenu;
                break;
            case G9Menu.HELP_MENU:
                menu = helpMenu;
                break;
            default:
                break;
        }
        return menu;
    }

    /**
     * Internal use. Toggles the edit menu actions (cut, copy, paste, select
     * all). Depending on the paramter, these are either enabled or disabled.
     * 
     * @param enable if <code>true</code> actions are enabled.
     */
    public synchronized void toggleEditMenu(boolean enable) {
        G9EditMenu menu = (G9EditMenu) getMenu(G9Menu.EDIT_MENU, null);
        if (menu != null) {
            menu.toggleEditComponents(enable);
        }
    }
    
}
