/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Insets;
import java.util.Date;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JTextField;
import javax.swing.border.BevelBorder;
import javax.swing.border.Border;
import javax.swing.border.TitledBorder;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.JTextComponent;
import javax.swing.text.PlainDocument;

import no.esito.log.Logger;
import no.g9.domain.G9Enums;
import no.g9.support.CustomDate;
import no.g9.support.FormatHelper;
import no.g9.support.G9Consts;

/**
 * Repository for non-generated resources.
 */
@SuppressWarnings({"rawtypes"})
public class FrameworkRepository {

    /**
     * Set the name if deleted lines should have a color different from red.
     */
    public static String ListblockLineDeletedColorName;

    private static Color ListblockLineDeletedColor;

    private static final Color ListblockLineNewColor = new Color(
            (float) (1 / 3.0), (float) (1 / 3.0), (float) 1.0);

    private static  Color ListblockLineChangedColor = Color.blue;

    private static final Font ListblockFont = new Font("Arial", Font.PLAIN, 12);

    /**
     * When setting a border frame on a radio group, the top inset on that frame
     * need to be supplied by the generator. For all other components, the
     * default top-frame-inset is used.
     */
    public static final int NO_RADIO_GROUP_TOP_INSET = -1;

    /** The logger */
    private static Logger log = Logger.getLogger(FrameworkRepository.class);

    /**
     * Constructor.
     */
    private FrameworkRepository() {
        // Empty.
    }

    /**
     * @param color (missing javadoc)
     */
    public static void setListblockLineChangedColor(Color color) {
    	ListblockLineChangedColor = color;
    }

    /** (missing javadoc) */
    static class DateDocument extends PlainDocument {

        /** Reference to property change support */
        protected transient java.beans.PropertyChangeSupport propertyChange;

        private String fieldFormat = "";

        private Date fieldDate = new Date();

        /**
         * Default constructor.
         */
        public DateDocument() {
            this(CustomDate.getDatetimeInputFormat(G9Consts.DT_DATE));
        }

        /**
         * @param aFormat (missing javadoc)
         */
        public DateDocument(String aFormat) {
            super();
            setFormat(aFormat);
        }

        /**
         * The addPropertyChangeListener method was generated to support the
         * propertyChange field.
         *
         * @param listener the property change listener
         */
        public synchronized void addPropertyChangeListener(
                java.beans.PropertyChangeListener listener) {
            getPropertyChange().addPropertyChangeListener(listener);
        }

        /**
         * The firePropertyChange method was generated to support the
         * propertyChange field.
         *
         * @param propertyName name of property
         * @param oldValue old property value
         * @param newValue new propert value
         */
        public void firePropertyChange(String propertyName, Object oldValue,
                Object newValue) {
            getPropertyChange().firePropertyChange(propertyName, oldValue,
                    newValue);
        }

        /**
         * Gets the date property (java.util.Date) value.
         *
         * @return The date property value.
         * @see #setDate
         */
        public Date getDate() {
            return fieldDate;
        }

        /**
         * Gets the property (java.lang.String) value.
         *
         * @return The property value.
         * @see #setFormat
         */
        public String getFormat() {
            return fieldFormat;
        }

        /**
         * Accessor for the propertyChange field.
         *
         * @return the property change field.
         */
        protected java.beans.PropertyChangeSupport getPropertyChange() {
            if (propertyChange == null) {
                propertyChange = new java.beans.PropertyChangeSupport(this);
            }
            return propertyChange;
        }

        @Override
        public void insertString(int offset, String str, AttributeSet attr)
                throws BadLocationException {
            if (str == null || str.length() == 0)
                return;
            if (getFormat() == null || getFormat().length() == 0) {
                System.out.println("-- No  set.");
                return;
            }
            if (offset >= getFormat().length())
                return;
            char aChar = str.charAt(0);
            char fChar = getFormat().charAt(offset);
            if (Character.isDigit(aChar) == Character.isLetter(fChar)) {
                if (!Character.isLetterOrDigit(fChar)) {
                    // if char is not a number
                } else {
                    super.insertString(offset, str, attr);
                    offset++;
                    if (offset < getFormat().length()) {
                        char bChar = getFormat().charAt(offset);
                        // if the next char in the is not a digit append the
                        // sign.
                        if (!Character.isLetterOrDigit(bChar)) {
                            if (getLength() == offset
                                    || getText(offset, 1).charAt(0) != bChar) {
                                super.insertString(offset, "" + bChar, attr);
                            }
                        }
                    } else {
                        if (offset == getFormat().length()) {
                            java.text.SimpleDateFormat sdf = new java.text.SimpleDateFormat(
                                    getFormat());
                            java.text.ParsePosition pos;
                            pos = new java.text.ParsePosition(0);
                            Date visited = sdf.parse(getText(0,
                                    offset), pos);
                            if (visited == null)
                                return;
                            setDate(visited);
                        }
                    }
                }
            }
        }

        @Override
        public void remove(int offset, int length) throws BadLocationException {
            if (getFormat() == null || getFormat().length() == 0) {
                System.out.println("-- No  set.");
                return;
            }
            //
            //
            super.remove(offset, length);
            char bChar = getFormat().charAt(offset);
            // if the next char in the is not a digit append the sign.
            if (!Character.isLetterOrDigit(bChar)) {
                super.remove(--offset, 1);
            }
        }

        /**
         * The removePropertyChangeListener method was generated to support the
         * propertyChange field.
         *
         * @param listener the property change listener
         */
        public synchronized void removePropertyChangeListener(
                java.beans.PropertyChangeListener listener) {
            getPropertyChange().removePropertyChangeListener(listener);
        }

        /**
         * Sets the date property (java.util.Date) value.
         *
         * @param date The new value for the property.
         * @see #getDate
         */
        public void setDate(Date date) {
            Date oldValue = fieldDate;
            fieldDate = date;
            firePropertyChange("date", oldValue, date);
        }

        /**
         * Sets the date format
         *
         * @param formatString String describing the new format
         * @see #getFormat()
         */
        public void setFormat(String formatString) {
            String oldValue = fieldFormat;
            fieldFormat = formatString;
            firePropertyChange("", oldValue, formatString);
        }
    }

    /** (missing javadoc) */
    static class UpperCaseDocument extends PlainDocument {

        private int maxLength = 0;

        /**
         * Constructor
         *
         * @param length max input length
         */
        public UpperCaseDocument(int length) {
            super();
            maxLength = length;
        }

        @Override
        public void insertString(int offs, String str, AttributeSet a)
                throws BadLocationException {
            if (str != null && maxLength > 0 && offs + str.length() > maxLength) {
                return;
            }
            if (str == null) {
                return;
            }
            char[] upper = str.toCharArray();
            for (int i = 0; i < upper.length; i++) {
                upper[i] = Character.toUpperCase(upper[i]);
            }
            super.insertString(offs, new String(upper), a);
        }
    }

    /** (missing javadoc) */
    static class LowerCaseDocument extends PlainDocument {

        private int maxLength = 0;

        /**
         * Constructor.
         *
         * @param length max input length
         */
        public LowerCaseDocument(int length) {
            super();
            maxLength = length;
        }

        @Override
        public void insertString(int offs, String str, AttributeSet a)
                throws BadLocationException {
            if (str != null && maxLength > 0 && offs + str.length() > maxLength) {
                return;
            }
            if (str == null) {
                return;
            }
            char[] lower = str.toCharArray();
            for (int i = 0; i < lower.length; i++) {
                lower[i] = Character.toLowerCase(lower[i]);
            }
            super.insertString(offs, new String(lower), a);
        }
    }

    /** (missing javadoc) */
    static class WholeNumberDocument extends PlainDocument {

        private int maxLength = 0;

        /**
         * Constructor.
         *
         * @param length max input length
         */
        public WholeNumberDocument(int length) {
            super();
            maxLength = length;
        }

        @Override
        public void insertString(int offs, String str, AttributeSet a)
                throws BadLocationException {
            if (str == null) {
                return;
            }
            if (maxLength > 0 && offs + str.length() > maxLength) {
                return;
            }
            char[] source = str.toCharArray();
            char[] result = new char[source.length];
            int j = 0;
            for (int i = 0; i < result.length; i++) {
                if (Character.isDigit(source[i])
                        || (i == 0 && source[i] == '-')) {
                    result[j++] = source[i];
                } else {
                    /*
                     * If this was a decimal number, we break if the character
                     * was a '.'
                     */
                    if (source[i] == '.') {
                        break;
                    }
                    // toolkit.beep();
                    System.err.println("insertString whole document: "
                            + source[i]);
                }
            }
            super.insertString(offs, new String(result, 0, j), a);
        }
    }

    /** (missing javadoc) */
    static class LengthDocument extends PlainDocument {

        private int maxLength = 0;

        /**
         * Constructor
         *
         * @param length max length
         */
        public LengthDocument(int length) {
            super();
            maxLength = length;
        }

        @Override
        public void insertString(int offs, String str, AttributeSet a)
                throws BadLocationException {
            if (str != null && maxLength > 0 && offs + str.length() > maxLength) {
                return;
            }
            super.insertString(offs, str, a);
        }
    }

    /** (missing javadoc) */
    static class DecimalNumberDocument extends PlainDocument {

        private int maxLength = 0;

        /**
         * Constructor
         *
         * @param length max length
         */
        public DecimalNumberDocument(int length) {
            super();
            maxLength = length;
        }

        @Override
        public void insertString(int offs, String str, AttributeSet a)
                throws BadLocationException {
            if (str == null) {
                return;
            }
            if (maxLength > 0 && offs + str.length() > maxLength) {
                return;
            }
            char[] source = str.toCharArray();
            char[] result = new char[source.length];
            int j = 0;
            boolean foundDot = false;
            for (int i = 0; i < result.length; i++) {
                if ((offs + i == 0 && source[i] == '-')
                        || Character.isDigit(source[i]) || source[i] == '.'
                        && !foundDot) {
                    if (source[i] == '.') {
                        foundDot = true;
                    }
                    result[j++] = source[i];
                } else {
                    java.awt.Toolkit.getDefaultToolkit().beep();
                    return;
                }
            }
            super.insertString(offs, str, a);
        }
    }

    /**
     * Titled border gets a default edge spacing, which we don't want. This
     * class fixes the problem.
     */
    private static final class G9TitleBorder extends TitledBorder {

        // Adjust top-border-inset when displaying a radio group.
        // Fixes SUP-154
        private final int radioGroupTopInset;



        G9TitleBorder(String title) {
            super(title);
            radioGroupTopInset = FrameworkRepository.NO_RADIO_GROUP_TOP_INSET;
        }

        G9TitleBorder(String title, int radioGroupTopInset) {
            super(title);
            this.radioGroupTopInset = radioGroupTopInset;
        }


        /**
         * We really don't want the default edge spacing!
         *
         * @see javax.swing.border.TitledBorder#paintBorder(java.awt.Component,
         *      java.awt.Graphics, int, int, int, int)
         */
        @Override
        public void paintBorder(Component c, Graphics g, int x, int y,
                int width, int height) {
            super.paintBorder(c, g, x - EDGE_SPACING, y - EDGE_SPACING, width
                    + 2 * EDGE_SPACING, height + 2 * EDGE_SPACING);
        }

        @Override
        public Insets getBorderInsets(Component c) {
            Insets insets = super.getBorderInsets(c);
            if (radioGroupTopInset != FrameworkRepository.NO_RADIO_GROUP_TOP_INSET) {
                insets.top = radioGroupTopInset;
            }
            if (log.isTraceEnabled()) {
                if (radioGroupTopInset != FrameworkRepository.NO_RADIO_GROUP_TOP_INSET) {
                    log.trace("Using supplied radio group to inset: " + radioGroupTopInset);
                } else {
                    log.trace("Using default top inset " + insets.top);
                }
            }
            return insets;
        }

    }

    /**
     * @param comp (missing javadoc)
     * @param borderType (missing javadoc)
     * @param title (missing javadoc)
     */
    static public void setBorder(JComponent comp, int borderType, String title) {
        setBorder(comp, borderType, title, FrameworkRepository.NO_RADIO_GROUP_TOP_INSET);
    }

    /**
     * Internal use, sets border on a component.
     *
     * @param comp (missing javadoc)
     * @param borderType (missing javadoc)
     * @param title (missing javadoc)
     * @param radioGroupTopInset (missing javadoc)
     */
    static public void setBorder(JComponent comp, int borderType, String title, int radioGroupTopInset) {
        Border border = null;
        if (log.isTraceEnabled()) {
            log.trace("Setting border on " + comp.getName() + ". Border type: "
                    + borderType + ", title: " + title
                    + ", radioGroupTopInset: " + radioGroupTopInset);
        }
        if (borderType > 1 && !title.equals("")) {
            border = new G9TitleBorder(title, radioGroupTopInset);
            if (borderType == 3) {
                ((TitledBorder) border).setBorder(new BevelBorder(
                        BevelBorder.LOWERED));
            }
        } else {
            switch (borderType) {
            case 1:
                border = BorderFactory.createEmptyBorder();
                break;
            case 2:
                border = BorderFactory.createEtchedBorder();
                break;
            case 3:
                border = BorderFactory.createBevelBorder(BevelBorder.LOWERED);
                break;
            default:
                break;
            }
        }
        comp.setBorder(border);
    }

    /**
     * Creates an uper case text component
     *
     * @param text the text component
     * @param length length of text
     */
    public static void setUpperCase(JTextComponent text, int length) {
        text.setDocument(new UpperCaseDocument(length));
    }

    /**
     * Creates a lower case text component
     *
     * @param text the text component
     * @param length max length
     */
    public static void setLowerCase(JTextComponent text, int length) {
        text.setDocument(new LowerCaseDocument(length));
    }

    /**
     * Creates a whole number text component
     *
     * @param text the text component
     * @param length max length
     */
    public void setWholeNumberField(JTextComponent text, int length) {
        text.setDocument(new WholeNumberDocument(length));
    }

    /**
     * Internal use.
     *
     * @param datatype (missing javadoc)
     * @param text (missing javadoc)
     * @param caseConvertion (missing javadoc)
     * @param enums (missing javadoc)
     * @param enumClass (missing javadoc)
     * @param displayrule (missing javadoc)
     * @param maxLength (missing javadoc)
     * @param blankWhenZero (missing javadoc)
     */
    @SuppressWarnings("deprecation")
	public static void setDatatype(int datatype, JTextComponent text, int caseConvertion,
            G9Enums enums, Class enumClass,
            String displayrule, int maxLength,
            boolean blankWhenZero) {
        String outputFormat;
        String inputFormat;
        int sign;
        boolean useJoda = false;
        boolean useJavaTime = false;
        boolean useG9Numeric = false;
        if (datatype > 3000) {
        	useG9Numeric = true;
        	datatype -= 3000;
        }
        else if (datatype > 2000) {
        	useJavaTime = true;
        	datatype -= 2000;
        }
        else if (datatype > 1000) {
            // This is a flag for Joda time
            useJoda = true;
            datatype -= 1000;
        }
        switch (datatype) {
        case G9Consts.DT_TEXT:
        case G9Consts.DT_VARTEXT:
        case G9Consts.DT_LONGTEXT:
            text.setDocument(new TextDocument(datatype, null, null, maxLength, caseConvertion));
            break;
        case G9Consts.DT_SHORTINT:
        case G9Consts.DT_LONGINT:
        case G9Consts.DT_LONGLONG:
        case G9Consts.DT_NUMERIC:
        case G9Consts.DT_REAL:
        case G9Consts.DT_DOUBLE:
            outputFormat = FormatHelper.getNumericFormat(displayrule);
            inputFormat = FormatHelper.getNumericInputFormat(outputFormat);
            sign = FormatHelper.getSign(displayrule);
            maxLength = inputFormat.length();
            text.setDocument(new NumberDocument(datatype, inputFormat,
                    outputFormat, maxLength, sign, blankWhenZero, useG9Numeric));
            break;
        case G9Consts.DT_DATE:
        case G9Consts.DT_TIME:
        case G9Consts.DT_TIMESTAMP:
            inputFormat = CustomDate.getDatetimeInputFormat(datatype);
            outputFormat = FormatHelper.getDatetimeFormat(displayrule);
            if (outputFormat == null) {
                outputFormat = CustomDate.getDatetimeOutputFormat(datatype);
            }
            if (useJoda) {
                if (datatype == G9Consts.DT_TIMESTAMP) {
                    text.setDocument(new DateTimeDocument(datatype, inputFormat, outputFormat, maxLength));
                }
                else {
                    text.setDocument(new DateMidnightDocument(datatype, inputFormat, outputFormat, maxLength));
                }
            } else if (useJavaTime) {
            	if (datatype == G9Consts.DT_DATE) {
                    text.setDocument(new LocalDateDocument(datatype, inputFormat, outputFormat, maxLength));
            	} else if (datatype == G9Consts.DT_TIME) {
                    text.setDocument(new LocalTimeDocument(datatype, inputFormat, outputFormat, maxLength));
            	} else { // G9Consts.DT_TIMESTAMP
                    text.setDocument(new LocalDateTimeDocument(datatype, inputFormat, outputFormat, maxLength));
            	}
            } else {
                text.setDocument(new no.g9.client.support.DateDocument(datatype, inputFormat, outputFormat, maxLength));
            }
            break;
        case G9Consts.DT_ENUMERATION:
            text.setDocument(new EnumeratorDocument( datatype, null, null, maxLength,enums, enumClass));
            break;
        default: // others
            text.setDocument(new G9Document(datatype, null, null, maxLength));
            break;
        }
    }

    /**
     * Internal use only! Returns the approriate G9Document based on the
     * parameters
     *
     * @param datatype integer describing the datatype
     * @param caseConvertion (missing javadoc)
     * @param enums (missing javadoc)
     * @param enumClass (missing javadoc)
     * @param displayrule (missing javadoc)
     * @param maxLength (missing javadoc)
     * @param blankWhenZero (missing javadoc)
     * @return the appropriate G9Document
     */
    public static G9Document getDocument(int datatype, int caseConvertion,
            G9Enums enums, Class enumClass,
            String displayrule, int maxLength,
            boolean blankWhenZero) {

        JTextField tmp = new JTextField();
        setDatatype(datatype, tmp, caseConvertion, enums, enumClass, displayrule, maxLength, blankWhenZero);
        return (G9Document) tmp.getDocument();
    }

    /**
     * Internal use. Gets the listblock color
     *
     * @param name the name of the color
     * @return the color
     */
    public static Color getColor(String name) {
        if (name.equals("ListblockLineDeletedColor")) {
            if (ListblockLineDeletedColorName == null) {
                return Color.red;
            }
            if (ListblockLineDeletedColor == null) {
                ListblockLineDeletedColor = Color
                        .getColor(ListblockLineDeletedColorName);
            }
            return ListblockLineDeletedColor;
        } else if (name.equals("ListblockLineNewColor")) {
            return ListblockLineNewColor;
        } else if (name.equals("ListblockLineChangedColor")) {
            return ListblockLineChangedColor;
        }
        return null;
    }

    /**
     * Gets the font to use in listblock
     *
     * @param name the name of the listblock font
     * @return the font
     */
    public static Font getFont(String name) {
        if (name.equals("ListblockFont")) {
            return ListblockFont;
        }
        return null;
    }

}
