/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.util.List;
import java.util.Set;
import java.util.Map;

import no.g9.exception.G9ClientException;

/** Methods available to all listblocks. */
@SuppressWarnings("rawtypes")
public interface Listblock {

    /**
     * Returns a list of strings with the data in the listblock as html tables.
     *
     * @param maxRows The number of rows in the listblock for a split.
     * @param incHidden If true, the table will Include hidden fields in output.
     * @param excludedColumns String set of columns to be excloded. They must be
     *            on the form "objselection.attribute"
     * @param tagOptions A map containing the following keys: TH - Added to each
     *            TH. TD - Added to each TD. TABLE - added to table tag. TR -
     *            added to each TR.
     * @return Complete html.
     */
    public List getAsHTMLTable(int maxRows, boolean incHidden,
            Set excludedColumns, Map tagOptions);

    /**
     * Setting update when manipulated to false will not make the GUI update
     * itself when the data in the listblock is modified using
     * insert/update/delete calls. When data is beeing bulk loaded into the
     * datastructure, it might be preferably to call
     * setUpdateWhenManipulated(false) prior to inserting data and call
     * setUpdateWhenManipulated(true) afterwards. A normal use of would be:
     *
     * <pre>
     *    listblockMethodsObj.setUpdateWhenManipulated(false);
     *    ...do all inserts
     *    listblockMethodsObj.setUpdateWhenManipulated(true);
     *    listblockMethodsObj.forceClean();
     * </pre>
     *
     * @param f Default setting is true.
     */
    public void setUpdateWhenManipulated(boolean f);

    /**
     * Returns number of lines in the listblock, regardless of the
     * listblocklines status.
     *
     * @return Number of elements.
     */
    public int size();

    /**
     * Sets which popup menu to be used for this listblock. It is required that
     * the dialog had a popup menu to start with. If argument sent in is the
     * empty string, the the popup menu will be disabled. (Same effect will
     * occure if the menuName is not a valid popup name.)
     *
     * @param menuName The menu name to use.
     */
    public void setPopupMenu(String menuName);

    /**
     * Returns true if the component is focused.
     *
     * @return True if component has focus, false if not.
     */
    public boolean hasFocus();

    /**
     * Request focus to this component. It is not granted that this component
     * will be granted the focus after this call.
     */
    public void requestFocus();

    /**
     * Returns the last row in the listblock
     *
     * @return The row number, Index starting at 0.
     */
    public int getLastRowLine();

    // /**
    // * Sets the size of the columns of the given JTable to an
    // * appropriate size based on the size of the column headings and
    // * the contents of the column.
    // *
    // */
    // public void setBestFitColumnWidths();

    /**
     * Set or enables listblock editing within the listblock (has hidden edit
     * fields). Only valid if it was editable in the first place).
     *
     * @param f If true, the listblock may be edited in, if false, editing is
     *            disabled.
     */
    public void setListblockEditing(boolean f);

    /**
     * Scrolls the listblock to make sure that a given row is shown to the user.
     *
     * @param row The row number, starting with Index 0.
     */
    public void makeRowVisible(int row);

    /**
     * Returns the set of role names displayed by the listblock.
     *
     * @return a Set containing role names displayable by the listblock.
     */
    public Set getRoleNamesInUse();

    /**
     * Inserts a listblock line at the end of the list block.
     *
     * @param listBlockLine the line to insert.
     */
    public void insertNewListblockLine(ListblockLine listBlockLine);

    /**
     * Returns a new list block line object. The returned object is initially
     * empty.
     *
     * @return an empty list block line.
     */
    public ListblockLine getNewListblockLine();

    /**
     * Forwards call to the hook method before insert.
     *
     * @param line - listblock line to insert
     * @return true if the line should be inserted.
     */
    public boolean callHooksBeforeInsert(ListblockLine line);

    /**
     * Returns a list of ListblockLine containing the changed lines in the
     * listblock.
     *
     * @return a List of changed list block lines
     */
    public List getChangedLines();

    /**
     * Gets the obtained version of all changed lines.
     *
     * @return a List of changed lines with updated values.
     */
    public List obtainChangedLines();

    /**
     * Returns a list of ListblockLine containing all lines in the listblock.
     *
     * @return a List with all list block lines
     */
    public List allLines();

    /**
     * Marks all lines as not deleted, changed or new, without copying of old
     * data
     */
    public void forceClean();

    /**
     * Purges all lines from the list block.
     */
    public void purgeAllLines();

    /**
     * Unselects any selected line from the listblock.
     */
    public void clearSelectedLines();

    /**
     * Returns the position of the selected line in the list.
     *
     * @return The row number of the selected line, or -1 if no lines are
     *         selected.
     */
    int getSelectedLine();

    /**
     * Returns a list containig the selected lines.
     *
     * @return a List of selected <code>ListblockLine</code>s.
     */
    public List getSelectedLinesObject();

    /**
     * Returns a list of indexes of all selected lines in the list.
     *
     * @return An array of integers containing the indices of all selected
     *         lines, or an empty array if no lines are selected.
     */
    int[] getSelectedLines();

    /**
     * Selects a line in the list. The previous selection is cleared.
     *
     * @param line The row to select.
     */
    void setSelectedLine(int line);

    /**
     * Selects a line in the list. The previous selection is cleared.
     *
     * @param line The row to select.
     */
    void setSelectedLine(ListblockLine line);

    /**
     * Selects a set of lines in the list. The previous selection is cleared.
     *
     * @param lines An array of integers containing the indices of all lines to
     *            select.
     */
    void setSelectedLines(int lines[]);

    /**
     * Returns the G9Table responsible for displaying this table
     *
     * @return the G9Table displaying this table's data.
     */
    G9Table getTable();

    /**
     * Hides the specified column. If the column is already hidden, nothing
     * happens.
     *
     * @param attributeName a String denoting the fully qualified attribute
     *            name, e.g. "roleName.attr"
     * @throws G9ClientException if the specified column does not exist.
     */
    public void hideColumn(String attributeName);

    /**
     * Hides the specified colunms.
     *
     * @param attributeNames a List of attribute names of columns to hide.
     * @throws G9ClientException if the specified column does not exist.
     * @see #hideColumn(String)
     */
    public void hideColumn(List attributeNames);

    /**
     * Shows the specified column. If the column is already visible, nothing
     * happens.
     *
     * @param attributeName a String denoting the fully qualified attribute
     *            name, e.g. "roleName.attr"
     * @throws G9ClientException if the specified column does not exist.
     */
    public void showColumn(String attributeName);

    /**
     * Shows the specified columns.
     *
     * @param attributeNames a List of attribute names of columns to hide.
     * @throws G9ClientException if the specified column does not exist.
     * @see #showColumn(String)
     */
    public void showColumn(List attributeNames);

    /**
     * Sets the header value of the specified column. The header value is the
     * object use to display the header (typically a String).
     *
     * @param attributeName a String denoting the fully qualified attribute
     *            name, e.g. "roleName.attr"
     * @param value the header value.
     * @throws G9ClientException if the specified column does not exist.
     */
    public void setColumnHeaderValue(String attributeName, String value);

    /**
     * Recursivly clears the edit fields of this table.
     */
    public void clearEditFields();

    /**
     * Returns the line that keeps an instance of the specified domain object,
     * or <code>null</code> if no such line is found. The search is performed
     * from the top of the listblock, and returns immediately if a match is
     * found.
     *
     * @param domainObject the domain object to search for
     * @return the line holding the domain object or <code>null</code> if no
     *         such line is found.
     */
    public ListblockLine findInList(Object domainObject);

    /**
     * Set the checkForeginNode property. If the <code>doCheck</code> parameter
     * differs from <code>null</code> the global checkForeingNode property is
     * overridden for this Listblock instance, and the overridden value will be
     * used when checkRowSelect method on the listblock role's node determines
     * which nodes to check. In order to use the application's checkForegingNode
     * property, pass a <code>null</code>-reference to this method.
     *
     * @param doCheck the value of the checkForeginNodes property.
     * @see #checkForeignNodes()
     * @see ApplicationMethods#setCheckForeignNodes(boolean)
     * @see ApplicationMethods#checkForeignNodes()
     */
    public void setCheckForeignNodes(Boolean doCheck);

    /**
     * Returns the boolean value of the checkForeignNodes property. If the
     * property is not set, the application's value is returned.
     *
     * @return the value of the checkForeignNodes property.
     * @see #setCheckForeignNodes(Boolean)
     * @see ApplicationMethods#setCheckForeignNodes(boolean)
     * @see ApplicationMethods#checkForeignNodes()
     */
    public boolean checkForeignNodes();

    /**
     * Returns the listblock line with the specified row number, or <code>
     * null</code>
     * if no such row exists.
     *
     * @param rowIndex the row index
     * @return the listblock line with the specified index.
     */
    public ListblockLine getLine(int rowIndex);

}
