/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import no.g9.domain.DomainUtil;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;

/**
 * Support class that holds information about an object and it's role.
 */
@SuppressWarnings({"unchecked", "rawtypes"})
public class RoleObject {

    private AbstractNode node;

    private String roleName;

    /** used to detect loops in associations. */
    boolean visited = false;

    // key = associationRoleObject,
    // value = associationObject
    private Map associations = new HashMap();

    private AssociationAccess parentAssocciation = null;

    // key = listBlock,
    // value = Class[] params
    private Map listBlocks = new HashMap();

    // key = roleName,
    // value = listBlock
    private List simpleBlocks = new ArrayList();

    private boolean isUpRelated;

    /** The minimum constant value for a relation type */
    protected static final int MIN_RELATION_TYPE = 1;

    /** The maximum constant value for a relation type */
    protected static final int MAX_RELATION_TYPE = 4;

    /** Constant denoting a one-relation */
    public static final int ONE_RELATED = 1;

    /** Constant denoting a one - in - many relation (e.g. up-related) */
    public static final int ONE_IN_MORE_RELATED = 2;

    /** Constant denoting a many-relation */
    public static final int MORE_RELATED = 3;

    /** Constant denoting a parent-association */
    public static final int PARENT_RELATION = 4;

    /**
     * Instanciates a new RoleObject with the specified role name.
     *
     * @param roleName the roleName representet by this role object.
     */
    public RoleObject(String roleName) {
        this.roleName = roleName;
    }

    /**
     * Internal use! Sets the abstract node associated with this role object.
     *
     * @param node (missing javadoc)
     */
    public void setNode(AbstractNode node) {
        this.node = node;
    }

    /**
     * Internal use! Gets the abstract node associated with this role object.
     *
     * @return the abstract node
     */
    public AbstractNode getNode() {
        return node;
    }

    /**
     * Adds an association to the list of associated objects.
     *
     * @param associationRoleObject The g9 specified role name
     * @param associationName The name of the association in the domain model
     * @param associationType the type of association
     * @param useAccessMethod if true, the getter and setter is used to access
     *            the association
     * @param paramType the param type of the association.
     */
    public void addAssociation(Object associationRoleObject,
            String associationName, int associationType,
            boolean useAccessMethod, Class paramType) {
        AssociationAccess anAssociation = new AssociationAccess(
                associationName, associationType, useAccessMethod,
                paramType);
        if (associationType == PARENT_RELATION) {
            parentAssocciation = anAssociation;
        } else {
            associations.put(associationRoleObject, anAssociation);
        }
    }

    /**
     * Returns the parent domain instanceo of the specified child instance
     *
     * @param childObject the child instance
     * @return the parent of the child instance
     */
    public Object getParentDomainObject(Object childObject) {
        Object parentDomainObject = null;
        if (parentAssocciation != null) {
            parentDomainObject = parentAssocciation
                    .getAssociationOnType(childObject);
        }
        return parentDomainObject;
    }

    /**
     * Detects if there is associative loop
     *
     * @return <code>true</code> if there is loop
     */
    public boolean hasLoopInAssociations() {
        visited = true;
        boolean loopFound = false;
        Iterator it = associations.keySet().iterator();
        while (it.hasNext() && !loopFound) {
            RoleObject aRoleObject = (RoleObject) it.next();
            if (aRoleObject.visited || aRoleObject.hasLoopInAssociations()) {
                loopFound = true;
            }
        }
        visited = false;
        return loopFound;
    }

    /**
     * Gets the path to from this role object to the specified role object, and
     * returns a <code>List</code> where each sucessive entry represents a node
     * on the path to the target. E.g: if the path to 'C', is A -&gt; B -&gt; C, and
     * this node is 'A', the returned list contains the elements {A, B, C}, in
     * that order. If no path exists, <code>null</code> is returned.
     *
     * @param targetRoleName the name of the target node
     * @return a list of role objects representing the root to the specified
     *         target node.
     */
    public List getPathToRole(String targetRoleName) {
        List path = new ArrayList();

        // base case - target found
        if (getRoleName().equals(targetRoleName)) {
            path.add(this);
            return path;
        }

        // iterate associations, call recursivly on each association
        Iterator it = associations.keySet().iterator();
        while (it.hasNext()) {
            RoleObject anAssociation = (RoleObject) it.next();
            List possiblePath = anAssociation
                    .getPathToRole(targetRoleName);
            if (possiblePath != null) {
                path.add(this);
                path.addAll(possiblePath);
                return path; // path found, return.
            }
        }
        return null; // <- no path found
    }

    /**
     * Returns a Set of Sets with all relations. E.g. if domain class A is
     * associated with B and C, the set {{b1, b2, bn}, {c1, c2, cn}} is
     * returned. Note that the set holds all associated objects, also those
     * whose cardinality is one.
     *
     * @param actualObject (missing javadoc)
     * @return set of associations
     */
    public Set getAssociations(Object actualObject) {
        Iterator it = associations.values().iterator();
        Set associationSet = new HashSet();
        while (it.hasNext()) {
            AssociationAccess anAssosiation = (AssociationAccess) it
                    .next();
            associationSet.add(anAssosiation.getAssociation(actualObject));
        }
        return associationSet;
    }

    /**
     * Returns the role name of this role
     * @return the name of this role
     */
    public String getRoleName() {
        return roleName;
    }

    /**
     * Gets the associationAcces instance used for accessing the associations on
     * the specified role object
     *
     * @param associationRole the association role
     * @return the association access instance
     */
    public AssociationAccess getAssociationAccess(
            RoleObject associationRole) {
        return (AssociationAccess) associations.get(associationRole);
    }

    /**
     * Returns an iterator with the AssociationAccess instances of this role
     *
     * @return an iterator with assiciation access objects.
     */
    public Iterator getAssociationRoles() {
        return associations.keySet().iterator();
    }

    /**
     * Adds a list block.
     *
     * @param listBlock the list block
     * @param params the parameters excepcted by the listblock line
     */
    public void addListBlock(Object listBlock, Class[] params) {
        listBlocks.put(listBlock, params);
    }

    /**
     * Returns a list of list blocks that displays the domain object
     *
     * @return All listblocks that displays the domain object.
     */
    public Map getListBlocks() {
        return listBlocks;
    }

    /**
     * Adds a simple block to the list of simple blocks that can display the
     * domain object.
     *
     * @param simpleBlock the simple block
     */
    public void addSimpleBlock(Object simpleBlock) {
        simpleBlocks.add(simpleBlock);
    }

    /**
     * Gets the list of simple blocks that can display the domain object
     *
     * @return a list of simple blocks.
     */
    public List getSimpleBlocks() {
        return simpleBlocks;
    }

    /**
     * Two role objects are considered equal if they have the same role name.
     *
     * @param obj the reference object with which to compare.
     * @return <code>true</code> if this object has the same role name as the
     *         obj argument; <code>false</code> otherwise.
     */
    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof RoleObject)) {
            return false;
        }
        return getRoleName().equals(((RoleObject) obj).getRoleName());
    }

    @Override
    public int hashCode() {
        return getRoleName().hashCode();
    }

    /**
     * Gets the set of arguments for a list block
     *
     * @param aListBlock (missing javadoc)
     * @return an array of arguments
     */
    public Class[] getListBlockParams(Object aListBlock) {
        return (Class[]) listBlocks.get(aListBlock);
    }

    /**
     * Internal use. Returns a list of objects that can be used in reflection
     * code.
     *
     * @param domainObject the domain object
     * @param params array of parameters
     * @return the list of object arguments
     */
    public List getListBlockArgs(Object domainObject, Class[] params) {
        Set paramSet = new HashSet();
        paramSet.addAll(Arrays.asList(params));
        return buildListBlockArgs(domainObject, paramSet);
    }

    private Class getSuperClassFromSet(Object domainObject, Set params) {
        Class domainClass = DomainUtil.getDomainClass(domainObject);
        Class foundSuperClass = null;
        Iterator it = params.iterator();
        while (it.hasNext() && foundSuperClass == null) {
            Class possibleSuperClass = (Class) it.next();
            if (possibleSuperClass.isAssignableFrom(domainClass)) {
                foundSuperClass = possibleSuperClass;
            }
        }

        return foundSuperClass;
    }

    private List buildListBlockArgs(Object domainObject, Set params) {
        if (params.size() == 0) {
            return null;
        }

        List myAssociations = null;

        Class argClass = getSuperClassFromSet(domainObject, params);

        if (argClass != null) {
            myAssociations = new ArrayList();
            myAssociations.add(domainObject);
            params.remove(argClass);
        }

        // Check all associations:
        Iterator it = associations.keySet().iterator();
        while (it.hasNext()) {
            RoleObject associatedRole = (RoleObject) it.next();

            AssociationAccess access = (AssociationAccess) associations
                    .get(associatedRole);

            // If association is more - related, we cannot know which
            // one to display in the listblock!
            if (access.ASSOCIATION_TYPE == MORE_RELATED) {
                continue;
            }

            Set associationSet = access.getAssociation(domainObject);
            // Thus, the associationSet should only contain at most one
            // member at this point.
            if (associationSet != null && associationSet.size() > 1) {
                Object[] args = { getClass(), associatedRole.getRoleName() };

                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CF_MORE_RELATION_IN_LISTBLOCK, args);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(msg);
            }

            Iterator tmpIt = null;
            if (associationSet != null) {
                tmpIt = associationSet.iterator();
            }
            Object associatedObject = null;
            List associationList = null;
            assert (associationSet != null);
            if (tmpIt != null && tmpIt.hasNext()) {
                associatedObject = associationSet.iterator().next();
                associationList = associatedRole.buildListBlockArgs(
                        associatedObject, params);
            }

            if (associationList != null) {
                if (myAssociations == null) {
                    myAssociations = new ArrayList();
                }
                myAssociations.addAll(associationList);
            }
        }

        return myAssociations;
    }

    /**
     * Internal use Gets a list of arguments used in reflection to access a
     * listblock.
     *
     * @param actualObject the actual object
     * @param roleObject the role object
     * @param aListBlock a list block
     * @param params the parameter array
     * @param args the argument array
     * @param curIndex the index of the current argument
     */
    public void getListBlockArgs(Object actualObject,
            RoleObject roleObject, Listblock aListBlock, Class[] params,
            Object[] args, int curIndex) {
        if (DomainUtil.getDomainClass(actualObject) == params[curIndex]
                && aListBlock.getRoleNamesInUse().contains(
                        roleObject.getRoleName())) {
            args[curIndex++] = actualObject;
        }
        if (curIndex < args.length) {
            Iterator associationRoles = associations.keySet().iterator();
            while (associationRoles.hasNext()) {
                RoleObject anAssociation = (RoleObject) associationRoles
                        .next();
                if (aListBlock.getRoleNamesInUse().contains(
                        anAssociation.getRoleName())) {
                    AssociationAccess associationAcces = (AssociationAccess) associations
                            .get(anAssociation);
                    Set argument = associationAcces
                            .getAssociation(actualObject);
                    if (argument != null && argument.size() == 1) {
                        for (int i = curIndex; i < args.length; i++) {
                            Object foundArgument = argument.iterator()
                                    .next();
                            if (args[i] == null
                                    && params[i] == DomainUtil
                                            .getDomainClass(foundArgument)) {
                                args[i] = foundArgument;
                            }
                        }

                    }
                }
            }
        }

        if (curIndex < args.length) {
            Iterator associationRoleIterator = associations.keySet()
                    .iterator();
            while (associationRoleIterator.hasNext()) {
                RoleObject anAssociationRole = (RoleObject) associationRoleIterator
                        .next();
                AssociationAccess anAssociation = (AssociationAccess) associations
                        .get(anAssociationRole);
                Set possiblePathToArumentsSet = anAssociation
                        .getAssociation(actualObject);
                if (possiblePathToArumentsSet.size() == 1) {
                    Object possiblePathToArgument = possiblePathToArumentsSet
                            .iterator().next();
                    getListBlockArgs(possiblePathToArgument,
                            anAssociationRole, aListBlock, params, args,
                            curIndex);
                }
            }
        }
    }

    /**
     * Class used to access a relation.
     */
    public class AssociationAccess {
        private boolean useAccessMethod;

        private Class[] param;

        private String associationName;

        private String getPrefix = "get";

        private String setPrefix = "set";

        private final int ASSOCIATION_TYPE;

        /**
         * Default constructor. Creates a default association access object with
         * no association-name and -object, no cardianlity, and the use access
         * method property set to false.
         */
        public AssociationAccess() {
            ASSOCIATION_TYPE = 0;
        }

        /**
         * Instanciates a new association access object.
         *
         * @param associationName the domain name of the association
         * @param associationType constant
         * @param useAccessMethod if <code>true</code>, methods will be used to
         *            access this association.
         * @param paramType the parameter type
         */
        public AssociationAccess(String associationName,
                final int associationType, boolean useAccessMethod,
                Class paramType) {
            if (associationType < MIN_RELATION_TYPE
                    || associationType > MAX_RELATION_TYPE) {
                Object[] args = { getClass(),
                        Integer.toString(associationType) };
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CF_ILLEGAL_ASSOCIATION_TYPE, args);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(msg);
            }
            this.associationName = associationName;
            ASSOCIATION_TYPE = associationType;
            this.useAccessMethod = useAccessMethod;
            param = new Class[] { paramType };
        }

        /**
         * Returns the association type of this role object
         *
         * @return the association type
         */
        public int getAssociationType() {
            return ASSOCIATION_TYPE;
        }

        /**
         * Gets the association as a Set
         *
         * @param actualObject the object to get the association from
         * @return the Set with the association
         */
        public Set getAssociation(Object actualObject) {
            return useAccessMethod ? getAssociationUsingMethod(actualObject)
                    : getAssociationUsingField(actualObject);
        }

        /**
         * Sets the association on the actual object to the association value
         * object
         *
         * @param actualObject the (domain) object) where the association should
         *            be set
         * @param associationValue the value of the association to set
         */
        public void setAssociation(Object actualObject,
                Object associationValue) {
            if (useAccessMethod) {
                setAssociationUsingMethod(actualObject, associationValue);
            } else {
                setAssociationUsingField(actualObject, associationValue);
            }
        }

        /**
         * Returns all changed lines from the specified listblock.
         *
         * @param theListBlock the listblock
         * @return a List of changed lines.
         */
        public List obtainChangedLinesFromListBlock(Listblock theListBlock) {
            return theListBlock.obtainChangedLines();
        }

        /**
         * Returns all lines from the specified listblock.
         *
         * @param theListBlock the listblock
         * @return a List of all lines from the listblock
         */
        public List obtainLinesFromListBlock(Listblock theListBlock) {
            return theListBlock.allLines();
        }

        // //////// Get and set methods /////////////////////

        /**
         * @return Returns the associationName.
         */
        public String getAssociationName() {
            return associationName;
        }

        /**
         * @param associationName The associationName to set.
         */
        public void setAssociationName(String associationName) {
            this.associationName = associationName;
        }

        /**
         * @return Returns the useAccessMethod.
         */
        public boolean isUseAccessMethod() {
            return useAccessMethod;
        }

        /**
         * @param useAccessMethod The useAccessMethod to set.
         */
        public void setUseAccessMethod(boolean useAccessMethod) {
            this.useAccessMethod = useAccessMethod;
        }

        @Override
        public String toString() {
            StringBuffer sb = new StringBuffer(roleName + " {");
            Iterator it = associations.keySet().iterator();
            while (it.hasNext()) {
                RoleObject anAssociation = (RoleObject) it.next();
                if (anAssociation.hasLoopInAssociations()) {
                    sb.append("associative loop");
                } else {
                    sb.append(anAssociation);
                }
                sb.append(", ");
            }
            return sb.substring(0, sb.length() - 2) + "}";
        }

        // //////////////// Helper methods ///////////////////////

        private Set getAssociationUsingField(Object actualObject) {
            Object anAssociation = getAssociationTypeField(actualObject, associationName);

            if (!(anAssociation instanceof Set)) {
                Set tmp = new HashSet();
                if (anAssociation != null) {
                    tmp.add(anAssociation);
                }
                anAssociation = tmp;
            }

            return (Set) anAssociation;
        }

        private Set getAssociationUsingMethod(Object actualObject) {
            Object anAssociation = getAssociationOnTypeMethod(actualObject, associationName);

            if (!(anAssociation instanceof Set)) {
                Set tmp = new HashSet();
                if (anAssociation != null) {
                    tmp.add(anAssociation);
                }
                anAssociation = tmp;
            }

            return (Set) anAssociation;

        }

        private void setAssociationUsingMethod(Object actualObject,
                Object associationValue) {
            Object[] args = new Object[] { associationValue };
            Exception ex = null;
            try {
                Method m = actualObject.getClass().getMethod(
                        asSetMethod(associationName), param);
                // Work-around for bug #4533479 in java
                m.setAccessible(true);
                m.invoke(actualObject, args);
            } catch (IllegalAccessException e) {
                ex = e;
            } catch (InvocationTargetException e) {
                ex = e;
            } catch (NoSuchMethodException e) {
                ex = e;
            }
            if (ex != null) {
                Object[] msgArgs = new Object[] { getClass(),
                        asSetMethod(associationName) + "()",
                        actualObject.getClass(), ex.getMessage() };
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD, msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(ex, msg);
            }

        }

        private void setAssociationUsingField(Object actualObject, Object associationValue) {
            Exception ex = null;
            try {
                Field f = actualObject.getClass().getField(associationName);
                // Work-around for bug #4533479 in java
                f.setAccessible(true);
                f.set(actualObject, associationValue);
            } catch (IllegalAccessException e) {
                ex = e;
            } catch (NoSuchFieldException e) {
                ex = e;
            }
            if (ex != null) {
                Object [] msgArgs = new Object[] {
                        getClass(), associationName,
                        actualObject.getClass(), ex.getMessage()
                };
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD, msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(ex, msg);
            }
        }

        private String asGetMethod(String name) {
            StringBuffer methodName = new StringBuffer(name);
            methodName.setCharAt(0, Character.toUpperCase(methodName
                    .charAt(0)));
            return getPrefix + methodName.toString();
        }

        private String asSetMethod(String name) {
            StringBuffer methodName = new StringBuffer(name);
            methodName.setCharAt(0, Character.toUpperCase(methodName
                    .charAt(0)));
            return setPrefix + methodName.toString();
        }

        /**
         * Gets the association based on the type of fromObject
         *
         * @param fromObject the type of the association
         * @return the association object
         */
        public Object getAssociationOnType(Object fromObject) {
            if (associationName.contains(".")) {
                Object toObject = fromObject;
                for (String path : associationName.split("\\.")) {
                    toObject = useAccessMethod ? getAssociationOnTypeMethod(toObject, path) : getAssociationTypeField(toObject, path);
                }
                return toObject;
            }
            return useAccessMethod ? getAssociationOnTypeMethod(fromObject, associationName) : getAssociationTypeField(fromObject, associationName);
        }

        private Object getAssociationTypeField(Object actualObject, String assocName) {
            Exception ex = null;
            try {
                Field f = actualObject.getClass().getField(assocName);
                // Work-around for bug #4533479 in java
                f.setAccessible(true);
                return f.get(actualObject);
            } catch (IllegalAccessException e) {
                ex = e;
            } catch (NoSuchFieldException e) {
                ex = e;
            }
            // If here, some exception got caught...
            Object[] msgArgs = new Object[] {
                    getClass(), assocName,
                    actualObject.getClass(), ex.getMessage()
            };
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD, msgArgs);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9ClientFrameworkException(ex, msg);
        }

        private Object getAssociationOnTypeMethod(Object actualObject, String assocName) {
            Exception ex = null;
            try {
                Method m = actualObject.getClass().getMethod(
                        asGetMethod(assocName), (Class[]) null);
                // Work-around for bug #4533479 in java
                m.setAccessible(true);
                return m.invoke(actualObject, (Object[]) null);
            } catch (IllegalAccessException e) {
                ex = e;
            } catch (InvocationTargetException e) {
                ex = e;
           } catch (NoSuchMethodException e) {
               ex = e;
           }
           // ex != null
           Object[] msgArgs = new Object[] { getClass(),
                   asGetMethod(assocName) + "()",
                   actualObject.getClass(), ex.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD, msgArgs);
           MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
           throw new G9ClientFrameworkException(ex, msg);
        }
    }

    /**
     * Returns <code>true</code> if this role is up-related.
     *
     * @return <code>true</code> if this role is up-related.
     */
    public boolean isUpRelated() {
        return isUpRelated;
    }

    /**
     * Sets the up-related property.
     *
     * @param isUpRelated the value of the property.
     */
    public void setUpRelated(boolean isUpRelated) {
        this.isUpRelated = isUpRelated;
    }

}
