/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.util.HashSet;
import java.util.Set;

import no.g9.client.event.EnableEvent;
import no.g9.client.event.EnableListener;
import no.g9.client.event.EnableNotifier;

/**
 * Class used for mangaing enabling and disabling components implementing the
 * {@link EnableNotifier} interface.
 */
public class EnableManager implements EnableListener {

    /** Used in synchronization */
    private static Object mutex = new Object();

    /**
     * Set of components that should not be enabled when EnableManager.disable
     * is invoked.
     */
    private Set<EnableNotifier> stayDisabled = new HashSet<EnableNotifier>();

    @Override
    public void componentEnable(EnableEvent evt) {
        synchronized (mutex) {
            boolean enabled = ((Boolean) evt.getNewValue()).booleanValue();

            EnableNotifier notifierComponent = (EnableNotifier) evt.getSource();

            if (enabled) {
                stayDisabled.remove(notifierComponent);
                notifierComponent.removeEnableListener(this);
                notifierComponent.setEnabled(false);
                notifierComponent.addEnableListener(this);
            } else {
                stayDisabled.add(notifierComponent);
            }
        }
    }

    /**
     * Disables the specified component. If the component is already disabled,
     * it is stored as a component that should <em>not</em> get enabled when
     * invoking <code>enable(EnableNotifier)</code>. If a component in the mean
     * time is either disabled or enabled outside this manager, it will stay
     * disabled until <code>enable(EnableNotifier)</code> is invoked. At that
     * time, it is determined if the component should really be enabled or
     * disabled.
     * 
     * @param component the component to disable.
     */
    private void disable(EnableNotifier component) {


        if (!component.isEnabled()) {
            stayDisabled.add(component);
        } else {
            component.setEnabled(false);
        }
        component.addEnableListener(this);
    }

    /**
     * Enables the specified component if it is not in the set of components
     * that should stay disabled.
     * 
     * @see #disable(EnableNotifier)
     * @param component the component to enable.
     */
    private void enable(EnableNotifier component) {
        component.removeEnableListener(this);
        if (!stayDisabled.remove(component)) {
            component.setEnabled(true);
        }
    }

    /**
     * Invokes enable or disable on the specified component depending on the
     * second parameter.
     * 
     * @param component the component to enable or disable
     * @param enable if <code>true</code> the component is enabled.
     */
    public void setEnabled(EnableNotifier component, boolean enable) {
        synchronized (mutex) {
            if (enable) {
                enable(component);
            } else {
                disable(component);
            }
        }
    }
}
