/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component.menu;

import javax.swing.Action;
import javax.swing.JMenuItem;

import no.g9.client.event.EnableEvent;
import no.g9.client.event.EnableListener;
import no.g9.client.event.EnableNotifier;

/**
 * G9 wrapper class for JMenuItem.
 */
public class G9MenuItem extends JMenuItem implements EnableNotifier {

    /**
     * Create a new G9 menu item with the given title and name.
     * 
     * @param title the menu item title (what is displayed)
     * @param name the menu item component name
     */
    public G9MenuItem(String title, String name) {
        super(title);
        if (name != null) {
            this.setName(name);
        }
    }

    /**
     * Create a new G9 menu item with the given title and name equals
     * <code>null</code>. The name can be set later by the method
     * <code>setName(String name)</code>
     * 
     * @param title the menu item title (what is displayed)
     */
    public G9MenuItem(String title) {
        super(title);
    }

    /**
     * Creates a menu item whose properties are taken from the specified
     * <code>Action</code>.
     * 
     * @param a the action of the <code>JMenuItem</code>
     * @since 1.3
     */
    public G9MenuItem(Action a) {
        super(a);
        // setName((String) a.getValue(Action.NAME));
    }

    /**
     * Adds an EnableListener to this component. The listener will be notified
     * when the enabled status of this component changes.
     * 
     * @param listener the EnableListener to register on this component.
     * @see #removeEnableListener(EnableListener)
     * @see #fireEnableEvent(EnableEvent)
     */
    @Override
    public void addEnableListener(EnableListener listener) {
        listenerList.add(EnableListener.class, listener);
    }

    /**
     * Notifies all registered EnableListener that an EnableEvent has occured.
     * 
     * @param e the enable event that occured
     * @see #addEnableListener(EnableListener)
     * @see #removeEnableListener(EnableListener)
     */
    @Override
    public void fireEnableEvent(EnableEvent e) {
        Object[] listeners = listenerList.getListenerList();

        for (int i = listeners.length - 2; i >= 0; i -= 2) {

            if (listeners[i] == EnableListener.class) {
                ((EnableListener) listeners[i + 1]).componentEnable(e);
            }
        }
    }

    /**
     * Removes the specified EnableListener from the list of listeneres that
     * will be notified when an EnableEvent occurs.
     * 
     * @param listener the EnableListener to remove.
     * @see #addEnableListener(EnableListener)
     * @see #removeEnableListener(EnableListener)
     */
    @Override
    public void removeEnableListener(EnableListener listener) {
        listenerList.remove(EnableListener.class, listener);

    }

    /*
     * {@inheritDoc}
     * 
     * @see EnableNotifier#setEnabled(boolean)
     */
    @Override
    public void setEnabled(boolean b) {
        boolean oldValue = isEnabled();
        boolean newValue = b;
        super.setEnabled(b);
        fireEnableEvent(new EnableEvent(this, oldValue, newValue));
    }

}
