/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;

import no.esito.log.Logger;
import no.g9.support.CustomDate;

import java.text.DateFormat;
import java.text.ParseException;
import java.util.Date;

/**
 * <code>Document</code> model for fields with date formatting.
 */
public class DateDocument extends G9Document {

    /** The input formatter */
    private DateFormat inputFormatter;

    /** The output formatter */
    private DateFormat outputFormatter;

    /**
     * @param datatype (missing javadoc)
     * @param inputFormat (missing javadoc)
     * @param outputFormat (missing javadoc)
     * @param maxLength (missing javadoc)
     */
    public DateDocument(int datatype, String inputFormat, String outputFormat,
            int maxLength) {
        super(datatype, inputFormat, outputFormat, maxLength);
        inputFormatter = new java.text.SimpleDateFormat(inputFormat);
        inputFormatter.setLenient(false);
        outputFormatter = new java.text.SimpleDateFormat(outputFormat);
    }

    @Override
    public void insertString(int offset, String str, AttributeSet attr)
            throws BadLocationException {
        if (str == null || str.length() == 0) {
            return;
        }
        if (!getInputMode()) {
            super.insertString(offset, str, attr);
            return;
        }
        String format = getInputFormat();
        if (str.length()+getLength() > format.length()) {
            return;
        }

        char aChar = str.charAt(0);
        char fChar = format.charAt(offset);
        if (Character.isDigit(aChar) == Character.isLetter(fChar)) {
            if (Character.isLetterOrDigit(fChar) || !Character.isLetterOrDigit(aChar)) {
                super.insertString(offset, str, attr);
                offset++;
                if (offset < format.length()) {
                    char bChar = format.charAt(offset);
                    // if the next char in the format is not a digit append it
                    if (!Character.isLetterOrDigit(bChar)
                            && (getLength() == offset || getText(offset, 1)
                                    .charAt(0) != bChar)) {
                        super.insertString(offset, "" + bChar, attr);

                    }
                }
            }
        }
    }

    private String dateToString(Date date) {
        String retVal = "";
        if (date != null) {
            retVal = getInputMode() ? inputFormatter.format(date)
                    : outputFormatter.format(date);

        }       
        return retVal;
    }
    
    private Date stringToDate(String str) {
        Date date = null;
        if (str != null) {
            try {
                date = getInputMode() ? inputFormatter.parse(str)
                        : outputFormatter.parse(str);
            } catch (ParseException e) {
                // date remains null
            }
        }
        return date;
    }
    
    
    @Override
    public synchronized Object transform(Object o) {
        return stringToDate(dateToString((Date) o));
    }
    
    
    @Override
    public synchronized void setValue(Object value) {
        String buf = null;
        super.setValue(value);
        if (value != null) {
            buf = (getInputMode() ? inputFormatter.format((Date) value)
                    : outputFormatter.format((Date) value));
        }
        try {
            replace(0, getLength(), buf, null);
        } catch (BadLocationException e) {
            Logger.getLogger(DateDocument.class).warn("Failed to set value.", e);
        }
    }

    @Override
    public synchronized Object getValue() {
        Object result = super.getValue();
        if (getInputMode()) {
            String strValue = null;
            try {
                strValue = getText(0, getLength());
            } catch (BadLocationException e) {
                Logger.getLogger(DateDocument.class).warn("Failed to get value.", e);
            }
            if (strValue != null) {
                if (!(strValue.length() == 0 && result == null)) {
                    try {
                        result = inputFormatter.parse(strValue);
                    } catch (ParseException e) {
                        if (strValue.length() == 0) {
                            result = null;
                        }
                    }
                }
            }
        }
        return result;
    }

    /**
     * Format the value according to the internal format.
     * 
     * @see no.g9.client.support.G9FieldValue#format()
     */
    @Override
    public synchronized String format() {
        String str = "";
        Object value = getValue();
        if (value != null) {
            str = CustomDate.formatInternal(getDatatype(), value);
        }
        return str;
    }


    /**
     * Parse the date string according to the internal format.
     * 
     * @see no.g9.client.support.G9FieldValue#parse(java.lang.String)
     */
    @Override
    public synchronized Object parse(String str) throws ParseException {
        Object value = null;
        if (str != null && str.length() != 0) {
            value = CustomDate.parseInternal(getDatatype(), str);
        }
        return value;
    }

    /**
     * Returns the output text representation of the value object when display
     * rules are applied.
     * 
     * @return the output text representation of the value object.
     */
    @Override
    public synchronized String getOutputText() {
        return getValue() != null ? outputFormatter.format((Date) getValue()): null;
    }
    
}
