/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import no.g9.exception.G9BaseException;

/**
 * A class for creating a queue of Runnables.
 * <p>
 * The runnables are executed in a FIFO fashion.
 */
@SuppressWarnings({"unchecked", "rawtypes"})
public class G9Worker {

    /** Maps the controller to the corresponding worker */
    private static Map<Object, G9Worker> controllerToWorker;

    /**
     * The worker
     */
    private Worker worker;

    /**
     * Count the number of workers
     */
    private static int workerCount = 0;

    /**
     * Thread-safe enqueuing of task on a G9Worker.
     * 
     * @param controller the controlling object
     * @param task the task to enqueue
     * @return the G9Worker responsible for invoking the task.
     * @see #getWorker(Object)
     * @see #enqueue(Runnable)
     */
    public static synchronized G9Worker enqueueTask(Object controller,
            Runnable task) {
        G9Worker worker = getWorker(controller);
        worker.enqueue(task);
        return worker;
    }

    /**
     * Returns an instance of G9Worker. Ensures that at most one instance is
     * created for each controller. Note that g9 Workers must be shut down
     * manually using {@link #relinquish()}. Failing to do so will result in a
     * memory leak.
     * 
     * @param controller the controller "owning" the worker
     * @return an instance of G9Worker
     * @see #enqueueTask(Object, Runnable)
     */
    public static synchronized G9Worker getWorker(Object controller) {
        if (controllerToWorker == null) {
            controllerToWorker = Collections.synchronizedMap(new HashMap());
        }
        G9Worker g9Worker = controllerToWorker
                .get(controller);

        if (g9Worker == null || !g9Worker.isRunning()) {
            String workerName;
            if (controller instanceof G9DialogController) {
                G9DialogController gdc = (G9DialogController) controller;
                workerName = gdc.getDialogName();
            } else {
                workerName = "#" + workerCount++;
            }
            g9Worker = new G9Worker(workerName);
            controllerToWorker.put(controller, g9Worker);
        }

        return g9Worker;
    }

    /**
     * Private access. Use the static method getWorker.
     * 
     * @param workerName the name of the worker, used to give worker thread a
     *            meaningfull name.
     */
    private G9Worker(String workerName) {
        worker = new Worker(workerName);
        worker.start();
    }

    /**
     * Waits on all tasks in the queue to finish before returning. If the worker
     * is interrupted while the calling thread of this method is waiting, the
     * return value is <code>false</code>, otherwise, the method waits until all
     * task on the queue is finished and no current task is executed, and then
     * returns <code>true</code>
     * 
     * @return <code>true</code> if all task are finished
     */
    public boolean waitOnQueue() {
        return worker.waitOnQueue();
    }

    /**
     * Puts a task on the queue.
     * 
     * @param task the task that is put on the queue.
     * @see #enqueueTask(Object, Runnable)
     */
    public synchronized void enqueue(Runnable task) {
        worker.enqueue(task);
    }

    /**
     * Removes all tasks from the queue, and stops the worker. After the worker
     * is relinquished, you should not enqueue any further tasks, unless you
     * create a new worker (using the getWorker() method).
     */
    public synchronized void relinquish() {
        worker.relinquish();
        synchronized (controllerToWorker) {
            Iterator it = controllerToWorker.keySet().iterator();
            while (it.hasNext()) {
                Object key = it.next();
                if (controllerToWorker.get(key) == this) {
                    controllerToWorker.remove(key);
                    break;
                }
            }
        }
        
    }

    /**
     * Returns true if this worker is currently doing a task.
     * 
     * @return true if worker is doing a task
     */
    public synchronized boolean isRunning() {
        return worker.isAlive();
    }

    /**
     * Returns true if this workers queue is empty
     * 
     * @return true if workers queue is empty
     */
    public synchronized boolean isEmpty() {
        return worker.isEmpty();
    }

    /**
     * The actual implementation of the queue.
     */
    private static class Worker extends Thread {
        /** do work flag */
        private boolean doWork = true;

        /** The runnable queue */
        private List<Runnable> queue = new ArrayList<Runnable>();

        /** The working flag */
        private boolean working = false;

        /**
         * Creates a new g9 worker thread. The name of this worker thread
         * will be prefixed with "G9Worker-".
         * 
         * @param workerName the name of the worker thread
         */
        public Worker(String workerName) {
            super("G9Worker-" + workerName);
            setPriority(Thread.NORM_PRIORITY);
        }

        /**
         * Waits on the worker thread until it is finished will all tasks. The
         * effect of invoking this method is that the invoking thread will pause
         * until the worker is finished with all task on the queue. If the
         * invoking thread is interrupted while waiting on the queue, the
         * returned value is <code>false</code>.
         * 
         * @return <code>true</code> if the queue is empty.
         */
        public synchronized boolean waitOnQueue() {
            while (!queue.isEmpty() || isWorking()) {
                try {
                    wait();
                } catch (InterruptedException e) {
                    return false;
                }

            }
            return queue.isEmpty();
        }

        /**
         * Returns true if the worker is busy
         * 
         * @return true if busy
         */
        private synchronized boolean isWorking() {
            return working;
        }

        /**
         * Puts a new task on the queue
         * 
         * @param task (missing javadoc)
         */
        public synchronized void enqueue(Runnable task) {
            queue.add(task);
            notify();
        }


        /**
         * Check if this worker queue is empty
         * 
         * @return <code>true</code> if the queue is empty
         */
        public synchronized boolean isEmpty() {
            return queue.size() == 0;
        }

        /**
         * Check if this worker is currently executing
         * 
         * @return <code>true</code> if the worker is working.
         */
        @SuppressWarnings("unused")
		public synchronized boolean isRunning() {
            return doWork;
        }

        /**
         * Starts the worker. When tasks are put on the queue, the worker will
         * perform them.
         */
        @Override
        public void run() {
            while (doWork) {
                try {
                    Runnable next = dequeue();
                    if (next == null) {
                        doWork = false;
                        break;
                    }
                    Thread task = new Thread(next);
                    startWork();
                    task.start();
                    try {
                        task.join();
                    } catch (InterruptedException e) {
                        task.interrupt();
                    }
                } catch (G9BaseException e) {
                    // already logged.
                } finally {
                    endWork();
                }
            }
        }

        /**
         * Stop working
         */
        private synchronized void endWork() {
            working = false;
            notify();

        }

        /**
         * Strart working
         */
        private synchronized void startWork() {
            working = true;
        }

        /**
         * Removes the oldest task from the queue
         * 
         * @return the task that has spent the longest time on the queue.
         */
        public synchronized Runnable dequeue() {
            while (queue.isEmpty()) {
                try {
                    wait();
                } catch (InterruptedException e) {
                    // woke up.
                }
            }
            startWork();
            notifyAll();
            return queue.remove(0);
        }

        /**
         * Finishes this worker thread, removing it from the list of running
         * threads.
         */
        public synchronized void relinquish() {
            enqueue(null);
        }

    } // end inner class Worker

}
