/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Color;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import no.g9.exception.G9ClientException;
import no.g9.support.Visitable;
import no.g9.support.Visitor;

/**
 * Contains common methods for object selection interfaces.
 */
@SuppressWarnings("rawtypes")
public interface ObjectSelectionNode extends Visitable<ObjectSelectionNode> {

    /**
     * The initial state of the node. No data is displayed, and the user has not
     * made any changes to the dialog. Invoking clear on a node will put it in
     * this state.
     */
    int CLEARED = 0;

    /**
     * The state that occures when an initially CLEAR node has been altered by
     * the user.
     */
    int EDITED = 1;

    /**
     * The state that occurs when an object has been displayed, but not altered
     * by the user.
     */
    int CLEAN = 2;

    /**
     * The state that occurs when an CLEAN node was altered by the user.
     */
    int DIRTY = 3;

    /**
     * Returns the current state of this node. The state is either CLEARED,
     * EDITED, CLEAN or DIRTY, see constants for details. The transformation
     * between states are as follows:
     * <ul>
     * <li>CLEARED -&gt; EDITED -&gt; CLEARED (user altered / unaltered data)</li>
     * <li>CLEARED -&gt; CLEAN (data displayed)</li>
     * <li>EDITED -&gt; CLEAN (data inserted then displayed)</li>
     * <li>CLEAN -&gt; DIRTY -&gt; CLEAN (user altered / unaltered data)
     * <li>DIRTY -&gt; CLEAN (data updated then displayed)
     * </ul>
     *
     * @return the current state of this node.
     */
    public int getState();

    /**
     * Sets the current state of this node. The state should be one of the
     * constants CLEARED, EDITED, CLEAN or DIRTY. Note that setting this state
     * will affect the possible outcomes of invoking {@link #getState()}.
     *
     * @param state the new state to set.
     * @return the state that was calculated based on the specified state value.
     */
    public int setState(int state);

    /**
     * Sets the foreground and background color of given node.
     *
     * @param foreground The foreground color
     * @param background The background color
     */
    public void setStyle(Color foreground, Color background);

    /**
     * Returns the representing the relatiion .attr from the dialog. If no
     * match, it returns null.
     *
     * @param attr The attribute to obtain.
     * @return The value of the object.
     */
    public String obtain(String attr);

    /**
     * Returns an object representing the from the dialog. The object returned
     * is a new object with attributes copied from the model. Only attributes
     * selected from the object selection in g9 are kept. (Hidden attributes
     * are also returned).
     *
     * @return Object of with attributes from the OS.
     */
    public Object obtainAsObject();

    /**
     * Creates a new domain object and writes dialog data to the object. The
     * effect is that the domain object that came from, say a db, is not
     * affected.
     *
     * @return a copy of the domain object, attributes are set from the dialog.
     */
    public Object peekAtObject();

    /**
     * Returns the Class object of the class this object selection displays.
     *
     * @return the Class representation
     */
    public Class getRepresentingClass();

    /**
     * Returns true if all fields in objct have a value, that is not null or "".
     *
     * @return True if value is set, false if not.
     */
    public boolean hasValue();

    /**
     * Clears fields from the relation from the dialog structure.
     */
    public void clear();

    /**
     * Clears a given field from the relation from the dialog structure.
     *
     * @param s The attribute to clear.
     */
    public void clear(String s);

    /**
     * Clears all fields exept the fields that are part of the main key.
     */
    public void clearKeepKeys();

    /**
     * Flags the object selection as changed.
     *
     * @param b The new changed value.
     * @deprecated use {@link ObjectSelectionNode#setState(int)}
     */
    @Deprecated
    public void setChanged(boolean b);

    /**
     * Sets all displayed var-fields in the object selection as not changed.
     */
    public void resetFields();

    /**
     * Flags the given attribute as changed in the object selection.
     *
     * @param b The new changed value.
     * @param s The attribute to change.
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public void setChanged(boolean b, String s);

    /**
     * Returns if the object is editable.
     *
     * @return True if enabled, false if not.
     */
    public boolean isEnabled();

    /**
     * Checks if a given field is possible to edit.
     *
     * @param s The field to check.
     * @return True if it is enabled, false if not.
     */
    public boolean isEnabled(String s);

    /**
     * Checks if node is visible.
     *
     * @return True if visible, false if not.
     */
    public boolean isShown();

    /**
     * Checks if a given node is visible.
     *
     * @param s The attribute to check visibility for.
     * @return True if visible, false if not.
     */
    public boolean isShown(String s);

    /**
     * Makes object selection visible/invisible.
     *
     * @param b True makes the object selection visibile, false it will
     *            disappear.
     */
    public void setShown(boolean b);

    /**
     * Makes this object selection enabled or disabled.
     *
     * @param b True enables it, false disables it.
     */
    public void setEnabled(boolean b);

    /**
     * Makes a given attribute in the object selection enabled or disabled.
     *
     * @param b True enables it, false disables it.
     * @param s The attribute to alter.
     */
    public void setEnabled(boolean b, String s);

    /**
     * Requests focus on the given field of this object selection node. It is
     * not granted that focus is altered after the call to this, as described in
     * javadoc for focus requests in swing. (In normal circumstances it will...)
     *
     * @param s The field to request focus for.
     */
    public void setFocus(String s);

    /**
     * Makes a given field in this object selection node visible or invisible.
     *
     * @param b True makes the attribute visibile, false it will disappear.
     * @param s The attribute to alter.
     */
    public void setShown(boolean b, String s);

    /**
     * Checks if a given attribute in the object selection has focus.
     *
     * @param s The attribute to check.
     * @return True means it has focus, false means it haven't got focus.
     */
    public boolean hasFocus(String s);

    /**
     * Checks if at least one attribute in the object selection contains a none
     * empty value. (Value different than 0 or a trimmed string with length).
     *
     * @return True if one field got value, false if all are empty.
     */
    public boolean hasOneValue();

    /**
     * Checks whether the current value in the display is valid with respect of
     * the data types in the object of the data selector.
     *
     * @return True if it is, false if not.
     */
    public boolean isConsistant();

    /**
     * Returns true if the given field has any value, that is not null,0 or "".
     *
     * @param field The field to check.
     * @return True if it has value, false if not.
     */
    public boolean hasValue(String field);

    /**
     * Sets if the given field should display '0' or blank when 0 value
     *
     * @param s The field to set.
     * @param val True to display 0, false to display blank.
     * @deprecated Display rules and blank when zero should be set in the model!
     */
    @Deprecated
    public void setShowZero(String s, boolean val);

    /**
     * Sets the parent node object.
     *
     * @param parentNode the parent node of this object selection.
     */
    public void setParentNode(ObjectSelectionNode parentNode);

    /**
     * Returns the parent node object.
     *
     * @return the parent of this node
     */
    public ObjectSelectionNode getParentNode();

    /**
     * Returns the path from the root node to this node.
     *
     * @return a List of nodes representing the path to this node.
     */
    public List getPathToNode();

    /**
     * Returns the root node of this object selection node.
     *
     * @return the root node in the object selection.
     */
    public ObjectSelectionNode getRootNode();

    /**
     * Returns a set containing the accessor objects for the children
     *
     * @return a set of children accessors.
     */
    public List getChildren();

    /**
     * Adds a child to the set of children
     *
     * @param child (missing javadoc)
     */
    public void addChild(ObjectSelectionNode child);

    /**
     * Forwards calls to the specific hook method.
     *
     * @param anObjectToDisplay the object to be displayed
     * @param roleName the role name of the object to display
     * @return <code>true</code> if the object should be displayed
     */
    public boolean callHookBeforeDisplay(Object anObjectToDisplay,
            String roleName);

    /**
     * Displays a domain object by casting it to the appropriate type and
     * forward the call to the special display method.
     *
     * @param domainObject the domain object to display.
     */
    public void display(Object domainObject);

    /**
     * Internal use. Implemented mehtods forwards the call to the correct
     * display method.
     *
     * @param domainObject the domain object holding the attribute to display
     * @param attribute the name of the attribute
     * @param resetState flag indicating if field should be reset.
     */
    public void osDisplay(Object domainObject, String attribute,
            boolean resetState);


    /**
     * Displays the collection of domain objects.
     *
     * @param domainObjects the domain objects to display
     * @param pathToTarget the path to the target node
     * @param findAll if <code>true</code> the display is invoked as part of a
     *            find all action.
     */
    public void displayAboveTarget(Collection domainObjects, List pathToTarget,
            boolean findAll);

    /**
     * Performs a recursive display. The display is performed on the EDT. If the
     * invoking thread is not the EDT, the thread is paused while the display is
     * performed.
     *
     * @param resultObject a collection of domain objects
     * @param pathToTarget the path to the target node
     * @param findAll <code>true</code> if the display is a result of a find all
     *            action.
     */
    public void displayOnEDT(final Collection resultObject,
            final List pathToTarget, final boolean findAll);

    /**
     * Recursivly obtains the the object from the dialog. This method exists to
     * overcome a pre java 1.5 deficiency which prohibits implementation to
     * return a sub class of the specified return type. Thus, the implementation
     * of this method is to simply invoke the recursiveObtain() - method and
     * upcast the result before returning.
     *
     * @param pathToTarget a List representing the path to the target
     * @return the obtained object.
     */
    public Object recursiveObtainObject(List pathToTarget);

    /**
     * (missing javadoc)
     *
     * @param pathToTarget (missing javadoc)
     * @param singleObtain (missing javadoc)
     * @return (missing javadoc)
     */
    public Object recursiveObtainObject(List pathToTarget, boolean singleObtain);

    /**
     * Returns the dialog controller.
     *
     * @return the dialog controller.
     */
    public G9DialogController getController();

    /**
     * Returns a set containing the names of the attributes/fields that are
     * changed since the initial display. If the length of the returned set is
     * zero, no atributes are changed.
     *
     * @return a Set with changed attributes.
     */
    public Set getChangedFields();

    /**
     * Returns a set containg the names of the attributes/fields that are a part
     * of the key and are changed since the initial display. If the length of
     * the returned set is zero, no key atributes are changed.
     *
     * @return a Set with changed key attributes
     */
    public Set getChangedKeyFields();

    /**
     * Returns a set containing all fields that have changed since the initial
     * display and wich are not part of the key, in other words the complement
     * of the set derived by getChangedKeyFields
     *
     * @return a set containing all changed, non-key, fields.
     */
    public Set getChangedNoKeyFields();

    /**
     * Check if the specified attribute is changed. The attribute is changed if
     * the user has edited the field representing the attribute, and the
     * contents of the field is different from the originally displayed
     * contents. The attribute can be specifed as either the attribute name, or
     * prefixed with the role-name (e.g. "id" or "Role.id"). If the specified
     * attribute is not found, a G9ClientException is thrown.
     *
     * @param attributeName the name of the attribute
     * @return <code>true</code> if the field has changed, otherwise <code>false
     *         </code>.
     * @throws G9ClientException if the attribute is not part of this role.
     * @see AbstractNode#getChangedNoKeyFields()
     * @see #getChangedKeyFields()
     * @see #getChangedFields()
     */
    public boolean isChanged(String attributeName) throws G9ClientException;

    /**
     * If the controller's ignoreCheckChange property is <code>false</code>,
     * determin if it is ok to proceed with find action. The following
     * conditions are checked:
     * <ol>
     * <li>No data fields in this node, exept key-fields, are changed <li>No
     * fields in child-nodes (recursivly) are changed.
     * </ol>
     * If these conditions are met, the return value is <code>true</code>.
     * Otherwise, a modal dialog is shown and the user (of the application) is
     * asked wether to proceed or not. If the user answers yes, the returned
     * value is <code>true</code>.
     *
     * @return <code>true</code> if it's ok to proceed with find.
     * @see G9DialogController#setIgnoreCheckChanged(boolean)
     */
    public boolean checkFind();

    /**
     * If the dialog controllers ignoreCheckChange is <code>false</code> this
     * method checks if the node (with children) is unchanged. To things are
     * considered:
     * <ol>
     * <li>First the currently displayed attribute values are compared against
     * the initially displayed attribute value. If all displayed values equals
     * the original diaplayed value (thus it is not changed by a user) the check
     * continues to the next face, otherwise it returns false.
     * <p>
     * This is done recursivly, thus the first node checked is a leaf-node. <li>
     * If the current node being checked is an up-related node, the relation to
     * the parent is checked. If the currently displayed (up-related) node
     * differs from the parent's child node, the return value is <code>false
     * </code>
     * </ol>
     *
     * @return <code>true</code> if no fields are changed and all up-related
     *         relations are intact.
     * @see G9DialogController#setIgnoreCheckChanged(boolean)
     */
    public boolean checkChanged();

    /**
     * If the controller's ignoreCheckChange property is <code>false</code>,
     * determin if it is ok to proceed with action delete. If he dialog does not
     * contain any changed data since it's last display, the return value is
     * <code>true</code>. Otherwise, a modal dialog is shown and the user (of
     * the application) is asked wether to proceed or not. If the user answers
     * "yes", the return value is <code>true</code>.
     *
     * @return <code>true</code> if it's ok to proceed with delte.
     * @see G9DialogController#setIgnoreCheckChanged(boolean)
     */
    public boolean checkDelete();

    /**
     * If the controller's ignoreCheckChange property is <code>false</code>,
     * determin if it is ok to proceed with action clear. If the dialog does not
     * contain any changed data since it's last display, the return value is
     * <code>true</code>. Otherwise, a model dialog is shown and the user (of
     * the application) is asked wether to proceed or not. If the user answers
     * "yes", the return value is <code>true</code>.
     *
     * @return <code>true</code> if it's ok to proceed with delete. *
     * @see G9DialogController#setIgnoreCheckChanged(boolean)
     */
    public boolean checkClear();

    /**
     * If the controller's ignoreCheckChange property is <code>false</code>,
     * determin if it is ok to proceed with action save. No specific check is
     * perfomed, but a hook is called for each node involved in the save action
     * (the sequence is from the top of the object selection). The returned
     * value from this method is always <code>true</code>, and the return value
     * from the specific hook methods are used to determin wether to proceed
     * with the check save or not.
     *
     * @return <code>true</code> if it is ok to proceed with action save.
     * @see G9DialogController#setIgnoreCheckChanged(boolean)
     */
    public boolean checkSave();

    /**
     * If the controller's ignoreCheckChange property is <code>false</code>,
     * determine if it is ok to proceed to change the selected row in a
     * listblock.
     *
     * @param oldSelection the index of the old selection
     * @param newSelection the index of the new selection
     * @return true if it is ok to proceed.
     * @see G9DialogController#setIgnoreCheckChanged(boolean)
     */
    public boolean checkRowSelection(int oldSelection, int newSelection);

    /**
     * If the dialog controllers Determin if it is ok to proceed with action
     * close window. If he dialog does not contain any changed data since it's
     * last display, the return value is <code>true</code>. Otherwise, a modal
     * dialog is shown and the user (of the application) is asked wether to
     * proceed or not. If the user answers "yes", the return value is <code>true
     * </code>.
     *
     * @return <code>true</code> if it's ok to proceed with close.
     * @see G9DialogController#setIgnoreCheckChanged(boolean)
     */
    public boolean checkClose();

    /**
     * Get the role name represented by this node.
     *
     * @return the role name of this node.
     */
    public String getRoleName();

    /**
     * Forwards a call to the appropriate check save hook.
     *
     * @return the result from the hook (see {@code hookCheckSave<RoleName>} for
     *         details).
     */
    public abstract boolean callCheckSaveHook();

    /**
     * Forwards a call to the appropriate check find hook.
     *
     * @return the result from the hook (see {@code hookCheckFind<RoleName>} for
     *         details).
     */
    public abstract boolean callCheckFindHook();

    /**
     * Forwards a call to the appropriate check clear hook.
     *
     * @return the result from the hook (see {@code hookCheckClear<RoleName>} for
     *         details).
     */
    public abstract boolean callCheckClearHook();

    /**
     * Forwards a call to the appropriate check delete hook.
     *
     * @return the result from the hook (see {@code hookCheckDelete<RoleName>} for
     *         details).
     */
    public abstract boolean callCheckDeleteHook();

    /**
     * Forwards a call to the appropriate check close hook.
     *
     * @return the result from the hook (see {@code hookCheckClose<RoleName>} for
     *         details).
     */
    public abstract boolean callCheckCloseHook();

    /**
     * Forwards a call to the appropriate check change hook.
     *
     * @return the result from the hook (see {@code hookCheckChange<RoleName>} for
     *         details).
     */
    public abstract boolean callCheckChangeHook();

    /**
     * Forwards a call to the appropriate check row select hook.
     *
     * @param currentSel index of current selection
     * @param newSel index of new selection
     * @return the result from the hook (see {@code hookCheckRowSelect<RoleName>} for
     *         details).
     */
    public abstract boolean callCheckRowSelectHook(int currentSel, int newSel);

    /**
     * @return (missing javadoc)
     */
    public abstract boolean inListblock();

    /**
     * (missing javadoc)
     *
     * @param createIfNull (missing javadoc)
     * @return (missing javadoc)
     */
    public abstract Set obtainSet(boolean createIfNull);

    /**
     * (missing javadoc)
     *
     * @return (missing javadoc)
     */
    public abstract Object greedyObtain();

    /**
     * Visit all nodes in this branch
     * @param visitor the visitor that gets the visit callback.
     */
    public abstract void visitBranch(Visitor<ObjectSelectionNode> visitor);

}
