/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;

import no.esito.log.Logger;
import no.g9.support.FormatHelper;
import no.g9.support.TypeTool;

/**
 * <code>Document</code> model for fields with text formatting.
 */
public class TextDocument extends G9Document {

    private int caseConvertion = 0;

    /**
     * @param datatype (missing javadoc)
     * @param inputFormat (missing javadoc)
     * @param outputFormat (missing javadoc)
     * @param maxLength (missing javadoc)
     * @param caseConvertion -1: lowercase, 0: nocase, 1: uppercase
     * @roseuid 44325EED024A
     */
    public TextDocument(int datatype, String inputFormat, String outputFormat, int maxLength,int caseConvertion) {
        super(datatype, inputFormat, outputFormat, maxLength);
        this.caseConvertion = caseConvertion;
    }

    @Override
    public void insertString(int offset, String str, AttributeSet attr) throws BadLocationException {
        if (str == null) {
            // Do nothing if str == null
            return;
        }
        int maxLength = getMaxLength();
        if (maxLength > 0 && getLength() + str.length() > maxLength) {
            int newlen = maxLength - getLength();
            if (newlen <= 0) {
                return;
            }
            str = str.substring(0, newlen);
        }
        if (caseConvertion == -1) {
            str = str.toLowerCase(FormatHelper.getLocale());
        } else if (caseConvertion == 1) {
            str = str.toUpperCase(FormatHelper.getLocale());
        }
        super.insertString(offset, str, attr);
    }

    @Override
    public synchronized void setValue(Object value) {
        super.setValue(value);
        StringBuffer buf = new StringBuffer();
        if (value != null) {
            buf.append(value.toString());
        }
        try {
            String str = caseConvertion == -1 ? buf.toString().toLowerCase(FormatHelper.getLocale()) :
                         caseConvertion == 1 ? buf.toString().toUpperCase(FormatHelper.getLocale()) :
                         buf.toString();
            replace(0,getLength(),str,null);
        } catch (BadLocationException e) {
            Logger.getLogger(TextDocument.class).warn("Failed to set value.", e);
        }
    }

    @Override
    public synchronized Object getValue() {
        Object result = super.getValue();
        if (getInputMode()) {
            String strValue = null;
            try {
                strValue = getText(0,getLength());
            } catch (BadLocationException e) {
                Logger.getLogger(TextDocument.class).warn("Failed to get value.", e);
            }
            if (strValue != null) {
                if (!(strValue.length() == 0 && result == null)) {
                    strValue = TypeTool.stripEnd(strValue);
                    if (caseConvertion == -1) {
                        strValue = strValue.toLowerCase(FormatHelper.getLocale());
                    }
                    else if (caseConvertion == 1) {
                        strValue = strValue.toUpperCase(FormatHelper.getLocale());
                    }
                    result = strValue;
                }
            }
        }
        return result;
    }

    @Override
    public synchronized Object parse(String s) {
        return s;
    }

    private String toInternal(String s) {
        if (s != null) {
            s = TypeTool.stripEnd(s);
            if (getMaxLength() > 0 && s.length() > getMaxLength()) {
                s = s.substring(0, getMaxLength());
            }
            if (caseConvertion == -1) {
                s = s.toLowerCase(FormatHelper.getLocale());
            } else if (caseConvertion == 1) {
                s = s.toUpperCase(FormatHelper.getLocale());
            }
        }
        return s;
    }

    @Override
    public synchronized Object transform(Object o) {
        return toInternal((String) o);
    }

    @Override
    public synchronized String getOutputText() {
        return (String) transform(getValue());
    }

}
