/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.awt.Dialog;
import java.awt.Frame;
import java.awt.GraphicsConfiguration;
import java.awt.HeadlessException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.KeyStroke;

import no.g9.client.support.G9DialogController;
import no.g9.client.support.G9DialogFrame;

/**
 * The class representing a dialog box.
 */
public class G9DialogBox extends JDialog {

    /**
     * The controller of this dialog box.
     */
    private G9DialogController controller;

    /**
     * Creates a non-modal dialog without a title and without a specified
     * <code>Frame</code> owner. A shared, hidden frame will be set as the owner
     * of the dialog.
     * <p>
     * This constructor sets the component's locale property to the value
     * returned by <code>JComponent.getDefaultLocale</code>.
     * 
     * @exception HeadlessException if GraphicsEnvironment.isHeadless() returns
     *                true.
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @see javax.swing.JComponent#getDefaultLocale
     */
    public G9DialogBox() throws HeadlessException {
        this((Frame)null, false);
    }

    /**
     * Creates a non-modal dialog without a title with the specified
     * <code>Dialog</code> as its owner.
     * <p>
     * This constructor sets the component's locale property to the value
     * returned by <code>JComponent.getDefaultLocale</code>.
     * 
     * @param owner the non-null <code>Dialog</code> from which the dialog is
     *            displayed
     * @exception HeadlessException if GraphicsEnvironment.isHeadless() returns
     *                true.
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @see javax.swing.JComponent#getDefaultLocale
     */
    public G9DialogBox(Dialog owner) throws HeadlessException {
        this(owner, false);
    }

    /**
     * Creates a modal or non-modal dialog without a title and with the
     * specified owner dialog.
     * <p>
     * This constructor sets the component's locale property to the value
     * returned by <code>JComponent.getDefaultLocale</code>.
     * 
     * @param owner the non-null <code>Dialog</code> from which the dialog is
     *            displayed
     * @param modal true for a modal dialog, false for one that allows other
     *            windows to be active at the same time
     * @exception HeadlessException if GraphicsEnvironment.isHeadless() returns
     *                true.
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @see javax.swing.JComponent#getDefaultLocale
     */
    public G9DialogBox(Dialog owner, boolean modal)
            throws HeadlessException {
        this(owner, null, modal);
    }

    /**
     * Creates a non-modal dialog without a title with the specified
     * <code>Frame</code> as its owner. If <code>owner</code> is
     * <code>null</code>, a shared, hidden frame will be set as the owner of the
     * dialog.
     * <p>
     * This constructor sets the component's locale property to the value
     * returned by <code>JComponent.getDefaultLocale</code>.
     * 
     * @param owner the <code>Frame</code> from which the dialog is displayed
     * @exception HeadlessException if GraphicsEnvironment.isHeadless() returns
     *                true.
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @see javax.swing.JComponent#getDefaultLocale
     */
    public G9DialogBox(Frame owner) throws HeadlessException {
        this(owner, false);
    }

    /**
     * Creates a modal or non-modal dialog without a title and with the
     * specified owner <code>Frame</code>. If <code>owner</code> is
     * <code>null</code>, a shared, hidden frame will be set as the owner of the
     * dialog.
     * <p>
     * This constructor sets the component's locale property to the value
     * returned by <code>JComponent.getDefaultLocale</code>.
     * 
     * @param owner the <code>Frame</code> from which the dialog is displayed
     * @param modal true for a modal dialog, false for one that allows others
     *            windows to be active at the same time
     * @exception HeadlessException if GraphicsEnvironment.isHeadless() returns
     *                true.
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @see javax.swing.JComponent#getDefaultLocale
     */
    public G9DialogBox(Frame owner, boolean modal)
            throws HeadlessException {
        this(owner, null, modal);
    }

    /**
     * Creates a non-modal dialog with the specified title and with the
     * specified owner dialog.
     * <p>
     * This constructor sets the component's locale property to the value
     * returned by <code>JComponent.getDefaultLocale</code>.
     * 
     * @param owner the non-null <code>Dialog</code> from which the dialog is
     *            displayed
     * @param title the <code>String</code> to display in the dialog's title bar
     * @exception HeadlessException if GraphicsEnvironment.isHeadless() returns
     *                true.
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @see javax.swing.JComponent#getDefaultLocale
     */
    public G9DialogBox(Dialog owner, String title)
            throws HeadlessException {
        this(owner, title, false);
    }

    /**
     * Creates a modal or non-modal dialog with the specified title and the
     * specified owner frame.
     * <p>
     * This constructor sets the component's locale property to the value
     * returned by <code>JComponent.getDefaultLocale</code>.
     * 
     * @param owner the non-null <code>Dialog</code> from which the dialog is
     *            displayed
     * @param title the <code>String</code> to display in the dialog's title bar
     * @param modal true for a modal dialog, false for one that allows other
     *            windows to be active at the same time
     * @exception HeadlessException if GraphicsEnvironment.isHeadless() returns
     *                true.
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @see javax.swing.JComponent#getDefaultLocale
     */
    public G9DialogBox(Dialog owner, String title, boolean modal)
            throws HeadlessException {
        super(owner, title, modal);
    }

    /**
     * Creates a non-modal dialog with the specified title and with the
     * specified owner frame. If <code>owner</code> is <code>null</code>, a
     * shared, hidden frame will be set as the owner of the dialog.
     * <p>
     * This constructor sets the component's locale property to the value
     * returned by <code>JComponent.getDefaultLocale</code>.
     * 
     * @param owner the <code>Frame</code> from which the dialog is displayed
     * @param title the <code>String</code> to display in the dialog's title bar
     * @exception HeadlessException if GraphicsEnvironment.isHeadless() returns
     *                true.
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @see javax.swing.JComponent#getDefaultLocale
     */
    public G9DialogBox(Frame owner, String title)
            throws HeadlessException {
        this(owner, title, false);  
    }

    /**
     * Creates a modal or non-modal dialog with the specified title and the
     * specified owner <code>Frame</code>. If <code>owner</code> is
     * <code>null</code>, a shared, hidden frame will be set as the owner of
     * this dialog. All constructors defer to this one.
     * <p>
     * NOTE: Any popup components (<code>JComboBox</code>,
     * <code>JPopupMenu</code>, <code>JMenuBar</code>) created within a modal
     * dialog will be forced to be lightweight.
     * <p>
     * This constructor sets the component's locale property to the value
     * returned by <code>JComponent.getDefaultLocale</code>.
     * 
     * @param owner the <code>Frame</code> from which the dialog is displayed
     * @param title the <code>String</code> to display in the dialog's title bar
     * @param modal true for a modal dialog, false for one that allows other
     *            windows to be active at the same time
     * @exception HeadlessException if GraphicsEnvironment.isHeadless() returns
     *                true.
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @see javax.swing.JComponent#getDefaultLocale
     */
    public G9DialogBox(Frame owner, String title, boolean modal)
            throws HeadlessException {
        super(owner, title, modal);
    }

    /**
     * Creates a modal or non-modal dialog with the specified title, owner
     * <code>Dialog</code>, and <code>GraphicsConfiguration</code>.
     * <p>
     * NOTE: Any popup components (<code>JComboBox</code>,
     * <code>JPopupMenu</code>, <code>JMenuBar</code>) created within a modal
     * dialog will be forced to be lightweight.
     * <p>
     * This constructor sets the component's locale property to the value
     * returned by <code>JComponent.getDefaultLocale</code>.
     * 
     * @param owner the <code>Dialog</code> from which the dialog is displayed
     * @param title the <code>String</code> to display in the dialog's title bar
     * @param modal true for a modal dialog, false for one that allows other
     *            windows to be active at the same time
     * @param gc the <code>GraphicsConfiguration</code> of the target screen
     *            device. If <code>gc</code> is <code>null</code>, the same
     *            <code>GraphicsConfiguration</code> as the owning Dialog is
     *            used.
     * @exception HeadlessException if GraphicsEnvironment.isHeadless()
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @see javax.swing.JComponent#getDefaultLocale returns true.
     * @since 1.4
     */
    public G9DialogBox(Dialog owner, String title, boolean modal,
            GraphicsConfiguration gc) throws HeadlessException {
        super(owner, title, modal, gc);
    }

    /**
     * Creates a modal or non-modal dialog with the specified title, owner
     * <code>Frame</code>, and <code>GraphicsConfiguration</code>.
     * <p>
     * NOTE: Any popup components (<code>JComboBox</code>,
     * <code>JPopupMenu</code>, <code>JMenuBar</code>) created within a modal
     * dialog will be forced to be lightweight.
     * <p>
     * This constructor sets the component's locale property to the value
     * returned by <code>JComponent.getDefaultLocale</code>.
     * 
     * @param owner the <code>Frame</code> from which the dialog is displayed
     * @param title the <code>String</code> to display in the dialog's title bar
     * @param modal true for a modal dialog, false for one that allows other
     *            windows to be active at the same time
     * @param gc the <code>GraphicsConfiguration</code> of the target screen
     *            device. If <code>gc</code> is <code>null</code>, the same
     *            <code>GraphicsConfiguration</code> as the owning Frame is
     *            used.
     * @exception HeadlessException if GraphicsEnvironment.isHeadless() returns
     *                true.
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @see javax.swing.JComponent#getDefaultLocale
     * @since 1.4
     */
    public G9DialogBox(Frame owner, String title, boolean modal,
            GraphicsConfiguration gc) {
        super(owner, title, modal, gc);
    }


    /**
     * Returns the g9 dialog controller for this dialog box.
     * 
     * @return the dialog controller
     */
    public G9DialogController getController() {
        return controller;
    }

    /**
     * Sets the g9 dialog controller for this dialog.
     * 
     * @param controller the dialog controller.
     */
    public void setController(G9DialogController controller) {
        this.controller = controller;
    }

    /**
     * This method is intended for internal use only, and is only public as an
     * implementation effect. Copies the input and action map from the specified
     * dialog frame.
     * 
     * @param diaFrame the dialog frame to copy action from.
     */
    public void copyActions(G9DialogFrame diaFrame) {
        InputMap diaInputMap = getRootPane().getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW);
        ActionMap diaActionMap = getRootPane().getActionMap();
        
        InputMap docInputMap = diaFrame.getRootPane().getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW);
        ActionMap docActionMap = diaFrame.getRootPane().getActionMap();
        
        KeyStroke[] keysInDocWindow = docInputMap.allKeys();

        Set<KeyStroke> setOfKeysInDoc = new HashSet<KeyStroke>();
        
        if (keysInDocWindow != null) {
            setOfKeysInDoc.addAll(Arrays.asList(keysInDocWindow));
            if (diaInputMap.allKeys() != null) {
                setOfKeysInDoc.removeAll(Arrays.asList(diaInputMap.allKeys()));
            }
            
            Iterator<KeyStroke> it = setOfKeysInDoc.iterator();
            while (it.hasNext()) {
                KeyStroke stroke = it.next();
                Object key = docInputMap.get(stroke);
                Action action = docActionMap.get(key);
                diaInputMap.put(stroke, key);
                diaActionMap.put(key, action);
            }
        }
    }

}
