/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component.menu;

import java.awt.Component;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JSeparator;

import no.g9.client.support.G9DialogFrame;
import no.g9.client.support.G9Render;
import no.g9.message.Message;
import no.g9.message.MessageSystem;

/**
 * Abstract class representing a menu which can be added to a G9MenuBar. In
 * addition to adding it to a G9MenuBar, the menues of the same type can be
 * merged according to rules specified in the G9 manual.
 */
public class G9Menu extends JMenu {

    /** Because of serialization */
    private static final long serialVersionUID = 1L;

    /** Constant denoting that the menu is a custom menu */
    public static final int CUSTOM_MENU = 0;

    /** Constant denoting that the menu is a file menu */
    public static final int FILE_MENU = 1;

    /** Constant denoting that the menu is an edit menu. */
    public static final int EDIT_MENU = 2;

    /** Constant denoting that the menu is a view menu. */
    public static final int VIEW_MENU = 3;

    /** Constant denoting that the menu is a window menu */
    public static final int WINDOW_MENU = 4;

    /** Constant denoting that the menu is a help menu */
    public static final int HELP_MENU = 5;

    /** Reference to the menu type */
    private int menuType;

    /** Menu default items added by the super class */
    private List<?> defaultItems = new Vector<Object>();

    /** Menu owner items added by the owner dialog */
    private List<Component> myComponents;

    /** The menu to merge into this menu */
    private G9Menu mergedFromMenu;

    /** The menu this menu is merged into */
    private G9Menu mergedIntoMenu;

    /** The defult menu title defined by the G9 model */
    private String defaultTitle;

    /** Set if the menu is in modal mode */
    private boolean isModal = false;

    /** If set, consecutive separators are reduced to one */
    private boolean removeConsecutiveSeparators = true;


    /**
     * Constructs a new <code>G9Menu</code> with the supplied title and
     * name.
     *
     * @param title the text for the menu title
     * @param menuType the menu type
     * @param name the name of the menu
     */
    public G9Menu(String title, int menuType, String name){
        this(title, menuType);
        this.setName(name);
    }

    /**
     * Constructs a new <code>JMenu</code> with the supplied string as its
     * title.
     *
     * @param title the text for the menu title
     * @param menuType the menu type
     */
    public G9Menu(String title, int menuType) {
        super(title);
        this.defaultTitle = title;
        this.menuType = menuType;
    }

    /**
     * Constructs a new <code>JMenu</code> with the supplied string as its text
     * and specified as a tear-off menu or not.
     *
     * @param s the text for the menu label
     * @param b can the menu be torn off (not yet implemented)
     * @param menuType the menu type
     */
    public G9Menu(String s, boolean b, int menuType) {
        this(s, menuType);
    }

    /**
     * Constructs a new <code>JMenu</code> whose properties are taken from the
     * <code>Action</code> supplied.
     *
     * @param a <code>Action</code>
     * @param menuType the menu type
     * @since 1.3
     */
    public G9Menu(Action a, int menuType) {
        this("", menuType);
        setAction(a);
    }

    /**
     * Reurns the menu type of this menu.
     *
     * @return the menu type
     */
    public int getMenuType() {
        return menuType;
    }

    /**
     * Creates a new menu item with the specified title and adds it to the menu.
     *
     * @param title the title of the menu item
     * @return the added menu item
     */
    public JMenuItem addG9Item(String title) {
        return addG9Item(new G9MenuItem(title));
    }

    /**
     * Creates a new menu item with the specified title and adds it to the menu.
     *
     * @param title the title of the menu item
     * @param name the programatic name of the menu
     * @return the added menu item
     */
    public JMenuItem addG9Item(String title, String name) {
        return addG9Item(new G9MenuItem(title, name));
    }

    /**
     * Adds a menu item to this menu.
     *
     * @param item the menu item to add
     * @return the added menu item
     */
    public synchronized JMenuItem addG9Item(JMenuItem item) {
        if (myComponents == null) {
            myComponents = new ArrayList<Component>();
        }

        myComponents.add(item);
        return add(item);
    }

    /**
     * Adds a separator to this menu.
     *
     * @param separator the separator to add
     * @return the added separator
     */
    public synchronized Component addG9Item(JSeparator separator) {
        if (myComponents == null) {
            myComponents = new ArrayList<Component>();
        }

        myComponents.add(separator);
        return add(separator);
    }

    /**
     * Returns an array containing all components added by one of the
     * addG9Item-methods.
     *
     * @return an array containing all components added by one of the
     *         addG9Item-methods.
     */
    public synchronized Component[] getG9Components() {
        if (myComponents == null) {
            return new Component[0];
        }

        Component[] retVal = new Component[myComponents.size()];
        for (int i = 0; i < retVal.length; i++) {
            retVal[i] = myComponents.get(i);
        }

        return retVal;
    }

    /**
     * Merges the specified menu with this menu.
     * <p>
     * Pleas note that the specified menu will lose it's comonents while it is
     * merged with this menu. When another menu is merged, the previously merged
     * menu regains it's components. This is due to the fact that menu items can
     * only belong to one menu - thus merging two menues involves moving menu
     * items from one menu to another.
     *
     * @param menu the menu to merge with this menu.
     * @return <code>true</code> if the merge where ok, otherwise
     *         <code>false</code>
     */
    public synchronized boolean mergeMenu(G9Menu menu) {

        /* No merged components to remove nor any components to merge into
         * this menu, therefore nothing has to be done and we simply return. */
        if(mergedFromMenu == null && menu == null && !(this instanceof G9WindowMenu)) {
            return true;
        }

        /* Merging is not allowed if this menu allready is merged into the
         * agrument menu. */
        if(menu != null && menu == mergedIntoMenu) {
            return false;
        }

        if(menu == null) {
            setText(defaultTitle);
        } else {
            menu.setMergedIntoMenu(this);
        }

        if(mergedFromMenu != null) {
            mergedFromMenu.setMergedIntoMenu(null);
        }

        mergedFromMenu = menu;

        createMenuComponents();
        toggleEnabledState(menu);

        return true;
    }

    /**
     * Enable or disable this menu and its menu items.
     *
     * @param enable if <code>true</code> the menu are enabled, otherwise
     *            disabled.
     */
    public synchronized void setEnabledWholeMenu(boolean enable) {
        setEnableOwnMenuItems(enable);
    }

    /**
     * Enables or disables all default and application menu items or their
     * assigned actions.
     *
     * @param isModal If <code>true</code> then all default and application menu
     *            items are disabled, otherwise they are enabled.
     */
    public synchronized void setModal(boolean isModal) {
        if(this.isModal == isModal)
            return;

        this.isModal = isModal;

        setEnableOwnMenuItems(!isModal);


        /*
         * Enable only if:
         *  - this is a window menu or not modal and not registered as disabled by the parentframe OR
         *  - this menu is merged with another menu and the parentframe of
         *    that menu has not registered the merged menu as disabled.
         */
        boolean enable =
            ((this instanceof G9WindowMenu || !isModal) && isEnabledInParentFrame(this)) ||
            (mergedFromMenu !=null && isEnabledInParentFrame(mergedFromMenu));

        this.setEnabled(enable);
    }

    /**
     * Returns <code>false<code> if menu is registered as diabled by its parent
     * frame, othewise <code>true</code>
     *
     * @param menu menu to check if registered as diabled by parentframe
     * @return <code>false<code> if menu is registered as diabled by its parent
     *          frame, othewise <code>true</code>
     */
    private boolean isEnabledInParentFrame(G9Menu menu) {
        if (menu == null || menu.getParentFrame() == null)
            return true;
        return menu.getParentFrame().isEnabledComponent(menu);
    }

    /**
     * Return <code>true</code> if this menu is modal otherwise
     * <code>false</code>.
     *
     * @return Returns <code>true</code> if this menu is modal otherwise
     *         <code>false</code>.
     */
    public synchronized boolean isModal() {
        return isModal;
    }

    /**
     * If set to <code>true</code>, consecutive separators are reduced to one
     *
     * @param remove The flag to set.
     */
    public synchronized void setRemoveConsecutiveSeparators(boolean remove) {
        if(this.removeConsecutiveSeparators != remove) {
            this.removeConsecutiveSeparators = remove;
            this.createMenuComponents();
        }
    }

    /**
     * Return true if consecutive separators are to be reduced to one
     *
     * @return Returns true if consecutive separators are to be reduced to one.
     */
    public synchronized boolean isRemoveConsecutiveSeparators() {
        return removeConsecutiveSeparators;
    }

    /**
     * Dynamicly creates the menu. This method is used each time the menu needs
     * to draw it self.
     */
    protected synchronized void createMenuComponents() {

        removeAll();

        /*
         * Add default menu items first
         */
        Iterator<?> it = defaultItems.iterator();
        while (it.hasNext()) {
            add((Component) it.next());
        }

        /*
         * Get the  menu items for this menu.
         */
        Component[] menuComponents = getG9Components();

        /*
         * Simply add the menuComponents if there are no other components to
         * merge into the menu.
         */
        if (mergedFromMenu == null) {
            for (int i = 0; i < menuComponents.length; i++) {
                add(menuComponents[i]);
            }
            return;
        }

        /*
         * Merge application menu items with dialog menu items
         */
        Component result[] = mergedFromMenu.getG9Components();

        for (int i = 0; i < menuComponents.length; i++) {
            int index = indexOf(result, menuComponents[i]);
            if (index > -1) {
                add(result[index]);
                result = remove(result, index);
                while (index < result.length
                        && indexOf(menuComponents, result[index]) == -1) {
                    add(result[index]);
                    result = remove(result, index);
                }
            } else {
                add(menuComponents[i]);
            }
        }

        for (int i = 0; i < result.length; i++) {
            add(result[i]);
        }
    }


    /**
     * Appends a component to the end of this menu and removes consecutive
     * separators if {@link #setRemoveConsecutiveSeparators(boolean)} is set.
     * Returns the component added or <code>null</code> if separator where
     * removed as a result of preventing consecutive separators.
     *
     * @param c the component to append to this menu
     * @return the added component or <code>null</code> if separator where
     *         removed as a result of preventing consecutive separators.
     */
    @Override
    public Component add(Component c) {
        if (removeConsecutiveSeparators && getMenuComponentCount() > 0) {
            Component lastAdded = getMenuComponent(getMenuComponentCount() - 1);
            if (lastAdded instanceof JSeparator && c instanceof JSeparator) {
                return null;
            }
        }

        return super.add(c);
    }

    /**
     * Removes the item at the given index from the given array.
     *
     * @param items the array to remove the given index from
     * @param index the index of the menu item to remove
     * @return a new array not containing the removed menu item
     */
    private Component[] remove(Component items[], int index) {
        if(items == null)
            return null;

        int len = items.length - 1;
        Component result[] = new Component[len];
        if(index == 0) {
            System.arraycopy(items,1,result,0,len);
        } else if (index == len) {
            System.arraycopy(items,0,result,0,len);
        } else {
            System.arraycopy(items,0,result,0, index);
            System.arraycopy(items,index+1,result,index, len-index);
        }

        return result;
    }

    /**
     * Searches for the first occurence of the given menu item in the given
     * <code>items</code> array. Two items are considered if they have the same
     * name(, not title).
     *
     * @param items the array to search in
     * @param item the menu item to find the index of
     * @return the index of the found menu item, otherwise -1.
     */
    private int indexOf(Component items[], Component item) {
        if (items == null || item == null)
            return -1;

        String thisName = item.getName();
        if (thisName == null)
            return -1;

        for (int i = 0; i < items.length; i++) {
            String anotherName = items[i].getName();
            if (thisName.equals(anotherName))
                return i;
        }

        return -1;
    }


    /**
     * Enables or disables this menu. The action taken depends on whether it or
     * the merged <code>menu</code> is registered as diabled menu at the parent
     * frame of this menu or the merged menu respectively, where the merged menu
     * takes presidence.
     *
     * @param menu the menu merged into this menu
     */
    private void toggleEnabledState(G9Menu menu) {
        G9DialogFrame parentFrame = (menu == null) ?
                this.getParentFrame() : menu.getParentFrame();
        if (parentFrame != null) {
            G9Menu m = (menu == null) ? this : menu;
            this.setEnabled(parentFrame.isEnabledComponent(m));
        }
    }

    /**
     * Enable or disable this menu. If this menu is merged into another menu
     * then that menu is enabled or disabled too. (Note that it is the
     * responsibility of the merged into menu to set itself as the merged into
     * menu!)
     *
     * @param b if <code>true</code> enable the menu item.
     * @see javax.swing.JMenuItem#setEnabled(boolean)
     */
    @Override
    public synchronized void setEnabled(boolean b) {
        super.setEnabled(b);
        if(mergedIntoMenu != null) {
            mergedIntoMenu.setEnabled(b);
        }
    }

    /**
     * Show or hide this menu. If this menu is merged into another menu then
     * that menu is shown or hidden too. (Note that it is the responsibility of
     * the merged into menu to set itself as the merged into menu!)
     *
     * @param b if <code>true</code> the item is made visible
     * @see javax.swing.JComponent#setVisible(boolean)
     */
    @Override
    public synchronized void setVisible(boolean b) {
        super.setVisible(b);
        if(mergedIntoMenu != null) {
            mergedIntoMenu.setVisible(b);
        }
    }

    /**
     * Enables or disables all original menu items (that is, not merged menu
     * items) or their assigned actions.
     *
     * @param enable If <code>true</code> then all original menu items are
     *            enabled, otherwise they are disabled.
     */
    private void setEnableOwnMenuItems(boolean enable) {
        Component appComponents[] = getG9Components();
        for (int i = 0; i < appComponents.length; i++) {
            boolean enableComp = enable;
            enableComp &= isEnabledMenuItem(appComponents[i]);
            setEnableMenuItem(appComponents[i], enableComp);
        }
    }

    /**
     * If the argument component is assigned with an action that action is
     * enable or disabled. If no action is assigned to the component then the
     * component itself are enable or disabled.
     *
     * @param comp the component enable or disable
     * @param enable if <code>true</code> the menu item is enabled.
     */
    private void setEnableMenuItem(Component comp, boolean enable) {
        if (comp instanceof AbstractButton) {
            Action action = ((AbstractButton) comp).getAction();
            if (action != null) {
                action.setEnabled(enable);
                return;
            }
        }
        comp.setEnabled(enable);
    }

    /**
     * Returns <code>false</code> if the agrument component is registered as
     * disabled component by the component parent view, otherwise
     * <code>true</code>.
     *
     * @param comp the component to check the state for
     * @return <code>false</code> if the component disabled, otherwise
     *         <code>true</code>.
     */
    private boolean isEnabledMenuItem(Component comp) {
        G9DialogFrame gdm = getParentFrame();
        if(gdm == null) return true;
        return gdm.isEnabledComponent(comp);
    }

    /**
     * Returns the parent frame to this menu.
     *
     * @return the parent frame to this menu or <code>null</code> if the menu
     *         bar do not have a parent menu bar.
     */
    protected G9DialogFrame getParentFrame() {
        G9MenuBar parentMenuBar = getParentMenuBar(this);
        if(parentMenuBar == null) return null;
        return parentMenuBar.getParentFrame();
    }

    /**
     * Returns the parent menu bar of this menu.
     *
     * @param comp this menu.
     * @return the parent menu bar of this menu or <code>null</code> if the menu
     *         bar do not have a parent menu bar.
     */
    private G9MenuBar getParentMenuBar(Component comp) {
        if(comp == null)
            return null;

        if(comp.getParent() instanceof G9MenuBar)
            return (G9MenuBar)comp.getParent();

        return getParentMenuBar(comp.getParent());
    }

    /**
     * The argument list is used as the default menu items for this menu.
     *
     * @param defaultItems the menu item to add
     */
    protected synchronized void setDefaultItem(final List<?> defaultItems) {
        this.defaultItems = defaultItems;
        createMenuComponents();
    }

    /**
     * Creates a menu item based on the given arguments.
     *
     * @param action the menu item action
     * @param accelerator the menu item accelerator
     * @param title the menu item title
     * @return the menu item
     */
    protected static G9MenuItem createMenuItem(final Action action,
            final String accelerator, final String title) {
        G9MenuItem menuItem = new G9MenuItem(action);
        if (accelerator != null) {
            menuItem.setAccelerator(G9Render.render(accelerator));
        }
        Message msg = MessageSystem.getMessageFactory().getMessage(title);
        menuItem.setText(msg.getMessageText());
        return menuItem;
    }

    /**
     * @param mergedIntoMenu The mergedIntoMenu to set.
     */
    private synchronized void setMergedIntoMenu(G9Menu mergedIntoMenu) {
        this.mergedIntoMenu = mergedIntoMenu;
    }

}
