/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.lang.reflect.InvocationTargetException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.SwingUtilities;

import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.*;

/**
 * Utility class used for clearing nodes from the object selction.
 */
@SuppressWarnings("rawtypes")
public class ClearTool {

    /** Map of role objects */
    Map roleObjects;

    /**
     * Creates a new instance.
     *
     * @param roleObjects the complemte map of role objects. Key = name of role,
     *            value = the role object.
     */
    public ClearTool(Map roleObjects) {
        this.roleObjects = roleObjects;
    }

    /**
     * Recursivly clears the specified node and all related children from the
     * object selection.
     *
     * @param roleName the name of the node to clear
     */
    public void clear(final String roleName) {
        Runnable task = new Runnable(){
            @Override
            public void run() {
                clear(getRoleObject(roleName));
            }
        };
        edt(task, "clear " + roleName);

    }

    /**
     * Recursivly clears the specified node and all related children from the
     * object selection.
     *
     * @param roleName the name of the node to clear
     */
    public void clearRole(final String roleName) {
        class Task implements Runnable {
            @Override
            public void run() {
                RoleObject role = getRoleObject(roleName);
                if (role == null) {
                    return;
                }

                List simpleBlocks = role.getSimpleBlocks();
                if (simpleBlocks != null) {
                    clearSimpleBlocks(simpleBlocks, false);
                }

                Set listBlocks = role.getListBlocks().keySet();

                if (listBlocks != null) {
                    Iterator it = listBlocks.iterator();
                    while (it.hasNext()) {
                        Listblock l = (Listblock) it.next();
                        l.clearSelectedLines();
                    }
                }

                clearMembers(roleName);
            }
        }

        edt(new Task(), "clearRole " + roleName);
    }

    /**
     * Performs the runnable on the edt.
     *
     * @param task the task to perform on the edt
     * @param msgArg the arguments to use if an exception is caught
     */
    private void edt(Runnable task, String msgArg) {
        String msgID = null;
        Exception ex = null;
        if (!SwingUtilities.isEventDispatchThread()) {
            try {
                SwingUtilities.invokeAndWait(task);
            } catch (InterruptedException e) {
                msgID = CRuntimeMsg.CT_INTERRUPTED;
                ex = e;
            } catch (InvocationTargetException e) {
                msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
                if (e.getCause() != null && e.getCause() instanceof Exception) {
                    ex = (Exception) e.getCause();
                } else {
                    ex = e;
                }
            } finally {
                if (msgID != null) {
                    Object[] msgArgs = {this.getClass(),
                            msgArg, ex};
                    Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                    MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                    throw new G9ClientFrameworkException(ex, msg);
                }
            }
        } else {
            task.run();
        }
    }

    /**
     * Clears all roles associated with the specified role.
     *
     * @param roleName the name of the object selection node.
     */
    public void clearMembers(final String roleName) {
        Runnable task = new Runnable() {
           @Override
        public void run() {
               RoleObject parent = getRoleObject(roleName);
               if (parent == null) {
                   return;
               }
               Iterator associations = parent.getAssociationRoles();
               while (associations.hasNext()) {
                   clear((RoleObject) associations.next());
               }
           }
        };
        edt(task, "clearMembers");
    }

    /**
     * Recursivly clears the specified role, expept the keys on the first node.
     *
     * @param roleName the name of the object selection node.
     */
    public void clearKeepKeys(final String roleName) {
        class Task implements Runnable {
            @Override
            public void run() {
                RoleObject ro = getRoleObject(roleName);
                List simpleBlocks = ro.getSimpleBlocks();
                clearSimpleBlocks(simpleBlocks, true);
                // Map listBlocks = ro.getListBlocks();
                // clearListBlocks(listBlocks.keySet());
                clearMembers(roleName);
            }
        }

        edt(new Task(), "clearKeepKeys, target: " + roleName);
    }

    /**
     * Clears the specified role object
     *
     * @param currentRoot the role object to clear.
     */
    private void clear(RoleObject currentRoot) {

        if (currentRoot == null) {
            return;
        }

        Set listBlocks = currentRoot.getListBlocks().keySet();
        if (listBlocks != null) {
            clearListBlocks(listBlocks);
        }

        List simpleBlocks = currentRoot.getSimpleBlocks();
        if (simpleBlocks != null) {
            clearSimpleBlocks(simpleBlocks, false);
        }

        AbstractNode currentNode = currentRoot.getNode();
        currentNode.clearAllAssociation();
		currentNode.setState(ObjectSelectionNode.CLEARED, false);

        Iterator associations = currentRoot.getAssociationRoles();

        if (associations == null) {
            return;
        }

        while (associations.hasNext()) {
            clear((RoleObject) associations.next());
        }

    }

    /**
     * Clears the simple blocks defined in the list of simple blocks.
     *
     * @param simpleBlocks the list of simple blocks
     * @param keepKeys if <code>true</code> keys are not cleared.
     */
    private void clearSimpleBlocks(List simpleBlocks, boolean keepKeys) {
        Iterator it = simpleBlocks.iterator();
        while (it.hasNext()) {
            ObjectSelectionNode os = (ObjectSelectionNode) it.next();
            if (keepKeys) {
                os.clearKeepKeys();
            } else {
                os.clear();
            }
        }
    }

    /**
     * Clears listblocks contained in the <code>listBlocks</code> set.
     *
     * @param listBlocks a set of listblocks to clear.
     */
    public void clearListBlocks(final Set listBlocks) {
        Runnable task = new Runnable() {
            @Override
            public void run() {

                Iterator it = listBlocks.iterator();
                while (it.hasNext()) {
                    Listblock listBlock = (Listblock) it.next();
                    listBlock.purgeAllLines();
                }
            }
        };
        edt(task, "clearListBlocks");
    }

    /**
     * Returns the role object associated with the specifed role name
     *
     * @param roleName the name of the object selection node.
     * @return the role object for the specified node
     */
    public RoleObject getRoleObject(String roleName) {
        Iterator valueIterator = roleObjects.values().iterator();
        while (valueIterator.hasNext()) {
            Iterator roleIterator = ((Set) valueIterator.next())
                    .iterator();
            while (roleIterator.hasNext()) {
                RoleObject ro = (RoleObject) roleIterator.next();
                if (ro.getRoleName().equals(roleName)) {

                    return ro; // <---- return point.
                }
            }
        }

        return null;
    }

}
