/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.DefaultKeyboardFocusManager;
import java.awt.KeyboardFocusManager;
import java.awt.Window;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.beans.PropertyVetoException;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.swing.AbstractButton;
import javax.swing.JComponent;
import javax.swing.JDesktopPane;
import javax.swing.JFrame;
import javax.swing.JInternalFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.KeyStroke;
import javax.swing.RootPaneContainer;
import javax.swing.SwingUtilities;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;
import javax.swing.event.InternalFrameListener;

import no.g9.client.component.EnableManager;
import no.g9.client.component.G9Button;
import no.g9.client.component.G9DesktopPane;
import no.g9.client.component.G9ToolBar;
import no.g9.client.component.menu.G9Menu;
import no.g9.client.component.menu.G9MenuBar;
import no.g9.client.component.menu.G9MenuItem;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.*;

/**
 * Utility class used for blocking dialogs and application from user input.
 */
@SuppressWarnings({"unchecked", "rawtypes"})
public class DialogBlocker {

    /**
     * If this attribute is <code>true</code> clicking a blocked window will
     * select it.
     */
    private static boolean doSelect = true;

    /** Constant used when toggeling various things. */
    private static final boolean BLOCK = true;

    /** Constant used when toggeling various things. */
    private static final boolean UNBLOCK = false;

    /** The keyboard blocker. */
    private static final KeyboardBlockManager keyBlock = new KeyboardBlockManager();

    /**
     * Map of blocked windows. Used to keep the original glass pane, and to
     * check if a component is already blocked.
     */
    private static final Map blockedWindows = new HashMap();

    /** The wait cursor */
    private static final Cursor WAIT_CURSOR = Cursor
            .getPredefinedCursor(Cursor.WAIT_CURSOR);

    /** The default cursor */
    private static final Cursor DEFAULT_CURSOR = Cursor.getDefaultCursor();

    /** The enable manager */
    private static EnableManager enableManager = new EnableManager();

    /** Flag signalling if application is blocked. */
    private static boolean applicationBlock = false;

    /** The map from a dialog to the Set of listblocks contained in that dialog. */
    private static final Map dialogToListblocks = new HashMap();


    /**
     * The glass pane used for blocking mouse events and showing the appropriate
     * cursor.
     */
    private static final class BlockPane extends JPanel {
        BlockPane() {
            setOpaque(false);
            setCursor(WAIT_CURSOR);
            addMouseListener(MOUSE_BLOCK);
        }
    }


    /**
     * The mouse listener that blocks mouse events, registred on block panes.
     */
    private static final MouseAdapter MOUSE_BLOCK = new MouseAdapter() {

        @Override public void mouseClicked(MouseEvent e) {
            listblockProcess(e);

            if (!doSelect) {
                return;
            }
            Component comp = (Component) e.getSource();

            while (comp != null && !(comp instanceof JInternalFrame)) {
                comp = comp.getParent();
            }

            if (comp != null) {
                JInternalFrame frame = (JInternalFrame) comp;
                try {
                    frame.setSelected(true);
                } catch (PropertyVetoException e1) {
                    // Can't select frame - nothing to do.
                }
            }
        }

        /**
         * Give G9Tables a crack at the event
         *
         * @param e the mouse event.
         */
        private void listblockProcess(MouseEvent e) {
            Component comp = (Component) e.getSource();

            while (! (comp instanceof RootPaneContainer)) {
                comp = comp.getParent();
            }

            if (comp instanceof JFrame) {
                return;
            }

            Set listblocks = (Set) dialogToListblocks.get(comp);
            boolean processed = false;
            Iterator it = listblocks.iterator();
            while (it.hasNext() && !processed) {
                G9Table table = (G9Table) it.next();
                processed = table.processDoubleClickMouseEvent(e);
            }


        }
    };

    /**
     * A listener registred on blocked internal frames. When the frame becomes
     * active, the glass pane needs to be raised in order to block the document
     * window. The listener is added and removed through the
     * <code>toggleListener(...)</code> method.
     */
    private final static InternalFrameListener frameAdapter = new InternalFrameAdapter() {
        @Override public void internalFrameActivated(InternalFrameEvent e) {
            JInternalFrame frame = (JInternalFrame) e.getSource();
            frame.getGlassPane().setVisible(true);
        }
    };

    /**
     * A listener registred on blocked windows (JFrame and JDialog). When the
     * frame becomes active, the glass pane needs to be raised in order to block
     * the document window. The listener is added and removed through the
     * <code>toggleListener(...)</code> method.
     */
    private final static WindowListener windowAdapter = new WindowAdapter() {
        @Override public void windowActivated(WindowEvent e) {
            RootPaneContainer rootPane = (RootPaneContainer) e.getSource();
            rootPane.getGlassPane().setVisible(true);
        }
    };

    /**
     * Adds or removes the window listener from the specified component.
     *
     * @param rootPane either a JDialog or a Window
     * @param block if (<code>true</code> the listener is added, otherwise it is
     *            removed.
     */
    private final static void toggleListener(RootPaneContainer rootPane, boolean block) {
        if (rootPane instanceof JInternalFrame) {
            JInternalFrame internalFrame = (JInternalFrame) rootPane;
            if (block) {
                internalFrame.addInternalFrameListener(frameAdapter);
            } else {
                internalFrame.removeInternalFrameListener(frameAdapter);
            }
        } else {
            Window window = (Window) rootPane;
            if (block) {
                window.addWindowListener(windowAdapter);
            } else {
                window.removeWindowListener(windowAdapter);
            }
        }
    }

    /**
     * Toggles the application edit menu.
     *
     * @param rootPane (missing javadoc)
     */
    private final static void toggleEditMenu(RootPaneContainer rootPane) {
        if (rootPane instanceof G9DialogFrame) {
            G9DialogFrame internalFrame = (G9DialogFrame) rootPane;
            JDesktopPane desktopPane = internalFrame.getDesktopPane();
            if (desktopPane != null && desktopPane instanceof G9DesktopPane) {
                ((G9DesktopPane) desktopPane).toggleEditMenu(internalFrame);
            }
        }
    }

    /**
     * Toggles the visibility of the new glass pane depending on the selected
     * state of the root pane.
     *
     * @param rootPane the window component.
     */
    private static void toggleGlassPaneVisibility(RootPaneContainer rootPane) {
        if (rootPane instanceof JInternalFrame) {
            JInternalFrame frame = (JInternalFrame) rootPane;
            if (frame.isSelected()) {
                frame.getGlassPane().setVisible(false);
            } else if (frame.isShowing()){
                frame.getDesktopPane().getDesktopManager().deactivateFrame(frame);
            }
        } else if (rootPane instanceof Window) {
            // JDialog and JFrame
            rootPane.getGlassPane().setVisible(!((Window) rootPane).isActive());
        }
    }

    /**
     * Blocks or unblocks the toolbar components of a given document window.
     *
     * @param documentWindow the document window that "owns" the toolbar
     * @param block if <code>true</code> the toolbar will be blocked.
     */
    private static void toggleToolBarComponents(RootPaneContainer documentWindow, boolean block) {
        if (documentWindow instanceof G9DialogFrame) {
            G9DialogFrame gdf = (G9DialogFrame) documentWindow;
            G9ToolBar toolbar = gdf.getG9ToolBar();
            if (toolbar == null) {
                return;
            }
            Component[] components = toolbar.getG9Components();
            for (int i = 0; i < components.length; i++) {
                if (components[i] instanceof G9Button) {
                    G9Button button = (G9Button) components[i];
                    button.setCursor(block ? WAIT_CURSOR : DEFAULT_CURSOR);
                    button.setBlocked(block);

                }
            }
        }
    }

    /**
     * Check if entire application is blocked.
     *
     * @return <code>true</code> if application is blocked.
     */
    private static boolean isApplicationBlocked() {
        return applicationBlock;
    }

    /**
     * Used to supress all keyboard actions on a blocked dialog except those
     * defined as key strokes allowed through.
     */
    private static class KeyboardBlockManager extends
            DefaultKeyboardFocusManager {

        /**
         * Set of key strokes that will not be blocked when a dialog is blocked.
         */
        private final Set keyStrokesAllowedThrough = new HashSet();

        /** Ctrl + FG */
        private final KeyStroke ctrlF6 = KeyStroke.getKeyStroke(
                KeyEvent.VK_F6, InputEvent.CTRL_DOWN_MASK);

        /** Ctrl + Shift + F6 */
        private final KeyStroke ctrlShiftF6 = KeyStroke.getKeyStroke(
                KeyEvent.VK_F6, InputEvent.CTRL_DOWN_MASK
                        | InputEvent.SHIFT_DOWN_MASK);

        /** Alt */
        private final KeyStroke alt = KeyStroke.getKeyStroke(
                KeyEvent.VK_ALT, 0);

        /** AltGr */
        private final KeyStroke altGr = KeyStroke.getKeyStroke(
                KeyEvent.VK_ALT_GRAPH, 0);

        /** Ctrl + F4 */
        private final KeyStroke ctrlF4 = KeyStroke.getKeyStroke(
                KeyEvent.VK_F4, InputEvent.CTRL_DOWN_MASK);

        /**
         * Default contsructor. Creates a new instance, and sets up the key
         * strokes that should not be blocked
         */
        public KeyboardBlockManager() {
            addAllowedKeyStroke(ctrlF6);
            addAllowedKeyStroke(ctrlF6);
            addAllowedKeyStroke(ctrlShiftF6);
            addAllowedKeyStroke(alt);
            addAllowedKeyStroke(altGr);
            addAllowedKeyStroke(ctrlF4);
        }

        /**
         * Adds the specified key stroke to the set of allowed key strokes.
         *
         * @param allowedKeyStroke the key stroke to add
         */
        private void addAllowedKeyStroke(KeyStroke allowedKeyStroke) {
            keyStrokesAllowedThrough.add(allowedKeyStroke);
        }

        /**
         * Gets the set of allowed key strokes that will be not be blocked.
         *
         * @return the set of allowed key strokes.
         */
        private Set getAllowedKeyStrokes() {
            return keyStrokesAllowedThrough;
        }

        /**
         * Check if the specified key event constitues a navigate key stroke or
         * a key stroke that maps to an application key stroke and not to a
         * blocked dialog.
         *
         * @param source the source of the event.
         * @param e the key event
         * @return <code>true</code> if the key event should pass through
         */
        private boolean isAllowedKey(Component source, KeyEvent e) {
            KeyStroke stroke = KeyStroke.getKeyStroke(e.getKeyCode(), e.getModifiers());
            boolean allowed = getAllowedKeyStrokes().contains(stroke);
            if (allowed && source instanceof JInternalFrame) {
                JInternalFrame fSource = (JInternalFrame) source;
                allowed = fSource.getRootPane().getInputMap(
                        JComponent.WHEN_IN_FOCUSED_WINDOW).get(stroke) == null;
            }
            if (allowed && source instanceof G9DialogFrame) {
                G9DialogFrame gSource = (G9DialogFrame) source;
                allowed = !gSource.isRegisteredMnemonic(stroke);
            }
            return allowed;
        }

        /**
         * Determines which key events that should be dispatched (blocked) by
         * this blocker. Events that origin from a component in a blocked dialog
         * will be blocked unless the key event constitues a key stroke that
         * should be allowed through (not dispatched by this dispatcher).
         * KeyStrokes origin from a JOptionPane are never blocked.
         *
         * @param e the key event
         * @return <code>true</code> if the event is blocked.
         */
        @Override
        public boolean dispatchKeyEvent(KeyEvent e) {
            Object o = e.getSource();
            Component comp = null;
            if (o instanceof Component) {
                comp = (Component) o;
                while (comp != null && !(comp instanceof RootPaneContainer)
                        && !(comp instanceof JOptionPane)) {
                    comp = comp.getParent();
                }
            }

            return !(comp instanceof JOptionPane)
                    && (isApplicationBlocked() || (comp != null
                            && !isAllowedKey(comp, e) && blockedWindows
                            .containsKey(comp)));
        }

    }

    /**
     * Updates the map of registred listblocks from a dialog.
     *
     * @param rootPane the dialog containing the listblocks
     * @param block <if>true</code> listblocks are added, else the key is
     *            removed.
     */
    private static void registerListblocks(RootPaneContainer rootPane, boolean block) {
        if (!block) {
            dialogToListblocks.remove(rootPane);
        } else {
            Set listblocks = new HashSet();
            dialogToListblocks.put(rootPane, listblocks);
            getListblocks((Component) rootPane, listblocks);
        }
    }

    /**
     * Recursivly updates the specified Set with the listblocks found in the
     * component (which might be a container)
     *
     * @param component the component (either a G9Table or a container)
     * @param listblocks the set of G9Tables to update.
     */
    private static void getListblocks(Component component, Set listblocks) {
        if (component instanceof G9Table) {
            listblocks.add(component);
        } else if (component instanceof Container) {
            Container container = (Container) component;
            Component[] components = container.getComponents();
            for (int i = 0; i < components.length; i++) {
                getListblocks(components[i], listblocks);
            }
        }
    }

    /**
     * Recursivly adds all key strokes from the menu components.
     *
     * @param component the menu component.
     */
    private static void addMenuKeyStrokes(Component component) {
        if (component instanceof JMenuItem) {
            JMenuItem menuItem = (JMenuItem) component;
            if (menuItem.getAccelerator() != null) {
                keyBlock.addAllowedKeyStroke(menuItem.getAccelerator());
            }

            if (menuItem instanceof G9Menu) {
                G9Menu g9Menu = (G9Menu) menuItem;
                Component[] menuItems = g9Menu.getG9Components();
                if (menuItems != null) {
                    for (int i = 0; i < menuItems.length; i++) {
                        addMenuKeyStrokes(menuItems[i]);
                    }
                }
            }
        }

    }


    /**
     * Blocks or unblocks the menus in the specified menuBar.
     *
     * @param documentWindow the document window where the menu bar resides.
     * @param block if <code>true</code> menu items are disabled.
     */
    private static void toggleMenuBar(RootPaneContainer documentWindow, boolean block) {
        if (documentWindow == null) {
            return;
        }
        G9MenuBar menuBar = null;
        if (documentWindow instanceof G9DialogFrame) {

            menuBar = ((G9DialogFrame) documentWindow).getG9MenuBar();
        }
        if (menuBar == null) {
            return;
        }
        Component[] components = menuBar.getG9Components();

        if (components == null) {
            return;
        }

        for (int i = 0; i < components.length; i++) {
            if (components[i] instanceof G9Menu) {
                G9Menu g9Menu = (G9Menu) components[i];
                Component[] mComponents = g9Menu.getG9Components();
                for (int j = 0; j < mComponents.length; j++) {
                    if (mComponents[j] instanceof G9MenuItem) {
                        G9MenuItem menuItem = (G9MenuItem) mComponents[j];
                        enableManager.setEnabled(menuItem, !block);
                    }
                }
            }
        }

    }

    // Sets up the keyboard block manager that blocks all key events (exept
    // ctrl+f6) in blocked dialogs.
    static {
        KeyboardFocusManager.getCurrentKeyboardFocusManager()
                .addKeyEventDispatcher(keyBlock);
    }


    /**
     * Block the specified container and displays a wait cursor. Invoking this
     * method on an already blocked container has no effect.
     *
     * @param rootPane the component to block.
     */
    public static void block(final RootPaneContainer rootPane) {

        BlockTask blockTask = new BlockTask(rootPane);
        if (!SwingUtilities.isEventDispatchThread()) {
            String msgID = null;
            Exception ex = null;
            try {
                SwingUtilities.invokeAndWait(blockTask);
            } catch (InterruptedException e) {
                msgID = CRuntimeMsg.CT_INTERRUPTED;
                ex = e;
            } catch (InvocationTargetException e) {
                msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
                if (e.getCause() != null && e.getCause() instanceof Exception) {
                    ex = (Exception) e.getCause();
                } else {
                    ex = e;
                }
            } finally {
                if (msgID != null) {
                    Object[] msgArgs = {DialogBlocker.class,
                            "block", ex};
                    Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                    MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                    throw new G9ClientFrameworkException(ex, msg);
                }
            }
        } else {
            blockTask.run();
        }
    }

    /**
     * The block task, should run on edt.
     */
    private static class BlockTask implements Runnable {

        /**
         * The root pane, set by the constructor.
         */
        private RootPaneContainer rootPane;

        /**
         * Constructor
         *
         * @param rootPane the root pane to block.
         */
        BlockTask(final RootPaneContainer rootPane) {
            this.rootPane = rootPane;
        }

        @Override
        public void run() {
            if (!blockedWindows.containsKey(rootPane) && ((Component) rootPane).isShowing()) {
                blockedWindows.put(rootPane, rootPane.getGlassPane());
                toggleListener(rootPane, BLOCK);
                toggleToolBarComponents(rootPane, BLOCK);
                toggleMenuBar(rootPane, BLOCK);
                toggleEditMenu(rootPane);


                if (!(rootPane instanceof JFrame)) {
                    registerListblocks(rootPane, BLOCK);
                }

                if (rootPane instanceof JInternalFrame) {
                    BlockPane blockPane = new BlockPane();
                    rootPane.setGlassPane(blockPane);
                    blockPane.setVisible(true);
                } else {
                    rootPane.getGlassPane().addMouseListener(MOUSE_BLOCK);
                    rootPane.getGlassPane().setCursor(WAIT_CURSOR);
                    rootPane.getGlassPane().setVisible(true);

                    if (rootPane instanceof JFrame) {
                        applicationBlock = BLOCK;
                    }
                }
            }
        }

    }

    /**
     * Unblocks the specified container and shows the default cusor. Invoking
     * this method on an unblocked container has no effect.
     *
     * @param rootPane the component to unblock.
     */
    public static void unblock(RootPaneContainer rootPane) {
        UnblockTask unblockTask = new UnblockTask(rootPane);
        if (!SwingUtilities.isEventDispatchThread()) {
        	String msgID = null;
            Exception ex = null;
            try {
                SwingUtilities.invokeAndWait(unblockTask);
            } catch (InterruptedException e) {
                msgID = CRuntimeMsg.CT_INTERRUPTED;
                ex = e;
            } catch (InvocationTargetException e) {
                msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
                if (e.getCause() != null && e.getCause() instanceof Exception) {
                    ex = (Exception) e.getCause();
                } else {
                    ex = e;
                }
            } finally {
                if (msgID != null) {
                    Object[] msgArgs = {DialogBlocker.class,
                            "unblock", ex};
                    Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                    MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                    throw new G9ClientFrameworkException(ex, msg);
                }
            }
        } else {
            unblockTask.run();
        }
    }

    /**
     * The unblock task, should run on edt.
     */
    private static class UnblockTask implements Runnable {
        /** The root pane, set by constructor */
        private RootPaneContainer rootPane;

        /**
         * Constructor
         *
         * @param rootPane the root pane to unblock.
         */
        UnblockTask(RootPaneContainer rootPane) {
            this.rootPane = rootPane;
        }

        @Override
        public void run() {
            Component glassPaneComponent = (Component) blockedWindows
            .remove(rootPane);
            if (glassPaneComponent != null) {
                toggleListener(rootPane, UNBLOCK);
                toggleToolBarComponents(rootPane, UNBLOCK);
                toggleMenuBar(rootPane, UNBLOCK);
                toggleEditMenu(rootPane);
                registerListblocks(rootPane, UNBLOCK);
                if (rootPane instanceof JInternalFrame) {
                    rootPane.setGlassPane(glassPaneComponent);
                } else {
                    rootPane.getGlassPane().removeMouseListener(MOUSE_BLOCK);
                    rootPane.getGlassPane().setCursor(DEFAULT_CURSOR);
                    if (rootPane instanceof JFrame) {
                        applicationBlock = UNBLOCK;
                    }
                }

                toggleGlassPaneVisibility(rootPane);
            }
        }
    }

    /**
     * Checks if the specified window component is blocked.
     *
     * @param rootPane the window component
     * @return <code>true</code> if blocked.
     */
    public static boolean isBlocked(Component rootPane) {
        return blockedWindows.containsKey(rootPane);
    }

    /**
     * <em>Internal use.</em> Invoked by the application controller.
     * <p>
     * Adds all registred accelerators from the menu bar to the list of allowed
     * key strokes.
     *
     * @param menuBar the application menu bar.
     */
    public static void addAllowedKeyStrokesFromMenuBar(JMenuBar menuBar) {
        Component[] menuBarComponents = menuBar.getComponents();
        if (menuBarComponents != null) {
            for (int i = 0; i < menuBarComponents.length; i++) {
                if (menuBarComponents[i] instanceof JMenu) {
                    JMenu menu = (JMenu) menuBarComponents[i];
                    addMenuKeyStrokes(menu);
                }
            }
        }
    }

    /**
     * * <em>Internal use.</em> Invoked by the application controller.
     * <p>
     * Adds the mnenmonics from the component. The component can be an
     * AbstractButton, or it can be a Container containing abstract buttons.
     *
     * @param component either a button or a container with buttons.
     */
    public static void addAllowedMnemonicsFromButtons(Component component) {
        if (component instanceof AbstractButton) {
            AbstractButton button = (AbstractButton) component;
            if (button.getMnemonic() != 0) {
                KeyStroke stroke = KeyStroke.getKeyStroke(button.getMnemonic(), InputEvent.ALT_DOWN_MASK);
                keyBlock.addAllowedKeyStroke(stroke);
            }
        }
        if (component instanceof Container) {
            Container container = (Container) component;
            Component[] containerComponents = container.getComponents();
            if (containerComponents != null) {
                for (int i = 0; i < containerComponents.length; i++) {
                    addAllowedMnemonicsFromButtons(containerComponents[i]);
                }
            }
        }
    }

    /**
     * Add the specified key stroke to the set of key strokes that will be
     * allowed through even if a dialog is blocked.
     *
     * @param stroke the key stroke to allow through.
     */
    public static void addAllowedKeyStroke(KeyStroke stroke) {
        keyBlock.addAllowedKeyStroke(stroke);
    }

}
