/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Color;
import java.awt.Component;
import java.awt.KeyboardFocusManager;
import java.awt.event.ActionEvent;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.DefaultCellEditor;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.KeyStroke;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import no.g9.client.component.G9ComboBox;
import no.g9.client.event.G9SelectedEvent;
import no.g9.client.event.G9SelectionListener;
import no.g9.client.event.G9VetoableSelectionEvent;
import no.g9.client.event.G9VetoableSelectionListener;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.*;

/**
 * The class representing a listblock.
 */
@SuppressWarnings({"unchecked", "rawtypes"})
public class G9Table extends JTable {

    /** The id */
    public String myStupidId = null;

    private Set editableCols = new HashSet();

    // NOT USED. Removed, and made setter a NOOP
    // private JScrollPane scrollPane;

    private boolean isBlocked = false;

    /** The timestamp of the last mouse released event */
    private long mouseReleasedTime = 0;

    /** Color used to render changed rows */
    private Color changedColor;

    /* The default color used to render rows */
    // NOT USED. Removed, and made setter a NOOP
    // private Color defaultColor;

    /** The cell render override */
    private CellRenderOverride cellRenderOverride = null;

    /**
     * The listblock
     */
    private Listblock listblock;

    /**
	 * @return the listblock
	 */
	public Listblock getListblock() {
		return listblock;
	}

    /**
     * Register the cell render override to use when rendering cells.
     *
     * @param renderOverride the cell render override
     */
	public void registerCellRenderOverride(CellRenderOverride renderOverride) {
		this.cellRenderOverride = renderOverride;
		renderOverride.setTable(this);
	}

    /**
     * Returns the cell render override (if any) that is used to override cell
     * renders
     *
     * @return the cell render override or <code>null</code> if no such override
     *         is registered.
     * @see #registerCellRenderOverride(CellRenderOverride)
     */
	public CellRenderOverride getRenderOverride() {
		return cellRenderOverride;
	}

    /**
     * Internal use. Returns the isBlocked property. A table that is blocked (
     * <code>isBlocked == true</code> will not process any mouse events.
     *
     * @return the value isBlocked property.
     */
    public boolean isBlocked() {
        return isBlocked;
    }

    /**
     * Internal use. Sets the isBlocked property. A table that is blocked (
     * <code>isBlocked == true</code> will not process any mouse events.
     *
     * @param isBlocked the value of the isBlocked property to set.
     */
    public void setBlocked(boolean isBlocked) {
        this.isBlocked = isBlocked;

        if (getTableHeader() instanceof ToolTipHeader) {
            ToolTipHeader header = (ToolTipHeader) getTableHeader();
            header.setBlocked(isBlocked);
        }
    }

    /**
     * Creates a new G9Table instance
     *
     * @param m the table model
     * @param id the id
     * @param listblock (missing javadoc)
     */
    public G9Table(AbstractTableModel m, String id, Listblock listblock) {
        super(m);
        myStupidId = id;
        this.listblock = listblock;
        setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
        initFocusPolicy();

        // Need to keep track of last mouse released event's timestamp
        addMouseListener(new MouseAdapter() {
            @Override public void mouseReleased(MouseEvent e) {
                mouseReleasedTime = e.getWhen();
            }
        });
    }

    /**
     * Internal use only! Invoked from DialogBlocker when a double click event
     * occurs, and gives this table a crack at the event.
     *
     * @param e the mouse event.
     * @return <code>true</code> if this table is processing the event.
     */
    public boolean processDoubleClickMouseEvent(MouseEvent e) {
        int multiClickInterval = ((Integer) getToolkit()
                .getDesktopProperty("awt.multiClickInterval")).intValue();

        boolean doProcessing = contains(e.getPoint());
        doProcessing = doProcessing
                && e.getID() == MouseEvent.MOUSE_CLICKED;
        doProcessing = doProcessing && e.getClickCount() == 2;
        doProcessing = e.getWhen() - mouseReleasedTime <= multiClickInterval;

        if (doProcessing) {
            MouseEvent doubleClick = new MouseEvent(this, e.getID(), e
                    .getWhen(), e.getModifiers(), e.getX(), e.getY(), e
                    .getClickCount(), e.isPopupTrigger());
            processMouseEvent(doubleClick);
        }
        return doProcessing;
    }

    /**
     * Sets up "tab" and "shift+tab" to shift focus to the next or previous
     * focus component (e.g. out of the table and onwards/backwards).
     */
    private void initFocusPolicy() {
        Action focusNextAction = new AbstractAction() {

            @Override
            public void actionPerformed(ActionEvent e) {
                KeyboardFocusManager.getCurrentKeyboardFocusManager()
                        .focusNextComponent();

            }

        };
        String focusNext = "FOCUS NEXT";
        KeyStroke tab = KeyStroke.getKeyStroke(KeyEvent.VK_TAB, 0);
        getInputMap(JComponent.WHEN_FOCUSED).put(tab, focusNext);
        getActionMap().put(focusNext, focusNextAction);

        Action focusPreviousAction = new AbstractAction() {

            @Override
            public void actionPerformed(ActionEvent e) {
                KeyboardFocusManager.getCurrentKeyboardFocusManager()
                        .focusPreviousComponent();
            }
        };
        String focusPrevious = "FOCUS PREVIOUS";
        KeyStroke shift_tab = KeyStroke.getKeyStroke(KeyEvent.VK_TAB,
                InputEvent.SHIFT_DOWN_MASK);
        getInputMap(JComponent.WHEN_FOCUSED).put(shift_tab, focusPrevious);
        getActionMap().put(focusPrevious, focusPreviousAction);
    }

    /**
     * Sets a column to be editable.
     *
     * @param col the index of the column that is editable.
     */
    public void setEditableCol(int col) {
        editableCols.add(new Integer(col));
    }

    @Override
    public boolean isCellEditable(int row, int col) {
        //return editableCols.contains(new Integer(col));
    	return super.isCellEditable(row, col);
    }

    /**
     * Sets the size of the columns of the given JTable to an appropriate size
     * based on the size of the column headings and the contents of the column.
     *
     * @param table The JTable to set the size of the columns.
     */
    public static void setBestFitColumnWidths(JTable table) {
        setBestFitColumnWidths(table, -1);
    }

    /**
     * Sets the size of a given column in the JTable to an appropriate size
     * based on the size of the column headings and the contents of the column.
     *
     * @param table The JTable to set the size of the column.
     * @param columnnumber The column to adjust.
     */
    public static void setBestFitColumnWidths(JTable table,
            int columnnumber) {
        JTableHeader header = table.getTableHeader();
        TableCellRenderer defaultHeaderRenderer = null;
        if (header != null) {
            defaultHeaderRenderer = header.getDefaultRenderer();
        }
        TableColumnModel columns = table.getColumnModel();
        TableModel data = table.getModel();
        int margin = columns.getColumnMargin();
        int rowCount = data.getRowCount();
        for (int col = columns.getColumnCount() - 1; col >= 0; --col) {
            TableColumn column = columns.getColumn(col);
            int columnIndex = column.getModelIndex();
            if (columnnumber > 0 && columnnumber != columnIndex) {
                continue;
            }
            int width = -1;
            TableCellRenderer h = column.getHeaderRenderer();
            if (h == null) {
                h = defaultHeaderRenderer;
            }
            if (h != null) { // Not explicitly impossible
                Component c = h.getTableCellRendererComponent(table,
                        column.getHeaderValue(), false, false, -1, col);
                width = c.getPreferredSize().width;
            }
            for (int row = rowCount - 1; row >= 0; --row) {
                TableCellRenderer r = table.getCellRenderer(row, col);
                Component c = r.getTableCellRendererComponent(table, data
                        .getValueAt(row, columnIndex), false, false, row,
                        col);
                if (c.getPreferredSize().width > width) {
                    width = c.getPreferredSize().width;
                }
            }
            if (width >= 0) {
                int finalWidth = width + margin + 5;
                column.setPreferredWidth(finalWidth);
            } else {
                Object[] args = { G9Table.class.getClass() };
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CF_INVALID_COLUMN_WIDTH, args);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);

                throw new G9ClientFrameworkException(msg);
            }
        }
    }

    @Override
    public TableCellRenderer getCellRenderer(int row, int column) {

        TableColumn tableColumn = getColumnModel().getColumn(column);
        TableCellRenderer renderer = tableColumn.getCellRenderer();
        if (renderer == null) {
            Class c = getColumnClass(column);
            if (c.equals(Object.class)) {
                Object o = getValueAt(row, column);
                if (o != null)
                    c = o.getClass();
            }
            renderer = getDefaultRenderer(c);
        }
        return renderer;
    }

    /** (missing javadoc) */
    HashSet addedFocusListener = new HashSet();

    private void myAddListener(DefaultCellEditor edIn, Component edComp) {
        final DefaultCellEditor theEditor = edIn;
        final Component editorComponent = edComp;
        if (!addedFocusListener.contains(editorComponent)) {
            addedFocusListener.add(editorComponent);
            editorComponent.addFocusListener(new FocusAdapter() {

                @Override
                public void focusLost(FocusEvent e) {
                    if (!(editorComponent instanceof JComboBox)
                            || ((JComboBox) editorComponent).isEditable()) {
                        theEditor.stopCellEditing();
                    }
                }

                @Override
                public void focusGained(FocusEvent e) {
                    if (editorComponent instanceof JComboBox) {
                        JComboBox box = (JComboBox) editorComponent;
                        if (!box.isPopupVisible()) {
                            box.setPopupVisible(true);
                        }
                    }
                }
            });
        }
        // if (editorComponent.isVisible() && !editorComponent.hasFocus())
        // {
        // editorComponent.requestFocus();
        // }
    }

    @Override
    public TableCellEditor getCellEditor(int row, int column) {
        if (row < 0 || column < 0) {
            return null;
        }
        TableCellEditor x = super.getCellEditor(row, column);
        if (x instanceof DefaultCellEditor) {
            DefaultCellEditor tx = (DefaultCellEditor) x;
            if (tx.getComponent() instanceof G9ComboBox) {
                G9ComboBox source = (G9ComboBox) tx.getComponent();
                tx = new DefaultCellEditor(source);
                source.setUsedAsListblockEditor(tx);
                getSelectionModel().setSelectionInterval(row, row);
                source.editingAboutToBegin();
                // G9ComboBox newbox = new G9ComboBox(source);
                myAddListener(tx, source);
                return tx;
            }
            myAddListener(tx, tx.getComponent());
        }
        return x;
    }

    @Override
    public boolean editCellAt(int row, int column) {
        boolean result = super.editCellAt(row, column);
        if (result) {
            final Component editorComponent = getEditorComponent();
            if (!editorComponent.isFocusOwner()) {
                editorComponent.requestFocus();
            }
        }
        return result;
    }

    @Override
    protected boolean processKeyBinding(KeyStroke ks, KeyEvent e,
            int condition, boolean pressed) {
        if (KeyEvent.VK_ENTER == e.getKeyCode()
                || KeyEvent.VK_ESCAPE == e.getKeyCode()) {
            return false;
        }

        return super.processKeyBinding(ks, e, condition, pressed);

    }

    /**
     * Process mouse events only if table is enabled and not blocked.
     *
     * @see java.awt.Component#processMouseEvent(java.awt.event.MouseEvent)
     */
    @Override
    protected void processMouseEvent(MouseEvent e) {
        if (this.isEnabled() && !this.isBlocked()) {
            super.processMouseEvent(e);
        }
    }

    /**
     * In addition to itself, all the table cells and the table header are
     * enabled or disabled according to <code>enable</code>.
     *
     * @see javax.swing.JComponent#setEnabled(boolean)
     */
    @Override
    public void setEnabled(boolean enable) {

        // Optimalization. Only neccesery to procced if value is
        // different from last time.
        if (this.isEnabled() == enable) {
            return;
        }

        // Enable/disable itself
        super.setEnabled(enable);

        // Enable/diable all the table cells renderers
        for (int col = 0; col < this.getColumnCount(); col++) {

            // Enable/disable header component renderer
            TableColumn column = this.getColumnModel().getColumn(col);
            this.setEnableSafe(column.getHeaderRenderer(), enable);

            // Enable/disable renderer for text (textfield, textarea,
            // combobox)
            this.setEnableSafe(this.getCellRenderer(0, col), enable);

            // Enable/disable renderer for boolean
            this.setEnableSafe(this.getDefaultRenderer(Boolean.class),
                    enable);

            // Force repaint so that the new rendersettings are applied.
            this.repaint();
        }
    }

    private void setEnableSafe(TableCellRenderer renderer, boolean enable) {
        if (renderer instanceof JComponent) {
            ((JComponent) renderer).setEnabled(enable);
        }
    }

    /**
     * Updates the selection models of the table, depending on the state of the
     * two flags: <code>toggle</code> and <code>extend</code>. All changes to
     * the selection that are the result of keyboard or mouse events received by
     * the UI are channeled through this method so that the behavior may be
     * overridden by a subclass.
     * <p>
     * This implementation uses the following conventions:
     * <ul>
     * <li> <code>toggle</code>: <em>false</em>, <code>extend</code>:
     * <em>false</em>. Clear the previous selection and ensure the new cell is
     * selected.
     * <li> <code>toggle</code>: <em>false</em>, <code>extend</code>:
     * <em>true</em>. Extend the previous selection to include the specified
     * cell.
     * <li> <code>toggle</code>: <em>true</em>, <code>extend</code>:
     * <em>false</em>. If the specified cell is selected, deselect it. If it is
     * not selected, select it.
     * <li> <code>toggle</code>: <em>true</em>, <code>extend</code>:
     * <em>true</em>. Leave the selection state as it is, but move the anchor
     * index to the specified location.
     * </ul>
     *
     * @param rowIndex affects the selection at <code>row</code>
     * @param columnIndex affects the selection at <code>column</code>
     * @param toggle see description above
     * @param extend if true, extend the current selection
     */
    @Override
    public void changeSelection(int rowIndex, int columnIndex,
            boolean toggle, boolean extend) {
        int selectedRow = getSelectedRow();
        boolean veto = false;
        if (selectedRow != rowIndex || toggle) { // new row or un-select

            veto = fireVetoableSelectionEvent(selectedRow, rowIndex);
            if (!veto) {
                // selection not vetoed, proceed with selection and fire
                // selected event.
                super.changeSelection(rowIndex, columnIndex, toggle,
                        extend);
                fireG9SelectedEvent(selectedRow, rowIndex);
            }
        } else {
            // same row as before selcted again...
            super.changeSelection(rowIndex, columnIndex, toggle, extend);
        }
    }

    /**
     * Adds a vetoable selection listener to the list of vetoable selection
     * listeners. These listeners will be notified each time the selected row is
     * about to change, giving the listener a chance to veto the selection
     * change.
     *
     * @param listener the listener to add
     */
    public void addVetoableSelectionListener(
            G9VetoableSelectionListener listener) {
        listenerList.add(G9VetoableSelectionListener.class, listener);
    }

    /**
     * Fires a vetoable selection event, notifying all registered listeners that
     * the selected row is about to change
     *
     * @param oldRow the currently selected row
     * @param newRow the row that will become selected if the selection event is
     *            not vetoed.
     * @return <code>true</code> if the selection event is vetoed.
     */
    protected boolean fireVetoableSelectionEvent(int oldRow, int newRow) {
        Object[] listeners = listenerList.getListenerList();
        G9VetoableSelectionEvent e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == G9VetoableSelectionListener.class) {
                if (e == null) {
                    e = new G9VetoableSelectionEvent(this, oldRow,
                            newRow);
                }
                ((G9VetoableSelectionListener) listeners[i + 1])
                        .vetoableSelectionChange(e);
            }
        }

        return e != null ? e.isVetoed() : false;
    }

    /**
     * Adds a selection listener to the list of listeners that will be notified
     * each time the selected row has changed.
     *
     * @param listener the listener to add
     */
    public void addG9SelectionListener(G9SelectionListener listener) {
        listenerList.add(G9SelectionListener.class, listener);
    }

    /**
     * Notifies all registered listeners that the selected row has changed
     *
     * @param oldRow the row that was selected prior to the change
     * @param newRow the row that is selcted.
     */
    public void fireG9SelectedEvent(int oldRow, int newRow) {
        Object[] listeners = listenerList.getListenerList();
        G9SelectedEvent e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == G9SelectionListener.class) {
                if (e == null) {
                    e = new G9SelectedEvent(this, oldRow, newRow);
                }
                ((G9SelectionListener) listeners[i + 1])
                        .selectionChanged(e);
            }
        }
    }

    /** (missing javadoc) */
    static class G9TextRenderer extends DefaultTableCellRenderer {
        private int alignment;

        /**
         * Creates a new G9TextRender
         *
         * @param alignment the text alignment
         */
        public G9TextRenderer(int alignment) {
            this.alignment = alignment;
        }

        @Override
        public Component getTableCellRendererComponent(JTable table,
                Object value, boolean isSelected, boolean hasFocus,
                int row, int column) {
            setHorizontalAlignment(alignment);
            return super.getTableCellRendererComponent(table, value,
                    isSelected, hasFocus, row, column);
        }
    }

    /**
     * Returns a renderer with the specified text alignment
     *
     * @param alignment the alignment
     * @return a cell renderer with the specified alignment
     */
    public TableCellRenderer getTextRenderer(int alignment) {
        return new G9TextRenderer(alignment);
    }

    @Override
    public boolean getScrollableTracksViewportWidth() {
        return false;
    }

    private Map indexToColumn = new HashMap();

    private Map columnToWidth = new HashMap();

    /* Holds info about the column width, used by show/hide column */
    private static class ColWidths {
        int min;
        int max;
        int pref;
        int width;

        ColWidths(int min, int max, int pref, int width) {
            this.min = min;
            this.max = max;
            this.pref = pref;
            this.width = width;
        }

        ColWidths() {
            // all fields are zero...
        }
    }

    private void setColWidths(TableColumn column, ColWidths widths) {
        column.setMinWidth(widths.min);
        column.setMaxWidth(widths.max);
        column.setPreferredWidth(widths.pref);
        column.setWidth(widths.width);
    }

    private ColWidths getColWidths(TableColumn column) {
        return new ColWidths(column.getMinWidth(), column.getMaxWidth(),
                column.getPreferredWidth(), column.getWidth());
    }

    /**
     * Internal use. Hides the specified column.
     *
     * @param columnIndex the index of the column to hide.
     */
    public void hideColumn(int columnIndex) {

        TableColumn column = getColumnModel().getColumn(columnIndex);
        if (column == null) {
            return;
        }
        Integer colIndex = new Integer(columnIndex);
        indexToColumn.put(colIndex, column);
        columnToWidth.put(column, getColWidths(column));
        setColWidths(column, new ColWidths());
        revalidate();
    }

    /**
     * Internal use! Shows the specified coulmn. If it was not previously
     * hidden, nothing happens.
     *
     * @param columnIndex the index of the column to show.
     */
    public void showColumn(int columnIndex) {
        TableColumn column = (TableColumn) indexToColumn
                .remove(new Integer(columnIndex));
        if (column == null) {
            return;
        }
        ColWidths widths = (ColWidths) columnToWidth.remove(column);
        setColWidths(column, widths);
    }

    /**
     * Internal use! The scrollPane is used to caluculate the width of the
     * table.
     *
     * @param scrollPane the scroll pane in which this table lies.
     */
    public void setScrollPaneInUse(JScrollPane scrollPane) {
        // this.scrollPane = scrollPane;
    }

    /**
     * Internal use. Check if a line is changed.
     *
     * @param row the row index for the wanted line.
     * @return true if the line is changed.
     */
	boolean isChanged(int row) {
		return listblock.getLine(row).isChanged();
	}

    /**
     * Internal use.
     *
     * @param changedColor the changedColor to set
     */
	public void setChangedLineColor(Color changedColor) {
		this.changedColor = changedColor;
	}

    /**
     * @return (missing javadoc)
     */
	public Color getChangedForgroundColor() {
		return changedColor;
	}

	/**
	 * @param defaultColor the defaultColor to set
	 */
	public void setDefaultLineColor(Color defaultColor) {
		// this.defaultColor = defaultColor;
	}

}
